"""Asplund et al. (2009) data.

This file was automatically created using the `asplund09_to_dict.py` parser available
in the dev/asplund09 folder.

The abundance data in this file are taken from:
Asplund et al. (2009)
10.1146/annurev.astro.46.060407.145222

Solar abundances are number abundances normed such that N_Si = 1e6
"""

import numpy as np

NaN = np.nan

asplund09_elements = {
    "H": [
        30902954325.135918,
        [1, 2],
        [0.9999800000000001, 2e-05],
        [30902336266.04942, 618059.0865027184],
    ],
    "He": [
        2630267991.8953815,
        [3, 4],
        [0.000166, 0.999834],
        [436624.4866546333, 2629831367.4087267],
    ],
    "Li": [
        0.34673685045253233,
        [6, 7],
        [0.0759, 0.9240999999999999],
        [0.026317326949347204, 0.3204195235031851],
    ],
    "Be": [0.7413102413009162, [9], [1.0], [0.7413102413009162]],
    "B": [
        15.488166189124792,
        [10, 11],
        [0.19899999999999998, 0.8009999999999999],
        [3.0821450716358334, 12.406021117488958],
    ],
    "C": [
        8317637.711026709,
        [12, 13],
        [0.988938, 0.011062],
        [8225628.002667331, 92009.70835937746],
    ],
    "N": [
        2089296.1308540408,
        [14, 15],
        [0.99771, 0.00229],
        [2084511.6427143852, 4784.488139655753],
    ],
    "O": [
        15135612.484362073,
        [16, 17, 18],
        [0.9976210000000001, 0.00037900000000000005, 0.002],
        [15099604.862261778, 5736.397131573227, 30271.224968724146],
    ],
    "F": [1122.018454301963, [19], [1.0], [1122.018454301963]],
    "Ne": [
        2630267.9918953814,
        [20, 21, 22],
        [0.929431, 0.002228, 0.068341],
        [2444652.609975316, 5860.237085942909, 179755.14483412224],
    ],
    "Na": [53703.17963702533, [23], [1.0], [53703.17963702533]],
    "Mg": [
        1230268.770812381,
        [24, 25, 26],
        [0.7898999999999999, 0.1, 0.1101],
        [971789.3020646997, 123026.87708123811, 135452.59166644316],
    ],
    "Al": [87096.35899560814, [27], [1.0], [87096.35899560814]],
    "Si": [
        1000000.0,
        [28, 29, 30],
        [0.9222969999999999, 0.046832000000000006, 0.030872],
        [922296.9999999999, 46832.00000000001, 30872.0],
    ],
    "P": [7943.282347242821, [31], [1.0], [7943.282347242821]],
    "S": [
        407380.27780411305,
        [32, 33, 34, 36],
        [0.9493, 0.0076, 0.0429, 0.0002],
        [386726.09771944454, 3096.090111311259, 17476.61391779645, 81.47605556082262],
    ],
    "Cl": [
        9772.37220955811,
        [35, 37],
        [0.7578, 0.2422],
        [7405.503660403137, 2366.8685491549745],
    ],
    "Ar": [
        77624.71166286927,
        [36, 38, 40],
        [0.845946, 0.153808, 0.000246],
        [65666.31433235761, 11939.301651442598, 19.09567906906584],
    ],
    "K": [
        3311.3112148259147,
        [39, 40, 41],
        [0.93132, 0.00147, 0.06721],
        [3083.890360591671, 4.867627485794094, 222.55322674844976],
    ],
    "Ca": [
        67608.29753919817,
        [40, 42, 43, 44, 46, 48],
        [0.96941, 0.00647, 0.00135, 0.02086, 4e-05, 0.00187],
        [
            65540.1597174741,
            437.4256850786122,
            91.27120167791755,
            1410.309086667674,
            2.7043319015679272,
            126.42751639830058,
        ],
    ],
    "Sc": [43.65158322401665, [45], [1.0], [43.65158322401665]],
    "Ti": [
        2754.2287033381685,
        [46, 47, 48, 49, 50],
        [0.0825, 0.07440000000000001, 0.7372, 0.0541, 0.0518],
        [
            227.22386802539893,
            204.91461552835977,
            2030.4174001008978,
            149.0037728505949,
            142.66904683291713,
        ],
    ],
    "V": [
        263.0267991895381,
        [50, 51],
        [0.0025, 0.9975],
        [0.6575669979738453, 262.36923219156427],
    ],
    "Cr": [
        13489.628825916532,
        [50, 52, 53, 54],
        [0.043449999999999996, 0.83789, 0.09501, 0.02365],
        [586.1243724860733, 11302.825096947203, 1281.6496347503296, 319.029721732926],
    ],
    "Mn": [8317.637711026708, [55], [1.0], [8317.637711026708]],
    "Fe": [
        977237.2209558112,
        [54, 56, 57, 58],
        [0.058449999999999995, 0.91754, 0.02119, 0.0028199999999999996],
        [57119.51556486716, 896654.239715795, 20707.65671205364, 2755.8089630953873],
    ],
    "Co": [3019.951720402019, [59], [1.0], [3019.951720402019]],
    "Ni": [
        51286.13839913648,
        [58, 60, 61, 62, 64],
        [0.680769, 0.262231, 0.011399, 0.036345, 0.009256],
        [
            34914.01315184174,
            13448.815358543958,
            584.6106916117567,
            1863.9947001166154,
            474.7044970224073,
        ],
    ],
    "Cu": [
        478.63009232263903,
        [63, 65],
        [0.6917, 0.30829999999999996],
        [331.06843485956944, 147.5616574630696],
    ],
    "Zn": [
        1122.018454301963,
        [64, 66, 67, 68, 70],
        [0.4863, 0.27899999999999997, 0.040999999999999995, 0.1875, 0.0062],
        [
            545.6375743270446,
            313.0431487502476,
            46.002756626380474,
            210.37846018161804,
            6.95651441667217,
        ],
    ],
    "Ga": [
        33.884415613920204,
        [69, 71],
        [0.60108, 0.39892000000000005],
        [20.367244537215154, 13.517171076705049],
    ],
    "Ge": [
        138.03842646028866,
        [70, 72, 73, 74, 76],
        [0.2084, 0.2754, 0.07730000000000001, 0.3628, 0.0761],
        [
            28.767208074324156,
            38.0157826471635,
            10.670370365380315,
            50.08034111979273,
            10.504724253627968,
        ],
    ],
    "As": [NaN, [75], [1.0], [NaN]],
    "Se": [
        NaN,
        [74, 76, 77, 78, 80, 82],
        [0.0089, 0.09369999999999999, 0.07629999999999999, 0.2377, 0.4961, 0.0873],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Br": [NaN, [79, 81], [0.5069, 0.49310000000000004], [NaN, NaN]],
    "Kr": [
        54.95408738576248,
        [78, 80, 82, 83, 84, 86],
        [0.00362, 0.02326, 0.11654999999999999, 0.11546, 0.56903, 0.17207999999999998],
        [
            0.19893379633646016,
            1.2782320725928351,
            6.404898884810616,
            6.344998929560135,
            31.270524345120425,
            9.456499357342006,
        ],
    ],
    "Rb": [
        10.232929922807536,
        [85, 87],
        [0.70844, 0.29156],
        [7.24941687451377, 2.983513048293765],
    ],
    "Sr": [
        22.90867652767779,
        [84, 86, 87, 88],
        [0.005580000000000001, 0.098678, 0.068961, 0.826781],
        [0.1278304150244421, 2.260582382398189, 1.579805242025188, 18.94045848822997],
    ],
    "Y": [5.011872336272734, [89], [1.0], [5.011872336272734]],
    "Zr": [
        11.748975549395302,
        [90, 91, 92, 94, 96],
        [
            0.5145000000000001,
            0.11220000000000001,
            0.17149999999999999,
            0.17379999999999998,
            0.027999999999999997,
        ],
        [
            6.044847920163884,
            1.318235056642153,
            2.0149493067212942,
            2.0419719504849034,
            0.32897131538306845,
        ],
    ],
    "Nb": [0.8912509381337477, [93], [1.0], [0.8912509381337477]],
    "Mo": [
        2.3442288153199176,
        [92, 94, 95, 96, 97, 98, 100],
        [0.14525, 0.09151, 0.15838, 0.16672, 0.09599, 0.24391, 0.09824],
        [
            0.340499235425218,
            0.21452037888992564,
            0.3712789597703685,
            0.39082982809013667,
            0.2250225239825589,
            0.5717808503446811,
            0.23029703881702868,
        ],
    ],
    "Ru": [
        1.7378008287493762,
        [96, 98, 99, 100, 101, 102, 104],
        [0.0554, 0.0187, 0.1276, 0.126, 0.17059999999999997, 0.3155, 0.1862],
        [
            0.09627416591271544,
            0.03249687549761334,
            0.2217433857484204,
            0.21896290442242142,
            0.29646882138464353,
            0.5482761614704282,
            0.3235785143131339,
        ],
    ],
    "Rh": [0.2511886431509582, [103], [1.0], [0.2511886431509582]],
    "Pd": [
        1.148153621496884,
        [102, 104, 105, 106, 108, 110],
        [0.0102, 0.1114, 0.22329999999999997, 0.2733, 0.2646, 0.11720000000000001],
        [
            0.011711166939268217,
            0.12790431343475286,
            0.2563827036802542,
            0.31379038475509835,
            0.3038014482480755,
            0.13456360443943483,
        ],
    ],
    "Ag": [
        0.2691534803926914,
        [107, 109],
        [0.51839, 0.48161000000000004],
        [0.1395264727007673, 0.12962700769192412],
    ],
    "Cd": [
        NaN,
        [106, 108, 110, 111, 112, 113, 114, 116],
        [0.0125, 0.0089, 0.1249, 0.128, 0.2413, 0.1222, 0.2873, 0.07490000000000001],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "In": [
        0.19498445997580494,
        [113, 115],
        [0.0429, 0.9571],
        [0.008364833332962032, 0.1866196266428429],
    ],
    "Sn": [
        3.3884415613920202,
        [112, 114, 115, 116, 117, 118, 119, 120, 122, 124],
        [
            0.0097,
            0.0066,
            0.0034000000000000002,
            0.1454,
            0.0768,
            0.2422,
            0.0859,
            0.3258,
            0.0463,
            0.0579,
        ],
        [
            0.0328678831455026,
            0.022363714305187334,
            0.01152070130873287,
            0.49267940302639973,
            0.26023231191490714,
            0.8206805461691473,
            0.29106713012357455,
            1.10395426070152,
            0.15688484429245053,
            0.19619076640459798,
        ],
    ],
    "Sb": [NaN, [121, 123], [0.5721, 0.4279], [NaN, NaN]],
    "Te": [
        NaN,
        [120, 122, 123, 124, 125, 126, 128, 130],
        [
            0.0009,
            0.0255,
            0.0089,
            0.047400000000000005,
            0.0707,
            0.1884,
            0.31739999999999996,
            0.3408,
        ],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "I": [NaN, [127], [1.0], [NaN]],
    "Xe": [
        5.370317963702532,
        [124, 126, 128, 129, 130, 131, 132, 134, 136],
        [
            0.00122,
            0.00108,
            0.02188,
            0.27255,
            0.04376,
            0.21693,
            0.26514,
            0.09789999999999999,
            0.07954,
        ],
        [
            0.006551787915717089,
            0.005799943400798735,
            0.1175025570458114,
            1.4636801610071253,
            0.2350051140916228,
            1.1649830758659903,
            1.4238861048960894,
            0.5257541286464779,
            0.4271550908328994,
        ],
    ],
    "Cs": [NaN, [133], [1.0], [NaN]],
    "Ba": [
        4.677351412871981,
        [130, 132, 134, 135, 136, 137, 138],
        [
            0.00106,
            0.00101,
            0.024169999999999997,
            0.06591999999999999,
            0.07854,
            0.11231999999999999,
            0.71698,
        ],
        [
            0.0049579924976442995,
            0.004724124927000701,
            0.11305158364911576,
            0.30833100513652095,
            0.3673591799669654,
            0.5253601106937809,
            3.3535674160009528,
        ],
    ],
    "La": [
        0.38904514499428045,
        [138, 139],
        [0.00091, 0.99909],
        [0.00035403108194479523, 0.3886911139123357],
    ],
    "Ce": [
        1.1748975549395302,
        [136, 138, 140, 142],
        [0.00185, 0.00251, 0.8845000000000001, 0.11114],
        [
            0.002173560476638131,
            0.002948992862898221,
            1.0391968873440145,
            0.1305781142559794,
        ],
    ],
    "Pr": [0.1621810097358933, [141], [1.0], [0.1621810097358933]],
    "Nd": [
        0.8128305161640994,
        [142, 143, 144, 145, 146, 148, 150],
        [0.27044, 0.12023, 0.23729, 0.08763, 0.17129999999999998, 0.05716, 0.05596],
        [
            0.21982188479141906,
            0.09772661295840968,
            0.19287655318057914,
            0.07122833813146003,
            0.1392378674189102,
            0.046461392303939926,
            0.04548599568454301,
        ],
    ],
    "Sm": [
        0.2818382931264461,
        [144, 147, 148, 149, 150, 152, 154],
        [0.030699999999999998, 0.1499, 0.1124, 0.1382, 0.0738, 0.2675, 0.2275],
        [
            0.008652435598981895,
            0.04224756013965427,
            0.03167862414741254,
            0.038950052110074844,
            0.02079966603273172,
            0.07539174341132433,
            0.06411821168626648,
        ],
    ],
    "Eu": [
        0.10232929922807536,
        [151, 153],
        [0.4781, 0.5219],
        [0.04892363796094283, 0.05340566126713253],
    ],
    "Gd": [
        0.36307805477010174,
        [152, 154, 155, 156, 157, 158, 160],
        [0.002, 0.0218, 0.14800000000000002, 0.2047, 0.1565, 0.2484, 0.2186],
        [
            0.0007261561095402035,
            0.007915101593988219,
            0.05373555210597507,
            0.07432207781143982,
            0.05682171557152092,
            0.09018858880489328,
            0.07936886277274424,
        ],
    ],
    "Tb": [0.06165950018614834, [159], [1.0], [0.06165950018614834]],
    "Dy": [
        0.38904514499428045,
        [156, 158, 160, 161, 162, 163, 164],
        [
            0.0005600000000000001,
            0.00095,
            0.02329,
            0.18889,
            0.25475000000000003,
            0.24896000000000001,
            0.2826,
        ],
        [
            0.00021786528119679708,
            0.0003695928877445664,
            0.009060861426916792,
            0.07348673743796963,
            0.09910925068729295,
            0.09685667929777607,
            0.10994415797538366,
        ],
    ],
    "Ho": [0.09332543007969923, [165], [1.0], [0.09332543007969923]],
    "Er": [
        0.25703957827688645,
        [162, 164, 166, 167, 168, 170],
        [0.0013900000000000002, 0.01601, 0.33503, 0.22869, 0.26978, 0.1491],
        [
            0.0003572850138048722,
            0.004115203648212952,
            0.08611596991010527,
            0.058782381156141164,
            0.06934413742753844,
            0.038324601121083775,
        ],
    ],
    "Tm": [0.03890451449942804, [169], [1.0], [0.03890451449942804]],
    "Yb": [
        0.2137962089502232,
        [168, 170, 171, 172, 173, 174, 176],
        [0.0012, 0.0298, 0.1409, 0.2169, 0.161, 0.32030000000000003, 0.13],
        [
            0.0002565554507402678,
            0.006371127026716651,
            0.030123885841086447,
            0.046372397721303416,
            0.034421189640985936,
            0.0684789257267565,
            0.027793507163529015,
        ],
    ],
    "Lu": [
        0.03890451449942804,
        [175, 176],
        [0.9717950000000001, 0.028205],
        [0.03780721266797168, 0.001097301831456368],
    ],
    "Hf": [
        0.21877616239495518,
        [174, 176, 177, 178, 179, 180],
        [
            0.0016200000000000001,
            0.05206,
            0.18606,
            0.27297,
            0.13629,
            0.35100000000000003,
        ],
        [
            0.0003544173830798274,
            0.011389487014281366,
            0.04070549277520536,
            0.059719329048950914,
            0.02981700317280844,
            0.07679043300062928,
        ],
    ],
    "Ta": [NaN, [180, 181], [0.00012, 0.99988], [NaN, NaN]],
    "W": [
        0.21877616239495518,
        [180, 182, 183, 184, 186],
        [0.0012, 0.265, 0.1431, 0.3064, 0.2843],
        [
            0.0002625313948739462,
            0.05797568303466313,
            0.03130686883871809,
            0.06703301615781426,
            0.062198062968885755,
        ],
    ],
    "Re": [NaN, [185, 187], [0.35662, 0.64338], [NaN, NaN]],
    "Os": [
        0.7762471166286927,
        [184, 186, 187, 188, 189, 190, 192],
        [0.0002, 0.01598, 0.012709999999999999, 0.13337, 0.16261, 0.26444, 0.4107],
        [
            0.00015524942332573855,
            0.01240442892372651,
            0.009866100852350684,
            0.10352807794476875,
            0.12622554363499172,
            0.20527078752129152,
            0.3188046907994041,
        ],
    ],
    "Ir": [
        0.7413102413009162,
        [191, 193],
        [0.373, 0.627],
        [0.27650872000524174, 0.4648015212956744],
    ],
    "Pt": [
        NaN,
        [190, 192, 194, 195, 196, 198],
        [
            0.00014000000000000001,
            0.00782,
            0.32966999999999996,
            0.33832,
            0.25242000000000003,
            0.07163,
        ],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Au": [0.25703957827688645, [197], [1.0], [0.25703957827688645]],
    "Hg": [
        NaN,
        [196, 198, 199, 200, 201, 202, 204],
        [
            0.0015,
            0.09970000000000001,
            0.16870000000000002,
            0.231,
            0.1318,
            0.2986,
            0.0687,
        ],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Tl": [
        0.24547089156850335,
        [203, 205],
        [0.29524, 0.7047599999999999],
        [0.07247282602668494, 0.1729980655418184],
    ],
    "Pb": [
        1.7378008287493762,
        [204, 206, 207, 208],
        [0.01997, 0.18582, 0.20562999999999998, 0.58858],
        [
            0.03470388255012505,
            0.3229181499982091,
            0.35734398441573423,
            1.0228348117853079,
        ],
    ],
    "Bi": [NaN, [209], [1.0], [NaN]],
    "Th": [0.032359365692962806, [232], [1.0], [0.032359365692962806]],
    "U": [NaN, [234, 235, 238], [2e-05, 0.24286000000000002, 0.75712], [NaN, NaN, NaN]],
}

asplund09_isotopes = {
    "H-1": [0.9999800000000001, 30902336266.04942],
    "H-2": [2e-05, 618059.0865027184],
    "He-3": [0.000166, 436624.4866546333],
    "He-4": [0.999834, 2629831367.4087267],
    "Li-6": [0.0759, 0.026317326949347204],
    "Li-7": [0.9240999999999999, 0.3204195235031851],
    "Be-9": [1.0, 0.7413102413009162],
    "B-10": [0.19899999999999998, 3.0821450716358334],
    "B-11": [0.8009999999999999, 12.406021117488958],
    "C-12": [0.988938, 8225628.002667331],
    "C-13": [0.011062, 92009.70835937746],
    "N-14": [0.99771, 2084511.6427143852],
    "N-15": [0.00229, 4784.488139655753],
    "O-16": [0.9976210000000001, 15099604.862261778],
    "O-17": [0.00037900000000000005, 5736.397131573227],
    "O-18": [0.002, 30271.224968724146],
    "F-19": [1.0, 1122.018454301963],
    "Ne-20": [0.929431, 2444652.609975316],
    "Ne-21": [0.002228, 5860.237085942909],
    "Ne-22": [0.068341, 179755.14483412224],
    "Na-23": [1.0, 53703.17963702533],
    "Mg-24": [0.7898999999999999, 971789.3020646997],
    "Mg-25": [0.1, 123026.87708123811],
    "Mg-26": [0.1101, 135452.59166644316],
    "Al-27": [1.0, 87096.35899560814],
    "Si-28": [0.9222969999999999, 922296.9999999999],
    "Si-29": [0.046832000000000006, 46832.00000000001],
    "Si-30": [0.030872, 30872.0],
    "P-31": [1.0, 7943.282347242821],
    "S-32": [0.9493, 386726.09771944454],
    "S-33": [0.0076, 3096.090111311259],
    "S-34": [0.0429, 17476.61391779645],
    "S-36": [0.0002, 81.47605556082262],
    "Cl-35": [0.7578, 7405.503660403137],
    "Cl-37": [0.2422, 2366.8685491549745],
    "Ar-36": [0.845946, 65666.31433235761],
    "Ar-38": [0.153808, 11939.301651442598],
    "Ar-40": [0.000246, 19.09567906906584],
    "K-39": [0.93132, 3083.890360591671],
    "K-40": [0.00147, 4.867627485794094],
    "K-41": [0.06721, 222.55322674844976],
    "Ca-40": [0.96941, 65540.1597174741],
    "Ca-42": [0.00647, 437.4256850786122],
    "Ca-43": [0.00135, 91.27120167791755],
    "Ca-44": [0.02086, 1410.309086667674],
    "Ca-46": [4e-05, 2.7043319015679272],
    "Ca-48": [0.00187, 126.42751639830058],
    "Sc-45": [1.0, 43.65158322401665],
    "Ti-46": [0.0825, 227.22386802539893],
    "Ti-47": [0.07440000000000001, 204.91461552835977],
    "Ti-48": [0.7372, 2030.4174001008978],
    "Ti-49": [0.0541, 149.0037728505949],
    "Ti-50": [0.0518, 142.66904683291713],
    "V-50": [0.0025, 0.6575669979738453],
    "V-51": [0.9975, 262.36923219156427],
    "Cr-50": [0.043449999999999996, 586.1243724860733],
    "Cr-52": [0.83789, 11302.825096947203],
    "Cr-53": [0.09501, 1281.6496347503296],
    "Cr-54": [0.02365, 319.029721732926],
    "Mn-55": [1.0, 8317.637711026708],
    "Fe-54": [0.058449999999999995, 57119.51556486716],
    "Fe-56": [0.91754, 896654.239715795],
    "Fe-57": [0.02119, 20707.65671205364],
    "Fe-58": [0.0028199999999999996, 2755.8089630953873],
    "Co-59": [1.0, 3019.951720402019],
    "Ni-58": [0.680769, 34914.01315184174],
    "Ni-60": [0.262231, 13448.815358543958],
    "Ni-61": [0.011399, 584.6106916117567],
    "Ni-62": [0.036345, 1863.9947001166154],
    "Ni-64": [0.009256, 474.7044970224073],
    "Cu-63": [0.6917, 331.06843485956944],
    "Cu-65": [0.30829999999999996, 147.5616574630696],
    "Zn-64": [0.4863, 545.6375743270446],
    "Zn-66": [0.27899999999999997, 313.0431487502476],
    "Zn-67": [0.040999999999999995, 46.002756626380474],
    "Zn-68": [0.1875, 210.37846018161804],
    "Zn-70": [0.0062, 6.95651441667217],
    "Ga-69": [0.60108, 20.367244537215154],
    "Ga-71": [0.39892000000000005, 13.517171076705049],
    "Ge-70": [0.2084, 28.767208074324156],
    "Ge-72": [0.2754, 38.0157826471635],
    "Ge-73": [0.07730000000000001, 10.670370365380315],
    "Ge-74": [0.3628, 50.08034111979273],
    "Ge-76": [0.0761, 10.504724253627968],
    "As-75": [1.0, NaN],
    "Se-74": [0.0089, NaN],
    "Se-76": [0.09369999999999999, NaN],
    "Se-77": [0.07629999999999999, NaN],
    "Se-78": [0.2377, NaN],
    "Se-80": [0.4961, NaN],
    "Se-82": [0.0873, NaN],
    "Br-79": [0.5069, NaN],
    "Br-81": [0.49310000000000004, NaN],
    "Kr-78": [0.00362, 0.19893379633646016],
    "Kr-80": [0.02326, 1.2782320725928351],
    "Kr-82": [0.11654999999999999, 6.404898884810616],
    "Kr-83": [0.11546, 6.344998929560135],
    "Kr-84": [0.56903, 31.270524345120425],
    "Kr-86": [0.17207999999999998, 9.456499357342006],
    "Rb-85": [0.70844, 7.24941687451377],
    "Rb-87": [0.29156, 2.983513048293765],
    "Sr-84": [0.005580000000000001, 0.1278304150244421],
    "Sr-86": [0.098678, 2.260582382398189],
    "Sr-87": [0.068961, 1.579805242025188],
    "Sr-88": [0.826781, 18.94045848822997],
    "Y-89": [1.0, 5.011872336272734],
    "Zr-90": [0.5145000000000001, 6.044847920163884],
    "Zr-91": [0.11220000000000001, 1.318235056642153],
    "Zr-92": [0.17149999999999999, 2.0149493067212942],
    "Zr-94": [0.17379999999999998, 2.0419719504849034],
    "Zr-96": [0.027999999999999997, 0.32897131538306845],
    "Nb-93": [1.0, 0.8912509381337477],
    "Mo-92": [0.14525, 0.340499235425218],
    "Mo-94": [0.09151, 0.21452037888992564],
    "Mo-95": [0.15838, 0.3712789597703685],
    "Mo-96": [0.16672, 0.39082982809013667],
    "Mo-97": [0.09599, 0.2250225239825589],
    "Mo-98": [0.24391, 0.5717808503446811],
    "Mo-100": [0.09824, 0.23029703881702868],
    "Ru-96": [0.0554, 0.09627416591271544],
    "Ru-98": [0.0187, 0.03249687549761334],
    "Ru-99": [0.1276, 0.2217433857484204],
    "Ru-100": [0.126, 0.21896290442242142],
    "Ru-101": [0.17059999999999997, 0.29646882138464353],
    "Ru-102": [0.3155, 0.5482761614704282],
    "Ru-104": [0.1862, 0.3235785143131339],
    "Rh-103": [1.0, 0.2511886431509582],
    "Pd-102": [0.0102, 0.011711166939268217],
    "Pd-104": [0.1114, 0.12790431343475286],
    "Pd-105": [0.22329999999999997, 0.2563827036802542],
    "Pd-106": [0.2733, 0.31379038475509835],
    "Pd-108": [0.2646, 0.3038014482480755],
    "Pd-110": [0.11720000000000001, 0.13456360443943483],
    "Ag-107": [0.51839, 0.1395264727007673],
    "Ag-109": [0.48161000000000004, 0.12962700769192412],
    "Cd-106": [0.0125, NaN],
    "Cd-108": [0.0089, NaN],
    "Cd-110": [0.1249, NaN],
    "Cd-111": [0.128, NaN],
    "Cd-112": [0.2413, NaN],
    "Cd-113": [0.1222, NaN],
    "Cd-114": [0.2873, NaN],
    "Cd-116": [0.07490000000000001, NaN],
    "In-113": [0.0429, 0.008364833332962032],
    "In-115": [0.9571, 0.1866196266428429],
    "Sn-112": [0.0097, 0.0328678831455026],
    "Sn-114": [0.0066, 0.022363714305187334],
    "Sn-115": [0.0034000000000000002, 0.01152070130873287],
    "Sn-116": [0.1454, 0.49267940302639973],
    "Sn-117": [0.0768, 0.26023231191490714],
    "Sn-118": [0.2422, 0.8206805461691473],
    "Sn-119": [0.0859, 0.29106713012357455],
    "Sn-120": [0.3258, 1.10395426070152],
    "Sn-122": [0.0463, 0.15688484429245053],
    "Sn-124": [0.0579, 0.19619076640459798],
    "Sb-121": [0.5721, NaN],
    "Sb-123": [0.4279, NaN],
    "Te-120": [0.0009, NaN],
    "Te-122": [0.0255, NaN],
    "Te-123": [0.0089, NaN],
    "Te-124": [0.047400000000000005, NaN],
    "Te-125": [0.0707, NaN],
    "Te-126": [0.1884, NaN],
    "Te-128": [0.31739999999999996, NaN],
    "Te-130": [0.3408, NaN],
    "I-127": [1.0, NaN],
    "Xe-124": [0.00122, 0.006551787915717089],
    "Xe-126": [0.00108, 0.005799943400798735],
    "Xe-128": [0.02188, 0.1175025570458114],
    "Xe-129": [0.27255, 1.4636801610071253],
    "Xe-130": [0.04376, 0.2350051140916228],
    "Xe-131": [0.21693, 1.1649830758659903],
    "Xe-132": [0.26514, 1.4238861048960894],
    "Xe-134": [0.09789999999999999, 0.5257541286464779],
    "Xe-136": [0.07954, 0.4271550908328994],
    "Cs-133": [1.0, NaN],
    "Ba-130": [0.00106, 0.0049579924976442995],
    "Ba-132": [0.00101, 0.004724124927000701],
    "Ba-134": [0.024169999999999997, 0.11305158364911576],
    "Ba-135": [0.06591999999999999, 0.30833100513652095],
    "Ba-136": [0.07854, 0.3673591799669654],
    "Ba-137": [0.11231999999999999, 0.5253601106937809],
    "Ba-138": [0.71698, 3.3535674160009528],
    "La-138": [0.00091, 0.00035403108194479523],
    "La-139": [0.99909, 0.3886911139123357],
    "Ce-136": [0.00185, 0.002173560476638131],
    "Ce-138": [0.00251, 0.002948992862898221],
    "Ce-140": [0.8845000000000001, 1.0391968873440145],
    "Ce-142": [0.11114, 0.1305781142559794],
    "Pr-141": [1.0, 0.1621810097358933],
    "Nd-142": [0.27044, 0.21982188479141906],
    "Nd-143": [0.12023, 0.09772661295840968],
    "Nd-144": [0.23729, 0.19287655318057914],
    "Nd-145": [0.08763, 0.07122833813146003],
    "Nd-146": [0.17129999999999998, 0.1392378674189102],
    "Nd-148": [0.05716, 0.046461392303939926],
    "Nd-150": [0.05596, 0.04548599568454301],
    "Sm-144": [0.030699999999999998, 0.008652435598981895],
    "Sm-147": [0.1499, 0.04224756013965427],
    "Sm-148": [0.1124, 0.03167862414741254],
    "Sm-149": [0.1382, 0.038950052110074844],
    "Sm-150": [0.0738, 0.02079966603273172],
    "Sm-152": [0.2675, 0.07539174341132433],
    "Sm-154": [0.2275, 0.06411821168626648],
    "Eu-151": [0.4781, 0.04892363796094283],
    "Eu-153": [0.5219, 0.05340566126713253],
    "Gd-152": [0.002, 0.0007261561095402035],
    "Gd-154": [0.0218, 0.007915101593988219],
    "Gd-155": [0.14800000000000002, 0.05373555210597507],
    "Gd-156": [0.2047, 0.07432207781143982],
    "Gd-157": [0.1565, 0.05682171557152092],
    "Gd-158": [0.2484, 0.09018858880489328],
    "Gd-160": [0.2186, 0.07936886277274424],
    "Tb-159": [1.0, 0.06165950018614834],
    "Dy-156": [0.0005600000000000001, 0.00021786528119679708],
    "Dy-158": [0.00095, 0.0003695928877445664],
    "Dy-160": [0.02329, 0.009060861426916792],
    "Dy-161": [0.18889, 0.07348673743796963],
    "Dy-162": [0.25475000000000003, 0.09910925068729295],
    "Dy-163": [0.24896000000000001, 0.09685667929777607],
    "Dy-164": [0.2826, 0.10994415797538366],
    "Ho-165": [1.0, 0.09332543007969923],
    "Er-162": [0.0013900000000000002, 0.0003572850138048722],
    "Er-164": [0.01601, 0.004115203648212952],
    "Er-166": [0.33503, 0.08611596991010527],
    "Er-167": [0.22869, 0.058782381156141164],
    "Er-168": [0.26978, 0.06934413742753844],
    "Er-170": [0.1491, 0.038324601121083775],
    "Tm-169": [1.0, 0.03890451449942804],
    "Yb-168": [0.0012, 0.0002565554507402678],
    "Yb-170": [0.0298, 0.006371127026716651],
    "Yb-171": [0.1409, 0.030123885841086447],
    "Yb-172": [0.2169, 0.046372397721303416],
    "Yb-173": [0.161, 0.034421189640985936],
    "Yb-174": [0.32030000000000003, 0.0684789257267565],
    "Yb-176": [0.13, 0.027793507163529015],
    "Lu-175": [0.9717950000000001, 0.03780721266797168],
    "Lu-176": [0.028205, 0.001097301831456368],
    "Hf-174": [0.0016200000000000001, 0.0003544173830798274],
    "Hf-176": [0.05206, 0.011389487014281366],
    "Hf-177": [0.18606, 0.04070549277520536],
    "Hf-178": [0.27297, 0.059719329048950914],
    "Hf-179": [0.13629, 0.02981700317280844],
    "Hf-180": [0.35100000000000003, 0.07679043300062928],
    "Ta-180": [0.00012, NaN],
    "Ta-181": [0.99988, NaN],
    "W-180": [0.0012, 0.0002625313948739462],
    "W-182": [0.265, 0.05797568303466313],
    "W-183": [0.1431, 0.03130686883871809],
    "W-184": [0.3064, 0.06703301615781426],
    "W-186": [0.2843, 0.062198062968885755],
    "Re-185": [0.35662, NaN],
    "Re-187": [0.64338, NaN],
    "Os-184": [0.0002, 0.00015524942332573855],
    "Os-186": [0.01598, 0.01240442892372651],
    "Os-187": [0.012709999999999999, 0.009866100852350684],
    "Os-188": [0.13337, 0.10352807794476875],
    "Os-189": [0.16261, 0.12622554363499172],
    "Os-190": [0.26444, 0.20527078752129152],
    "Os-192": [0.4107, 0.3188046907994041],
    "Ir-191": [0.373, 0.27650872000524174],
    "Ir-193": [0.627, 0.4648015212956744],
    "Pt-190": [0.00014000000000000001, NaN],
    "Pt-192": [0.00782, NaN],
    "Pt-194": [0.32966999999999996, NaN],
    "Pt-195": [0.33832, NaN],
    "Pt-196": [0.25242000000000003, NaN],
    "Pt-198": [0.07163, NaN],
    "Au-197": [1.0, 0.25703957827688645],
    "Hg-196": [0.0015, NaN],
    "Hg-198": [0.09970000000000001, NaN],
    "Hg-199": [0.16870000000000002, NaN],
    "Hg-200": [0.231, NaN],
    "Hg-201": [0.1318, NaN],
    "Hg-202": [0.2986, NaN],
    "Hg-204": [0.0687, NaN],
    "Tl-203": [0.29524, 0.07247282602668494],
    "Tl-205": [0.7047599999999999, 0.1729980655418184],
    "Pb-204": [0.01997, 0.03470388255012505],
    "Pb-206": [0.18582, 0.3229181499982091],
    "Pb-207": [0.20562999999999998, 0.35734398441573423],
    "Pb-208": [0.58858, 1.0228348117853079],
    "Bi-209": [1.0, NaN],
    "Th-232": [1.0, 0.032359365692962806],
    "U-234": [2e-05, NaN],
    "U-235": [0.24286000000000002, NaN],
    "U-238": [0.75712, NaN],
}
