"""NIST isotope data, retrieved in 2020.

This file was automatically created using the `nist_data.py` parser available in the
dev/nist_data folder.

The atomic weights are available for elements 1 through 118 and isotopic compositions
or abundances are given when appropriate. The atomic weights data were published by
J. Meija et al in Atomic Weights of the Elements 2013, and the isotopic compositions
data were published by M. Berglund and M.E. Wieser in Isotopic Compositions of the
Elements 2009. The relative atomic masses of the isotopes data were published by
M. Wang, G. Audi, A.H. Wapstra, F.G. Kondev, M. MacCormick, X. Xu1, and B. Pfeiffer in
The AME2012 Atomic Mass Evaluation.

The NIST data were exported on July 1, 2020 from the NIST database:
https://www.nist.gov/pml/atomic-weights-and-isotopic-compositions-relative-atomic-masses

The text file with the NIST data can be seen in dev/nist_data as well.
"""

import numpy as np

NaN = np.nan

elements_mass = {
    "H": 1.0079407540557772,
    "He": 4.002601932120929,
    "Li": 6.94003660291572,
    "Be": 9.012183065,
    "B": 10.811028046410001,
    "C": 12.010735896735248,
    "N": 14.006703211445798,
    "O": 15.999404924318277,
    "F": 18.99840316273,
    "Ne": 20.18004638052026,
    "Na": 22.989769282,
    "Mg": 24.3050516198371,
    "Al": 26.98153853,
    "Si": 28.08549870570596,
    "P": 30.97376199842,
    "S": 32.06478740612707,
    "Cl": 35.452937582608,
    "Ar": 39.947798563582005,
    "K": 39.098300910086,
    "Ca": 40.078022511017735,
    "Sc": 44.95590828,
    "Ti": 47.866744962722,
    "V": 50.941465037425004,
    "Cr": 51.9961317554337,
    "Mn": 54.93804391,
    "Fe": 55.845144433865904,
    "Co": 58.93319429,
    "Ni": 58.69334710994765,
    "Cu": 63.54603994583,
    "Zn": 65.37778252952499,
    "Ga": 69.7230660725936,
    "Ge": 72.6275501646868,
    "As": 74.92159457,
    "Se": 78.95938855701361,
    "Br": 79.90352778050999,
    "Kr": 83.7979999953261,
    "Rb": 85.46766359561973,
    "Sr": 87.61664446962,
    "Y": 88.9058403,
    "Zr": 91.22364159706,
    "Nb": 92.906373,
    "Mo": 95.95978854118802,
    "Tc": NaN,
    "Ru": 101.06494013916,
    "Rh": 102.905498,
    "Pd": 106.41532750734,
    "Ag": 107.868149634557,
    "Cd": 112.411557818268,
    "In": 114.81808662944559,
    "Sn": 118.7101125930106,
    "Sb": 121.75978367348,
    "Te": 127.60312648466041,
    "I": 126.9044719,
    "Xe": 131.29276144779053,
    "Cs": 132.905451961,
    "Ba": 137.3268916286322,
    "La": 138.90546887371266,
    "Ce": 140.1157307378545,
    "Pr": 140.9076576,
    "Nd": 144.24159603182702,
    "Pm": NaN,
    "Sm": 150.36635571193,
    "Eu": 151.96437812637998,
    "Gd": 157.25213064687998,
    "Tb": 158.9253547,
    "Dy": 162.499472819424,
    "Ho": 164.9303288,
    "Er": 167.259082649669,
    "Tm": 168.9342179,
    "Yb": 173.05415016631704,
    "Lu": 174.96681495785498,
    "Hf": 178.4849787234,
    "Ta": 180.9478756362269,
    "W": 183.841777550513,
    "Re": 186.20670454560002,
    "Os": 190.22485962823998,
    "Ir": 192.2160516521,
    "Pt": 195.08445686493104,
    "Au": 196.96656879,
    "Hg": 200.59916703455602,
    "Tl": 204.38341283936,
    "Pb": 207.216908063,
    "Bi": 208.9803991,
    "Po": NaN,
    "At": NaN,
    "Rn": NaN,
    "Fr": NaN,
    "Ra": NaN,
    "Ac": NaN,
    "Th": 232.0380558,
    "Pa": 231.0358842,
    "U": 238.0289104616574,
    "Np": NaN,
    "Pu": NaN,
    "Am": NaN,
    "Cm": NaN,
    "Bk": NaN,
    "Cf": NaN,
    "Es": NaN,
    "Fm": NaN,
    "Md": NaN,
    "No": NaN,
    "Lr": NaN,
    "Rf": NaN,
    "Db": NaN,
    "Sg": NaN,
    "Bh": NaN,
    "Hs": NaN,
    "Mt": NaN,
    "Ds": NaN,
    "Rg": NaN,
    "Cn": NaN,
    "Nh": NaN,
    "Fl": NaN,
    "Mc": NaN,
    "Lv": NaN,
    "Ts": NaN,
    "Og": NaN,
}

elements_z = {
    "H": 1,
    "He": 2,
    "Li": 3,
    "Be": 4,
    "B": 5,
    "C": 6,
    "N": 7,
    "O": 8,
    "F": 9,
    "Ne": 10,
    "Na": 11,
    "Mg": 12,
    "Al": 13,
    "Si": 14,
    "P": 15,
    "S": 16,
    "Cl": 17,
    "Ar": 18,
    "K": 19,
    "Ca": 20,
    "Sc": 21,
    "Ti": 22,
    "V": 23,
    "Cr": 24,
    "Mn": 25,
    "Fe": 26,
    "Co": 27,
    "Ni": 28,
    "Cu": 29,
    "Zn": 30,
    "Ga": 31,
    "Ge": 32,
    "As": 33,
    "Se": 34,
    "Br": 35,
    "Kr": 36,
    "Rb": 37,
    "Sr": 38,
    "Y": 39,
    "Zr": 40,
    "Nb": 41,
    "Mo": 42,
    "Tc": 43,
    "Ru": 44,
    "Rh": 45,
    "Pd": 46,
    "Ag": 47,
    "Cd": 48,
    "In": 49,
    "Sn": 50,
    "Sb": 51,
    "Te": 52,
    "I": 53,
    "Xe": 54,
    "Cs": 55,
    "Ba": 56,
    "La": 57,
    "Ce": 58,
    "Pr": 59,
    "Nd": 60,
    "Pm": 61,
    "Sm": 62,
    "Eu": 63,
    "Gd": 64,
    "Tb": 65,
    "Dy": 66,
    "Ho": 67,
    "Er": 68,
    "Tm": 69,
    "Yb": 70,
    "Lu": 71,
    "Hf": 72,
    "Ta": 73,
    "W": 74,
    "Re": 75,
    "Os": 76,
    "Ir": 77,
    "Pt": 78,
    "Au": 79,
    "Hg": 80,
    "Tl": 81,
    "Pb": 82,
    "Bi": 83,
    "Po": 84,
    "At": 85,
    "Rn": 86,
    "Fr": 87,
    "Ra": 88,
    "Ac": 89,
    "Th": 90,
    "Pa": 91,
    "U": 92,
    "Np": 93,
    "Pu": 94,
    "Am": 95,
    "Cm": 96,
    "Bk": 97,
    "Cf": 98,
    "Es": 99,
    "Fm": 100,
    "Md": 101,
    "No": 102,
    "Lr": 103,
    "Rf": 104,
    "Db": 105,
    "Sg": 106,
    "Bh": 107,
    "Hs": 108,
    "Mt": 109,
    "Ds": 110,
    "Rg": 111,
    "Cn": 112,
    "Nh": 113,
    "Fl": 114,
    "Mc": 115,
    "Lv": 116,
    "Ts": 117,
    "Og": 118,
}

isotopes_mass = {
    "H-1": 1.00782503223,
    "H-2": 2.01410177812,
    "H-3": 3.0160492779,
    "He-3": 3.0160293201,
    "He-4": 4.00260325413,
    "Li-6": 6.0151228874,
    "Li-7": 7.0160034366,
    "Be-9": 9.012183065,
    "B-10": 10.01293695,
    "B-11": 11.00930536,
    "C-12": 12.0,
    "C-13": 13.00335483507,
    "C-14": 14.0032419884,
    "N-14": 14.00307400443,
    "N-15": 15.00010889888,
    "O-16": 15.99491461957,
    "O-17": 16.9991317565,
    "O-18": 17.99915961286,
    "F-19": 18.99840316273,
    "Ne-20": 19.9924401762,
    "Ne-21": 20.993846685,
    "Ne-22": 21.991385114,
    "Na-23": 22.989769282,
    "Mg-24": 23.985041697,
    "Mg-25": 24.985836976,
    "Mg-26": 25.982592968,
    "Al-27": 26.98153853,
    "Si-28": 27.97692653465,
    "Si-29": 28.9764946649,
    "Si-30": 29.973770136,
    "P-31": 30.97376199842,
    "S-32": 31.9720711744,
    "S-33": 32.9714589098,
    "S-34": 33.967867004,
    "S-36": 35.96708071,
    "Cl-35": 34.968852682,
    "Cl-37": 36.965902602,
    "Ar-36": 35.967545105,
    "Ar-38": 37.96273211,
    "Ar-40": 39.9623831237,
    "K-39": 38.9637064864,
    "K-40": 39.963998166,
    "K-41": 40.9618252579,
    "Ca-40": 39.962590863,
    "Ca-42": 41.95861783,
    "Ca-43": 42.95876644,
    "Ca-44": 43.95548156,
    "Ca-46": 45.953689,
    "Ca-48": 47.95252276,
    "Sc-45": 44.95590828,
    "Ti-46": 45.95262772,
    "Ti-47": 46.95175879,
    "Ti-48": 47.94794198,
    "Ti-49": 48.94786568,
    "Ti-50": 49.94478689,
    "V-50": 49.94715601,
    "V-51": 50.94395704,
    "Cr-50": 49.94604183,
    "Cr-52": 51.94050623,
    "Cr-53": 52.94064815,
    "Cr-54": 53.93887916,
    "Mn-55": 54.93804391,
    "Fe-54": 53.93960899,
    "Fe-56": 55.93493633,
    "Fe-57": 56.93539284,
    "Fe-58": 57.93327443,
    "Co-59": 58.93319429,
    "Ni-58": 57.93534241,
    "Ni-60": 59.93078588,
    "Ni-61": 60.93105557,
    "Ni-62": 61.92834537,
    "Ni-64": 63.92796682,
    "Cu-63": 62.92959772,
    "Cu-65": 64.9277897,
    "Zn-64": 63.92914201,
    "Zn-66": 65.92603381,
    "Zn-67": 66.92712775,
    "Zn-68": 67.92484455,
    "Zn-70": 69.9253192,
    "Ga-69": 68.9255735,
    "Ga-71": 70.92470258,
    "Ge-70": 69.92424875,
    "Ge-72": 71.922075826,
    "Ge-73": 72.923458956,
    "Ge-74": 73.921177761,
    "Ge-76": 75.921402726,
    "As-75": 74.92159457,
    "Se-74": 73.922475934,
    "Se-76": 75.919213704,
    "Se-77": 76.919914154,
    "Se-78": 77.91730928,
    "Se-80": 79.9165218,
    "Se-82": 81.9166995,
    "Br-79": 78.9183376,
    "Br-81": 80.9162897,
    "Kr-78": 77.92036494,
    "Kr-80": 79.91637808,
    "Kr-82": 81.91348273,
    "Kr-83": 82.91412716,
    "Kr-84": 83.9114977282,
    "Kr-86": 85.9106106269,
    "Rb-85": 84.9117897379,
    "Rb-87": 86.909180531,
    "Sr-84": 83.9134191,
    "Sr-86": 85.9092606,
    "Sr-87": 86.9088775,
    "Sr-88": 87.9056125,
    "Y-89": 88.9058403,
    "Zr-90": 89.9046977,
    "Zr-91": 90.9056396,
    "Zr-92": 91.9050347,
    "Zr-94": 93.9063108,
    "Zr-96": 95.9082714,
    "Nb-93": 92.906373,
    "Mo-92": 91.90680796,
    "Mo-94": 93.9050849,
    "Mo-95": 94.90583877,
    "Mo-96": 95.90467612,
    "Mo-97": 96.90601812,
    "Mo-98": 97.90540482,
    "Mo-100": 99.9074718,
    "Tc-97": 96.9063667,
    "Tc-98": 97.9072124,
    "Tc-99": 98.9062508,
    "Ru-96": 95.90759025,
    "Ru-98": 97.9052868,
    "Ru-99": 98.9059341,
    "Ru-100": 99.9042143,
    "Ru-101": 100.9055769,
    "Ru-102": 101.9043441,
    "Ru-104": 103.9054275,
    "Rh-103": 102.905498,
    "Pd-102": 101.9056022,
    "Pd-104": 103.9040305,
    "Pd-105": 104.9050796,
    "Pd-106": 105.9034804,
    "Pd-108": 107.9038916,
    "Pd-110": 109.9051722,
    "Ag-107": 106.9050916,
    "Ag-109": 108.9047553,
    "Cd-106": 105.9064599,
    "Cd-108": 107.9041834,
    "Cd-110": 109.90300661,
    "Cd-111": 110.90418287,
    "Cd-112": 111.90276287,
    "Cd-113": 112.90440813,
    "Cd-114": 113.90336509,
    "Cd-116": 115.90476315,
    "In-113": 112.90406184,
    "In-115": 114.903878776,
    "Sn-112": 111.90482387,
    "Sn-114": 113.9027827,
    "Sn-115": 114.903344699,
    "Sn-116": 115.9017428,
    "Sn-117": 116.90295398,
    "Sn-118": 117.90160657,
    "Sn-119": 118.90331117,
    "Sn-120": 119.90220163,
    "Sn-122": 121.9034438,
    "Sn-124": 123.9052766,
    "Sb-121": 120.903812,
    "Sb-123": 122.9042132,
    "Te-120": 119.9040593,
    "Te-122": 121.9030435,
    "Te-123": 122.9042698,
    "Te-124": 123.9028171,
    "Te-125": 124.9044299,
    "Te-126": 125.9033109,
    "Te-128": 127.90446128,
    "Te-130": 129.906222748,
    "I-127": 126.9044719,
    "Xe-124": 123.905892,
    "Xe-126": 125.9042983,
    "Xe-128": 127.903531,
    "Xe-129": 128.9047808611,
    "Xe-130": 129.903509349,
    "Xe-131": 130.90508406,
    "Xe-132": 131.9041550856,
    "Xe-134": 133.90539466,
    "Xe-136": 135.907214484,
    "Cs-133": 132.905451961,
    "Ba-130": 129.9063207,
    "Ba-132": 131.9050611,
    "Ba-134": 133.90450818,
    "Ba-135": 134.90568838,
    "Ba-136": 135.90457573,
    "Ba-137": 136.90582714,
    "Ba-138": 137.905247,
    "La-138": 137.9071149,
    "La-139": 138.9063563,
    "Ce-136": 135.90712921,
    "Ce-138": 137.905991,
    "Ce-140": 139.9054431,
    "Ce-142": 141.9092504,
    "Pr-141": 140.9076576,
    "Nd-142": 141.907729,
    "Nd-143": 142.90982,
    "Nd-144": 143.910093,
    "Nd-145": 144.9125793,
    "Nd-146": 145.9131226,
    "Nd-148": 147.9168993,
    "Nd-150": 149.9209022,
    "Pm-145": 144.9127559,
    "Pm-147": 146.915145,
    "Sm-144": 143.9120065,
    "Sm-147": 146.9149044,
    "Sm-148": 147.9148292,
    "Sm-149": 148.9171921,
    "Sm-150": 149.9172829,
    "Sm-152": 151.9197397,
    "Sm-154": 153.9222169,
    "Eu-151": 150.9198578,
    "Eu-153": 152.921238,
    "Gd-152": 151.9197995,
    "Gd-154": 153.9208741,
    "Gd-155": 154.9226305,
    "Gd-156": 155.9221312,
    "Gd-157": 156.9239686,
    "Gd-158": 157.9241123,
    "Gd-160": 159.9270624,
    "Tb-159": 158.9253547,
    "Dy-156": 155.9242847,
    "Dy-158": 157.9244159,
    "Dy-160": 159.9252046,
    "Dy-161": 160.9269405,
    "Dy-162": 161.9268056,
    "Dy-163": 162.9287383,
    "Dy-164": 163.9291819,
    "Ho-165": 164.9303288,
    "Er-162": 161.9287884,
    "Er-164": 163.9292088,
    "Er-166": 165.9302995,
    "Er-167": 166.9320546,
    "Er-168": 167.9323767,
    "Er-170": 169.9354702,
    "Tm-169": 168.9342179,
    "Yb-168": 167.9338896,
    "Yb-170": 169.9347664,
    "Yb-171": 170.9363302,
    "Yb-172": 171.9363859,
    "Yb-173": 172.9382151,
    "Yb-174": 173.9388664,
    "Yb-176": 175.9425764,
    "Lu-175": 174.9407752,
    "Lu-176": 175.9426897,
    "Hf-174": 173.9400461,
    "Hf-176": 175.9414076,
    "Hf-177": 176.9432277,
    "Hf-178": 177.9437058,
    "Hf-179": 178.9458232,
    "Hf-180": 179.946557,
    "Ta-180": 179.9474648,
    "Ta-181": 180.9479958,
    "W-180": 179.9467108,
    "W-182": 181.94820394,
    "W-183": 182.95022275,
    "W-184": 183.95093092,
    "W-186": 185.9543628,
    "Re-185": 184.9529545,
    "Re-187": 186.9557501,
    "Os-184": 183.9524885,
    "Os-186": 185.953835,
    "Os-187": 186.9557474,
    "Os-188": 187.9558352,
    "Os-189": 188.9581442,
    "Os-190": 189.9584437,
    "Os-192": 191.961477,
    "Ir-191": 190.9605893,
    "Ir-193": 192.9629216,
    "Pt-190": 189.9599297,
    "Pt-192": 191.9610387,
    "Pt-194": 193.9626809,
    "Pt-195": 194.9647917,
    "Pt-196": 195.96495209,
    "Pt-198": 197.9678949,
    "Au-197": 196.96656879,
    "Hg-196": 195.9658326,
    "Hg-198": 197.9667686,
    "Hg-199": 198.96828064,
    "Hg-200": 199.96832659,
    "Hg-201": 200.97030284,
    "Hg-202": 201.9706434,
    "Hg-204": 203.97349398,
    "Tl-203": 202.9723446,
    "Tl-205": 204.9744278,
    "Pb-204": 203.973044,
    "Pb-206": 205.9744657,
    "Pb-207": 206.9758973,
    "Pb-208": 207.9766525,
    "Bi-209": 208.9803991,
    "Po-209": 208.9824308,
    "Po-210": 209.9828741,
    "At-210": 209.9871479,
    "At-211": 210.9874966,
    "Rn-211": 210.9906011,
    "Rn-220": 220.0113941,
    "Rn-222": 222.0175782,
    "Fr-223": 223.019736,
    "Ra-223": 223.0185023,
    "Ra-224": 224.020212,
    "Ra-226": 226.0254103,
    "Ra-228": 228.0310707,
    "Ac-227": 227.0277523,
    "Th-230": 230.0331341,
    "Th-232": 232.0380558,
    "Pa-231": 231.0358842,
    "U-233": 233.0396355,
    "U-234": 234.0409523,
    "U-235": 235.0439301,
    "U-236": 236.0455682,
    "U-238": 238.0507884,
    "Np-236": 236.04657,
    "Np-237": 237.0481736,
    "Pu-238": 238.0495601,
    "Pu-239": 239.0521636,
    "Pu-240": 240.0538138,
    "Pu-241": 241.0568517,
    "Pu-242": 242.0587428,
    "Pu-244": 244.0642053,
    "Am-241": 241.0568293,
    "Am-243": 243.0613813,
    "Cm-243": 243.0613893,
    "Cm-244": 244.0627528,
    "Cm-245": 245.0654915,
    "Cm-246": 246.0672238,
    "Cm-247": 247.0703541,
    "Cm-248": 248.0723499,
    "Bk-247": 247.0703073,
    "Bk-249": 249.0749877,
    "Cf-249": 249.0748539,
    "Cf-250": 250.0764062,
    "Cf-251": 251.0795886,
    "Cf-252": 252.0816272,
    "Es-252": 252.08298,
    "Fm-257": 257.0951061,
    "Md-258": 258.0984315,
    "Md-260": 260.10365,
    "No-259": 259.10103,
    "Lr-262": 262.10961,
    "Rf-267": 267.12179,
    "Db-268": 268.12567,
    "Sg-271": 271.13393,
    "Bh-272": 272.13826,
    "Hs-270": 270.13429,
    "Mt-276": 276.15159,
    "Ds-281": 281.16451,
    "Rg-280": 280.16514,
    "Cn-285": 285.17712,
    "Nh-284": 284.17873,
    "Fl-289": 289.19042,
    "Mc-288": 288.19274,
    "Lv-293": 293.20449,
    "Ts-292": 292.20746,
    "Og-294": 294.21392,
}

nist15_elements = {
    "H": [NaN, [1, 2, 3], [0.999885, 0.000115, 0.0], [NaN, NaN, NaN]],
    "He": [NaN, [3, 4], [1.34e-06, 0.99999866], [NaN, NaN]],
    "Li": [NaN, [6, 7], [0.0759, 0.9241], [NaN, NaN]],
    "Be": [NaN, [9], [1.0], [NaN]],
    "B": [NaN, [10, 11], [0.199, 0.801], [NaN, NaN]],
    "C": [NaN, [12, 13, 14], [0.9893, 0.0107, 0.0], [NaN, NaN, NaN]],
    "N": [NaN, [14, 15], [0.99636, 0.00364], [NaN, NaN]],
    "O": [NaN, [16, 17, 18], [0.99757, 0.00038, 0.00205], [NaN, NaN, NaN]],
    "F": [NaN, [19], [1.0], [NaN]],
    "Ne": [NaN, [20, 21, 22], [0.9048, 0.0027, 0.0925], [NaN, NaN, NaN]],
    "Na": [NaN, [23], [1.0], [NaN]],
    "Mg": [NaN, [24, 25, 26], [0.7899, 0.1, 0.1101], [NaN, NaN, NaN]],
    "Al": [NaN, [27], [1.0], [NaN]],
    "Si": [NaN, [28, 29, 30], [0.92223, 0.04685, 0.03092], [NaN, NaN, NaN]],
    "P": [NaN, [31], [1.0], [NaN]],
    "S": [
        NaN,
        [32, 33, 34, 36],
        [0.9499, 0.0075, 0.0425, 0.0001],
        [NaN, NaN, NaN, NaN],
    ],
    "Cl": [NaN, [35, 37], [0.7576, 0.2424], [NaN, NaN]],
    "Ar": [NaN, [36, 38, 40], [0.003336, 0.000629, 0.996035], [NaN, NaN, NaN]],
    "K": [NaN, [39, 40, 41], [0.932581, 0.000117, 0.067302], [NaN, NaN, NaN]],
    "Ca": [
        NaN,
        [40, 42, 43, 44, 46, 48],
        [0.96941, 0.00647, 0.00135, 0.02086, 4e-05, 0.00187],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Sc": [NaN, [45], [1.0], [NaN]],
    "Ti": [
        NaN,
        [46, 47, 48, 49, 50],
        [0.0825, 0.0744, 0.7372, 0.0541, 0.0518],
        [NaN, NaN, NaN, NaN, NaN],
    ],
    "V": [NaN, [50, 51], [0.0025, 0.9975], [NaN, NaN]],
    "Cr": [
        NaN,
        [50, 52, 53, 54],
        [0.04345, 0.83789, 0.09501, 0.02365],
        [NaN, NaN, NaN, NaN],
    ],
    "Mn": [NaN, [55], [1.0], [NaN]],
    "Fe": [
        NaN,
        [54, 56, 57, 58],
        [0.05845, 0.91754, 0.02119, 0.00282],
        [NaN, NaN, NaN, NaN],
    ],
    "Co": [NaN, [59], [1.0], [NaN]],
    "Ni": [
        NaN,
        [58, 60, 61, 62, 64],
        [0.68077, 0.26223, 0.011399, 0.036346, 0.009255],
        [NaN, NaN, NaN, NaN, NaN],
    ],
    "Cu": [NaN, [63, 65], [0.6915, 0.3085], [NaN, NaN]],
    "Zn": [
        NaN,
        [64, 66, 67, 68, 70],
        [0.4917, 0.2773, 0.0404, 0.1845, 0.0061],
        [NaN, NaN, NaN, NaN, NaN],
    ],
    "Ga": [NaN, [69, 71], [0.60108, 0.39892], [NaN, NaN]],
    "Ge": [
        NaN,
        [70, 72, 73, 74, 76],
        [0.2057, 0.2745, 0.0775, 0.365, 0.0773],
        [NaN, NaN, NaN, NaN, NaN],
    ],
    "As": [NaN, [75], [1.0], [NaN]],
    "Se": [
        NaN,
        [74, 76, 77, 78, 80, 82],
        [0.0089, 0.0937, 0.0763, 0.2377, 0.4961, 0.0873],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Br": [NaN, [79, 81], [0.5069, 0.4931], [NaN, NaN]],
    "Kr": [
        NaN,
        [78, 80, 82, 83, 84, 86],
        [0.00355, 0.02286, 0.11593, 0.115, 0.56987, 0.17279],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Rb": [NaN, [85, 87], [0.7217, 0.2783], [NaN, NaN]],
    "Sr": [NaN, [84, 86, 87, 88], [0.0056, 0.0986, 0.07, 0.8258], [NaN, NaN, NaN, NaN]],
    "Y": [NaN, [89], [1.0], [NaN]],
    "Zr": [
        NaN,
        [90, 91, 92, 94, 96],
        [0.5145, 0.1122, 0.1715, 0.1738, 0.028],
        [NaN, NaN, NaN, NaN, NaN],
    ],
    "Nb": [NaN, [93], [1.0], [NaN]],
    "Mo": [
        NaN,
        [92, 94, 95, 96, 97, 98, 100],
        [0.1453, 0.0915, 0.1584, 0.1667, 0.096, 0.2439, 0.0982],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Tc": [NaN, [97, 98, 99], [0.0, 0.0, 0.0], [NaN, NaN, NaN]],
    "Ru": [
        NaN,
        [96, 98, 99, 100, 101, 102, 104],
        [0.0554, 0.0187, 0.1276, 0.126, 0.1706, 0.3155, 0.1862],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Rh": [NaN, [103], [1.0], [NaN]],
    "Pd": [
        NaN,
        [102, 104, 105, 106, 108, 110],
        [0.0102, 0.1114, 0.2233, 0.2733, 0.2646, 0.1172],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Ag": [NaN, [107, 109], [0.51839, 0.48161], [NaN, NaN]],
    "Cd": [
        NaN,
        [106, 108, 110, 111, 112, 113, 114, 116],
        [0.0125, 0.0089, 0.1249, 0.128, 0.2413, 0.1222, 0.2873, 0.0749],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "In": [NaN, [113, 115], [0.0429, 0.9571], [NaN, NaN]],
    "Sn": [
        NaN,
        [112, 114, 115, 116, 117, 118, 119, 120, 122, 124],
        [
            0.0097,
            0.0066,
            0.0034,
            0.1454,
            0.0768,
            0.2422,
            0.0859,
            0.3258,
            0.0463,
            0.0579,
        ],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Sb": [NaN, [121, 123], [0.5721, 0.4279], [NaN, NaN]],
    "Te": [
        NaN,
        [120, 122, 123, 124, 125, 126, 128, 130],
        [0.0009, 0.0255, 0.0089, 0.0474, 0.0707, 0.1884, 0.3174, 0.3408],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "I": [NaN, [127], [1.0], [NaN]],
    "Xe": [
        NaN,
        [124, 126, 128, 129, 130, 131, 132, 134, 136],
        [
            0.000952,
            0.00089,
            0.019102,
            0.264006,
            0.04071,
            0.212324,
            0.269086,
            0.104357,
            0.088573,
        ],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Cs": [NaN, [133], [1.0], [NaN]],
    "Ba": [
        NaN,
        [130, 132, 134, 135, 136, 137, 138],
        [0.00106, 0.00101, 0.02417, 0.06592, 0.07854, 0.11232, 0.71698],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "La": [NaN, [138, 139], [0.0008881, 0.9991119], [NaN, NaN]],
    "Ce": [
        NaN,
        [136, 138, 140, 142],
        [0.00185, 0.00251, 0.8845, 0.11114],
        [NaN, NaN, NaN, NaN],
    ],
    "Pr": [NaN, [141], [1.0], [NaN]],
    "Nd": [
        NaN,
        [142, 143, 144, 145, 146, 148, 150],
        [0.27152, 0.12174, 0.23798, 0.08293, 0.17189, 0.05756, 0.05638],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Pm": [NaN, [145, 147], [0.0, 0.0], [NaN, NaN]],
    "Sm": [
        NaN,
        [144, 147, 148, 149, 150, 152, 154],
        [0.0307, 0.1499, 0.1124, 0.1382, 0.0738, 0.2675, 0.2275],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Eu": [NaN, [151, 153], [0.4781, 0.5219], [NaN, NaN]],
    "Gd": [
        NaN,
        [152, 154, 155, 156, 157, 158, 160],
        [0.002, 0.0218, 0.148, 0.2047, 0.1565, 0.2484, 0.2186],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Tb": [NaN, [159], [1.0], [NaN]],
    "Dy": [
        NaN,
        [156, 158, 160, 161, 162, 163, 164],
        [0.00056, 0.00095, 0.02329, 0.18889, 0.25475, 0.24896, 0.2826],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Ho": [NaN, [165], [1.0], [NaN]],
    "Er": [
        NaN,
        [162, 164, 166, 167, 168, 170],
        [0.00139, 0.01601, 0.33503, 0.22869, 0.26978, 0.1491],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Tm": [NaN, [169], [1.0], [NaN]],
    "Yb": [
        NaN,
        [168, 170, 171, 172, 173, 174, 176],
        [0.00123, 0.02982, 0.1409, 0.2168, 0.16103, 0.32026, 0.12996],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Lu": [NaN, [175, 176], [0.97401, 0.02599], [NaN, NaN]],
    "Hf": [
        NaN,
        [174, 176, 177, 178, 179, 180],
        [0.0016, 0.0526, 0.186, 0.2728, 0.1362, 0.3508],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Ta": [NaN, [180, 181], [0.0001201, 0.9998799], [NaN, NaN]],
    "W": [
        NaN,
        [180, 182, 183, 184, 186],
        [0.0012, 0.265, 0.1431, 0.3064, 0.2843],
        [NaN, NaN, NaN, NaN, NaN],
    ],
    "Re": [NaN, [185, 187], [0.374, 0.626], [NaN, NaN]],
    "Os": [
        NaN,
        [184, 186, 187, 188, 189, 190, 192],
        [0.0002, 0.0159, 0.0196, 0.1324, 0.1615, 0.2626, 0.4078],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Ir": [NaN, [191, 193], [0.373, 0.627], [NaN, NaN]],
    "Pt": [
        NaN,
        [190, 192, 194, 195, 196, 198],
        [0.00012, 0.00782, 0.3286, 0.3378, 0.2521, 0.07356],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Au": [NaN, [197], [1.0], [NaN]],
    "Hg": [
        NaN,
        [196, 198, 199, 200, 201, 202, 204],
        [0.0015, 0.0997, 0.1687, 0.231, 0.1318, 0.2986, 0.0687],
        [NaN, NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Tl": [NaN, [203, 205], [0.2952, 0.7048], [NaN, NaN]],
    "Pb": [
        NaN,
        [204, 206, 207, 208],
        [0.014, 0.241, 0.221, 0.524],
        [NaN, NaN, NaN, NaN],
    ],
    "Bi": [NaN, [209], [1.0], [NaN]],
    "Po": [NaN, [209, 210], [0.0, 0.0], [NaN, NaN]],
    "At": [NaN, [210, 211], [0.0, 0.0], [NaN, NaN]],
    "Rn": [NaN, [211, 220, 222], [0.0, 0.0, 0.0], [NaN, NaN, NaN]],
    "Fr": [NaN, [223], [0.0], [NaN]],
    "Ra": [NaN, [223, 224, 226, 228], [0.0, 0.0, 0.0, 0.0], [NaN, NaN, NaN, NaN]],
    "Ac": [NaN, [227], [0.0], [NaN]],
    "Th": [NaN, [230, 232], [0.0, 1.0], [NaN, NaN]],
    "Pa": [NaN, [231], [1.0], [NaN]],
    "U": [
        NaN,
        [233, 234, 235, 236, 238],
        [0.0, 5.4e-05, 0.007204, 0.0, 0.992742],
        [NaN, NaN, NaN, NaN, NaN],
    ],
    "Np": [NaN, [236, 237], [0.0, 0.0], [NaN, NaN]],
    "Pu": [
        NaN,
        [238, 239, 240, 241, 242, 244],
        [0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Am": [NaN, [241, 243], [0.0, 0.0], [NaN, NaN]],
    "Cm": [
        NaN,
        [243, 244, 245, 246, 247, 248],
        [0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
        [NaN, NaN, NaN, NaN, NaN, NaN],
    ],
    "Bk": [NaN, [247, 249], [0.0, 0.0], [NaN, NaN]],
    "Cf": [NaN, [249, 250, 251, 252], [0.0, 0.0, 0.0, 0.0], [NaN, NaN, NaN, NaN]],
    "Es": [NaN, [252], [0.0], [NaN]],
    "Fm": [NaN, [257], [0.0], [NaN]],
    "Md": [NaN, [258, 260], [0.0, 0.0], [NaN, NaN]],
    "No": [NaN, [259], [0.0], [NaN]],
    "Lr": [NaN, [262], [0.0], [NaN]],
    "Rf": [NaN, [267], [0.0], [NaN]],
    "Db": [NaN, [268], [0.0], [NaN]],
    "Sg": [NaN, [271], [0.0], [NaN]],
    "Bh": [NaN, [272], [0.0], [NaN]],
    "Hs": [NaN, [270], [0.0], [NaN]],
    "Mt": [NaN, [276], [0.0], [NaN]],
    "Ds": [NaN, [281], [0.0], [NaN]],
    "Rg": [NaN, [280], [0.0], [NaN]],
    "Cn": [NaN, [285], [0.0], [NaN]],
    "Nh": [NaN, [284], [0.0], [NaN]],
    "Fl": [NaN, [289], [0.0], [NaN]],
    "Mc": [NaN, [288], [0.0], [NaN]],
    "Lv": [NaN, [293], [0.0], [NaN]],
    "Ts": [NaN, [292], [0.0], [NaN]],
    "Og": [NaN, [294], [0.0], [NaN]],
}

nist15_isotopes = {
    "H-1": [0.999885, NaN],
    "H-2": [0.000115, NaN],
    "H-3": [0.0, NaN],
    "He-3": [1.34e-06, NaN],
    "He-4": [0.99999866, NaN],
    "Li-6": [0.0759, NaN],
    "Li-7": [0.9241, NaN],
    "Be-9": [1.0, NaN],
    "B-10": [0.199, NaN],
    "B-11": [0.801, NaN],
    "C-12": [0.9893, NaN],
    "C-13": [0.0107, NaN],
    "C-14": [0.0, NaN],
    "N-14": [0.99636, NaN],
    "N-15": [0.00364, NaN],
    "O-16": [0.99757, NaN],
    "O-17": [0.00038, NaN],
    "O-18": [0.00205, NaN],
    "F-19": [1.0, NaN],
    "Ne-20": [0.9048, NaN],
    "Ne-21": [0.0027, NaN],
    "Ne-22": [0.0925, NaN],
    "Na-23": [1.0, NaN],
    "Mg-24": [0.7899, NaN],
    "Mg-25": [0.1, NaN],
    "Mg-26": [0.1101, NaN],
    "Al-27": [1.0, NaN],
    "Si-28": [0.92223, NaN],
    "Si-29": [0.04685, NaN],
    "Si-30": [0.03092, NaN],
    "P-31": [1.0, NaN],
    "S-32": [0.9499, NaN],
    "S-33": [0.0075, NaN],
    "S-34": [0.0425, NaN],
    "S-36": [0.0001, NaN],
    "Cl-35": [0.7576, NaN],
    "Cl-37": [0.2424, NaN],
    "Ar-36": [0.003336, NaN],
    "Ar-38": [0.000629, NaN],
    "Ar-40": [0.996035, NaN],
    "K-39": [0.932581, NaN],
    "K-40": [0.000117, NaN],
    "K-41": [0.067302, NaN],
    "Ca-40": [0.96941, NaN],
    "Ca-42": [0.00647, NaN],
    "Ca-43": [0.00135, NaN],
    "Ca-44": [0.02086, NaN],
    "Ca-46": [4e-05, NaN],
    "Ca-48": [0.00187, NaN],
    "Sc-45": [1.0, NaN],
    "Ti-46": [0.0825, NaN],
    "Ti-47": [0.0744, NaN],
    "Ti-48": [0.7372, NaN],
    "Ti-49": [0.0541, NaN],
    "Ti-50": [0.0518, NaN],
    "V-50": [0.0025, NaN],
    "V-51": [0.9975, NaN],
    "Cr-50": [0.04345, NaN],
    "Cr-52": [0.83789, NaN],
    "Cr-53": [0.09501, NaN],
    "Cr-54": [0.02365, NaN],
    "Mn-55": [1.0, NaN],
    "Fe-54": [0.05845, NaN],
    "Fe-56": [0.91754, NaN],
    "Fe-57": [0.02119, NaN],
    "Fe-58": [0.00282, NaN],
    "Co-59": [1.0, NaN],
    "Ni-58": [0.68077, NaN],
    "Ni-60": [0.26223, NaN],
    "Ni-61": [0.011399, NaN],
    "Ni-62": [0.036346, NaN],
    "Ni-64": [0.009255, NaN],
    "Cu-63": [0.6915, NaN],
    "Cu-65": [0.3085, NaN],
    "Zn-64": [0.4917, NaN],
    "Zn-66": [0.2773, NaN],
    "Zn-67": [0.0404, NaN],
    "Zn-68": [0.1845, NaN],
    "Zn-70": [0.0061, NaN],
    "Ga-69": [0.60108, NaN],
    "Ga-71": [0.39892, NaN],
    "Ge-70": [0.2057, NaN],
    "Ge-72": [0.2745, NaN],
    "Ge-73": [0.0775, NaN],
    "Ge-74": [0.365, NaN],
    "Ge-76": [0.0773, NaN],
    "As-75": [1.0, NaN],
    "Se-74": [0.0089, NaN],
    "Se-76": [0.0937, NaN],
    "Se-77": [0.0763, NaN],
    "Se-78": [0.2377, NaN],
    "Se-80": [0.4961, NaN],
    "Se-82": [0.0873, NaN],
    "Br-79": [0.5069, NaN],
    "Br-81": [0.4931, NaN],
    "Kr-78": [0.00355, NaN],
    "Kr-80": [0.02286, NaN],
    "Kr-82": [0.11593, NaN],
    "Kr-83": [0.115, NaN],
    "Kr-84": [0.56987, NaN],
    "Kr-86": [0.17279, NaN],
    "Rb-85": [0.7217, NaN],
    "Rb-87": [0.2783, NaN],
    "Sr-84": [0.0056, NaN],
    "Sr-86": [0.0986, NaN],
    "Sr-87": [0.07, NaN],
    "Sr-88": [0.8258, NaN],
    "Y-89": [1.0, NaN],
    "Zr-90": [0.5145, NaN],
    "Zr-91": [0.1122, NaN],
    "Zr-92": [0.1715, NaN],
    "Zr-94": [0.1738, NaN],
    "Zr-96": [0.028, NaN],
    "Nb-93": [1.0, NaN],
    "Mo-92": [0.1453, NaN],
    "Mo-94": [0.0915, NaN],
    "Mo-95": [0.1584, NaN],
    "Mo-96": [0.1667, NaN],
    "Mo-97": [0.096, NaN],
    "Mo-98": [0.2439, NaN],
    "Mo-100": [0.0982, NaN],
    "Tc-97": [0.0, NaN],
    "Tc-98": [0.0, NaN],
    "Tc-99": [0.0, NaN],
    "Ru-96": [0.0554, NaN],
    "Ru-98": [0.0187, NaN],
    "Ru-99": [0.1276, NaN],
    "Ru-100": [0.126, NaN],
    "Ru-101": [0.1706, NaN],
    "Ru-102": [0.3155, NaN],
    "Ru-104": [0.1862, NaN],
    "Rh-103": [1.0, NaN],
    "Pd-102": [0.0102, NaN],
    "Pd-104": [0.1114, NaN],
    "Pd-105": [0.2233, NaN],
    "Pd-106": [0.2733, NaN],
    "Pd-108": [0.2646, NaN],
    "Pd-110": [0.1172, NaN],
    "Ag-107": [0.51839, NaN],
    "Ag-109": [0.48161, NaN],
    "Cd-106": [0.0125, NaN],
    "Cd-108": [0.0089, NaN],
    "Cd-110": [0.1249, NaN],
    "Cd-111": [0.128, NaN],
    "Cd-112": [0.2413, NaN],
    "Cd-113": [0.1222, NaN],
    "Cd-114": [0.2873, NaN],
    "Cd-116": [0.0749, NaN],
    "In-113": [0.0429, NaN],
    "In-115": [0.9571, NaN],
    "Sn-112": [0.0097, NaN],
    "Sn-114": [0.0066, NaN],
    "Sn-115": [0.0034, NaN],
    "Sn-116": [0.1454, NaN],
    "Sn-117": [0.0768, NaN],
    "Sn-118": [0.2422, NaN],
    "Sn-119": [0.0859, NaN],
    "Sn-120": [0.3258, NaN],
    "Sn-122": [0.0463, NaN],
    "Sn-124": [0.0579, NaN],
    "Sb-121": [0.5721, NaN],
    "Sb-123": [0.4279, NaN],
    "Te-120": [0.0009, NaN],
    "Te-122": [0.0255, NaN],
    "Te-123": [0.0089, NaN],
    "Te-124": [0.0474, NaN],
    "Te-125": [0.0707, NaN],
    "Te-126": [0.1884, NaN],
    "Te-128": [0.3174, NaN],
    "Te-130": [0.3408, NaN],
    "I-127": [1.0, NaN],
    "Xe-124": [0.000952, NaN],
    "Xe-126": [0.00089, NaN],
    "Xe-128": [0.019102, NaN],
    "Xe-129": [0.264006, NaN],
    "Xe-130": [0.04071, NaN],
    "Xe-131": [0.212324, NaN],
    "Xe-132": [0.269086, NaN],
    "Xe-134": [0.104357, NaN],
    "Xe-136": [0.088573, NaN],
    "Cs-133": [1.0, NaN],
    "Ba-130": [0.00106, NaN],
    "Ba-132": [0.00101, NaN],
    "Ba-134": [0.02417, NaN],
    "Ba-135": [0.06592, NaN],
    "Ba-136": [0.07854, NaN],
    "Ba-137": [0.11232, NaN],
    "Ba-138": [0.71698, NaN],
    "La-138": [0.0008881, NaN],
    "La-139": [0.9991119, NaN],
    "Ce-136": [0.00185, NaN],
    "Ce-138": [0.00251, NaN],
    "Ce-140": [0.8845, NaN],
    "Ce-142": [0.11114, NaN],
    "Pr-141": [1.0, NaN],
    "Nd-142": [0.27152, NaN],
    "Nd-143": [0.12174, NaN],
    "Nd-144": [0.23798, NaN],
    "Nd-145": [0.08293, NaN],
    "Nd-146": [0.17189, NaN],
    "Nd-148": [0.05756, NaN],
    "Nd-150": [0.05638, NaN],
    "Pm-145": [0.0, NaN],
    "Pm-147": [0.0, NaN],
    "Sm-144": [0.0307, NaN],
    "Sm-147": [0.1499, NaN],
    "Sm-148": [0.1124, NaN],
    "Sm-149": [0.1382, NaN],
    "Sm-150": [0.0738, NaN],
    "Sm-152": [0.2675, NaN],
    "Sm-154": [0.2275, NaN],
    "Eu-151": [0.4781, NaN],
    "Eu-153": [0.5219, NaN],
    "Gd-152": [0.002, NaN],
    "Gd-154": [0.0218, NaN],
    "Gd-155": [0.148, NaN],
    "Gd-156": [0.2047, NaN],
    "Gd-157": [0.1565, NaN],
    "Gd-158": [0.2484, NaN],
    "Gd-160": [0.2186, NaN],
    "Tb-159": [1.0, NaN],
    "Dy-156": [0.00056, NaN],
    "Dy-158": [0.00095, NaN],
    "Dy-160": [0.02329, NaN],
    "Dy-161": [0.18889, NaN],
    "Dy-162": [0.25475, NaN],
    "Dy-163": [0.24896, NaN],
    "Dy-164": [0.2826, NaN],
    "Ho-165": [1.0, NaN],
    "Er-162": [0.00139, NaN],
    "Er-164": [0.01601, NaN],
    "Er-166": [0.33503, NaN],
    "Er-167": [0.22869, NaN],
    "Er-168": [0.26978, NaN],
    "Er-170": [0.1491, NaN],
    "Tm-169": [1.0, NaN],
    "Yb-168": [0.00123, NaN],
    "Yb-170": [0.02982, NaN],
    "Yb-171": [0.1409, NaN],
    "Yb-172": [0.2168, NaN],
    "Yb-173": [0.16103, NaN],
    "Yb-174": [0.32026, NaN],
    "Yb-176": [0.12996, NaN],
    "Lu-175": [0.97401, NaN],
    "Lu-176": [0.02599, NaN],
    "Hf-174": [0.0016, NaN],
    "Hf-176": [0.0526, NaN],
    "Hf-177": [0.186, NaN],
    "Hf-178": [0.2728, NaN],
    "Hf-179": [0.1362, NaN],
    "Hf-180": [0.3508, NaN],
    "Ta-180": [0.0001201, NaN],
    "Ta-181": [0.9998799, NaN],
    "W-180": [0.0012, NaN],
    "W-182": [0.265, NaN],
    "W-183": [0.1431, NaN],
    "W-184": [0.3064, NaN],
    "W-186": [0.2843, NaN],
    "Re-185": [0.374, NaN],
    "Re-187": [0.626, NaN],
    "Os-184": [0.0002, NaN],
    "Os-186": [0.0159, NaN],
    "Os-187": [0.0196, NaN],
    "Os-188": [0.1324, NaN],
    "Os-189": [0.1615, NaN],
    "Os-190": [0.2626, NaN],
    "Os-192": [0.4078, NaN],
    "Ir-191": [0.373, NaN],
    "Ir-193": [0.627, NaN],
    "Pt-190": [0.00012, NaN],
    "Pt-192": [0.00782, NaN],
    "Pt-194": [0.3286, NaN],
    "Pt-195": [0.3378, NaN],
    "Pt-196": [0.2521, NaN],
    "Pt-198": [0.07356, NaN],
    "Au-197": [1.0, NaN],
    "Hg-196": [0.0015, NaN],
    "Hg-198": [0.0997, NaN],
    "Hg-199": [0.1687, NaN],
    "Hg-200": [0.231, NaN],
    "Hg-201": [0.1318, NaN],
    "Hg-202": [0.2986, NaN],
    "Hg-204": [0.0687, NaN],
    "Tl-203": [0.2952, NaN],
    "Tl-205": [0.7048, NaN],
    "Pb-204": [0.014, NaN],
    "Pb-206": [0.241, NaN],
    "Pb-207": [0.221, NaN],
    "Pb-208": [0.524, NaN],
    "Bi-209": [1.0, NaN],
    "Po-209": [0.0, NaN],
    "Po-210": [0.0, NaN],
    "At-210": [0.0, NaN],
    "At-211": [0.0, NaN],
    "Rn-211": [0.0, NaN],
    "Rn-220": [0.0, NaN],
    "Rn-222": [0.0, NaN],
    "Fr-223": [0.0, NaN],
    "Ra-223": [0.0, NaN],
    "Ra-224": [0.0, NaN],
    "Ra-226": [0.0, NaN],
    "Ra-228": [0.0, NaN],
    "Ac-227": [0.0, NaN],
    "Th-230": [0.0, NaN],
    "Th-232": [1.0, NaN],
    "Pa-231": [1.0, NaN],
    "U-233": [0.0, NaN],
    "U-234": [5.4e-05, NaN],
    "U-235": [0.007204, NaN],
    "U-236": [0.0, NaN],
    "U-238": [0.992742, NaN],
    "Np-236": [0.0, NaN],
    "Np-237": [0.0, NaN],
    "Pu-238": [0.0, NaN],
    "Pu-239": [0.0, NaN],
    "Pu-240": [0.0, NaN],
    "Pu-241": [0.0, NaN],
    "Pu-242": [0.0, NaN],
    "Pu-244": [0.0, NaN],
    "Am-241": [0.0, NaN],
    "Am-243": [0.0, NaN],
    "Cm-243": [0.0, NaN],
    "Cm-244": [0.0, NaN],
    "Cm-245": [0.0, NaN],
    "Cm-246": [0.0, NaN],
    "Cm-247": [0.0, NaN],
    "Cm-248": [0.0, NaN],
    "Bk-247": [0.0, NaN],
    "Bk-249": [0.0, NaN],
    "Cf-249": [0.0, NaN],
    "Cf-250": [0.0, NaN],
    "Cf-251": [0.0, NaN],
    "Cf-252": [0.0, NaN],
    "Es-252": [0.0, NaN],
    "Fm-257": [0.0, NaN],
    "Md-258": [0.0, NaN],
    "Md-260": [0.0, NaN],
    "No-259": [0.0, NaN],
    "Lr-262": [0.0, NaN],
    "Rf-267": [0.0, NaN],
    "Db-268": [0.0, NaN],
    "Sg-271": [0.0, NaN],
    "Bh-272": [0.0, NaN],
    "Hs-270": [0.0, NaN],
    "Mt-276": [0.0, NaN],
    "Ds-281": [0.0, NaN],
    "Rg-280": [0.0, NaN],
    "Cn-285": [0.0, NaN],
    "Nh-284": [0.0, NaN],
    "Fl-289": [0.0, NaN],
    "Mc-288": [0.0, NaN],
    "Lv-293": [0.0, NaN],
    "Ts-292": [0.0, NaN],
    "Og-294": [0.0, NaN],
}
