import sys, os
import yaml
from dateutil.parser import parse
from dateutil.relativedelta import relativedelta
from datetime import datetime, timedelta

from omg.common.config import Config

# This function is used to calculate age of the objects
# We compare the time reported in the yaml definition,
# with the timestamp of the yaml file (i.e, when it was generated).
# This should give us age of the object at the time of must-gather
# obj_time time is in standard format e.g: '2020-06-04T22:10:41Z'
# file_ts is unix ts e.g: 1590912494.0 (returned by os.path.getmtime)

def age(obj_time, file_ts):

    try:
        dt1 = parse(obj_time, ignoretz=True)
        dt2 = datetime.utcfromtimestamp(file_ts)
        rd = relativedelta(dt2, dt1)
    except:
        return 'Unknown'

    if rd.days > 0:
        return str(rd.days)+'d'
    elif rd.hours > 9:
        return str(rd.hours)+'h'
    elif rd.hours > 0 and rd.hours < 10:
        return str(rd.hours)+'h'+str(rd.minutes)+'m'
    elif rd.minutes > 9:
        return str(rd.minutes) + 'm'
    elif rd.minutes > 0 and rd.minutes < 10:
        return str(rd.minutes)+'m'+str(rd.seconds)+'s'
    else:
        return str(rd.seconds)+'s'


def age2(obj_time, obj2_time):

    try:
        dt1 = parse(obj_time, ignoretz=True)
        dt2 = parse(obj2_time, ignoretz=True)
        rd = relativedelta(dt2, dt1)
    except:
        return 'Unknown'

    if rd.days > 0:
        return str(rd.days)+'d'
    elif rd.hours > 9:
        return str(rd.hours)+'h'
    elif rd.hours > 0 and rd.hours < 10:
        return str(rd.hours)+'h'+str(rd.minutes)+'m'
    elif rd.minutes > 9:
        return str(rd.minutes) + 'm'
    elif rd.minutes > 0 and rd.minutes < 10:
        return str(rd.minutes)+'m'+str(rd.seconds)+'s'
    else:
        return str(rd.seconds)+'s'

# This is a helper function to load yaml files
# Input: yaml file path (yp)
# Output: python dict of the yaml
#
# This is handled in separate function instead of simply
# calling yaml.safe_load(),
# because some yamls generated by must-gather contain garbage lines
# at the end causing the yaml.safe_load() to error out.
# so if the first loading attempt fails, we will
# try to skip lines from the end and try to load the yaml
def load_yaml_file(yp):
    with open(yp, 'r') as yf:
        yd = yf.read()
        try:
            res = yaml.safe_load(yd)
            return res
        except:
            # yaml load failed
            # try skipping lines from the bottom
            # Until we are able to load the yaml file
            # We will try until > 1 lines are left
            lines_total = yd.count('\n')
            lines_skipped = 0
            while yd.count('\n') > 1:
                # skip last line
                yd = yd[:yd.rfind('\n')]
                lines_skipped += 1
                try:
                    res = yaml.safe_load(yd)
                    print("[WARN] Skipped " +
                        str(lines_skipped) + "/" + str(lines_total) +
                        " lines from the end of " + os.path.basename(yp) +
                        " to the load the yaml file properly")
                    return res
                except:
                    pass
            # Skipping lines from the bottom didn't help. Error out
            print("[ERROR] Invalid yaml file. Parsing error in ", yp)
            sys.exit(1)
