"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 *
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    CodeCommitTrigger["NONE"] = "None";
    CodeCommitTrigger["POLL"] = "Poll";
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 */
class CodeCommitSourceAction extends action_1.Action {
    /**
     *
     */
    constructor(props) {
        var _a;
        const branch = (_a = props.branch) !== null && _a !== void 0 ? _a : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /**
     * The variables emitted by this action.
     */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     */
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline, {
                    eventRole: this.props.eventRole,
                }),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node);
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : `-${this.branch}-`;
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
//# sourceMappingURL=data:application/json;base64,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