"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jenkinsArtifactsBounds = exports.JenkinsProvider = exports.BaseJenkinsProvider = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cdk = require("@aws-cdk/core");
const custom_action_registration_1 = require("../custom-action-registration");
/**
 *
 */
class BaseJenkinsProvider extends cdk.Construct {
    /**
     *
     */
    constructor(scope, id, version) {
        super(scope, id);
        this.version = version || '1';
    }
}
exports.BaseJenkinsProvider = BaseJenkinsProvider;
/**
 * A class representing Jenkins providers.
 *
 * @see #import
 */
class JenkinsProvider extends BaseJenkinsProvider {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.buildIncluded = false;
        this.testIncluded = false;
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
        if (props.forBuild === true) {
            this._registerBuildProvider();
        }
        if (props.forTest === true) {
            this._registerTestProvider();
        }
    }
    /**
     * Import a Jenkins provider registered either outside the CDK, or in a different CDK Stack.
     *
     * @param scope the parent Construct for the new provider.
     * @param id the identifier of the new provider Construct.
     * @param attrs the properties used to identify the existing provider.
     * @returns a new Construct representing a reference to an existing Jenkins provider
     */
    static fromJenkinsProviderAttributes(scope, id, attrs) {
        return new ImportedJenkinsProvider(scope, id, attrs);
    }
    /**
     * @internal
     */
    _registerBuildProvider() {
        if (this.buildIncluded) {
            return;
        }
        this.buildIncluded = true;
        this.registerJenkinsCustomAction('JenkinsBuildProviderResource', codepipeline.ActionCategory.BUILD);
    }
    /**
     * @internal
     */
    _registerTestProvider() {
        if (this.testIncluded) {
            return;
        }
        this.testIncluded = true;
        this.registerJenkinsCustomAction('JenkinsTestProviderResource', codepipeline.ActionCategory.TEST);
    }
    registerJenkinsCustomAction(id, category) {
        new custom_action_registration_1.CustomActionRegistration(this, id, {
            category,
            artifactBounds: exports.jenkinsArtifactsBounds,
            provider: this.providerName,
            version: this.version,
            entityUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}'),
            executionUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}/{ExternalExecutionId}'),
            actionProperties: [
                {
                    name: 'ProjectName',
                    required: true,
                    key: true,
                    queryable: true,
                },
            ],
        });
    }
}
exports.JenkinsProvider = JenkinsProvider;
class ImportedJenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
    }
    _registerBuildProvider() {
        // do nothing
    }
    _registerTestProvider() {
        // do nothing
    }
}
function appendToUrl(baseUrl, path) {
    return baseUrl.endsWith('/') ? baseUrl + path : `${baseUrl}/${path}`;
}
exports.jenkinsArtifactsBounds = {
    minInputs: 0,
    maxInputs: 5,
    minOutputs: 0,
    maxOutputs: 5,
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiamVua2lucy1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImplbmtpbnMtcHJvdmlkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMERBQTBEO0FBQzFELHFDQUFxQztBQUVyQyw4RUFBeUU7Ozs7QUFzR3pFLE1BQXNCLG1CQUFvQixTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7O0lBSzdELFlBQXNCLEtBQWdCLEVBQUUsRUFBVSxFQUFFLE9BQWdCO1FBQ2xFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLElBQUksR0FBRyxDQUFDO0lBQ2hDLENBQUM7Q0FXRjtBQXBCRCxrREFvQkM7Ozs7OztBQU9ELE1BQWEsZUFBZ0IsU0FBUSxtQkFBbUI7Ozs7SUFtQnRELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7UUFDbkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBSjFCLGtCQUFhLEdBQUcsS0FBSyxDQUFDO1FBQ3RCLGlCQUFZLEdBQUcsS0FBSyxDQUFDO1FBSzNCLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFFakMsSUFBSSxLQUFLLENBQUMsUUFBUSxLQUFLLElBQUksRUFBRTtZQUMzQixJQUFJLENBQUMsc0JBQXNCLEVBQUUsQ0FBQztTQUMvQjtRQUNELElBQUksS0FBSyxDQUFDLE9BQU8sS0FBSyxJQUFJLEVBQUU7WUFDMUIsSUFBSSxDQUFDLHFCQUFxQixFQUFFLENBQUM7U0FDOUI7SUFDSCxDQUFDOzs7Ozs7Ozs7SUFyQk0sTUFBTSxDQUFDLDZCQUE2QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWdDO1FBQ3hHLE9BQU8sSUFBSSx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFxQkQ7O09BRUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3RCLE9BQU87U0FDUjtRQUNELElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDO1FBQzFCLElBQUksQ0FBQywyQkFBMkIsQ0FBQyw4QkFBOEIsRUFBRSxZQUFZLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3RHLENBQUM7SUFFRDs7T0FFRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDckIsT0FBTztTQUNSO1FBQ0QsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUM7UUFDekIsSUFBSSxDQUFDLDJCQUEyQixDQUFDLDZCQUE2QixFQUFFLFlBQVksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDcEcsQ0FBQztJQUVPLDJCQUEyQixDQUFDLEVBQVUsRUFBRSxRQUFxQztRQUNuRixJQUFJLHFEQUF3QixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDckMsUUFBUTtZQUNSLGNBQWMsRUFBRSw4QkFBc0I7WUFDdEMsUUFBUSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQzNCLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixTQUFTLEVBQUUsV0FBVyxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsMEJBQTBCLENBQUM7WUFDbEUsWUFBWSxFQUFFLFdBQVcsQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGdEQUFnRCxDQUFDO1lBQzNGLGdCQUFnQixFQUFFO2dCQUNoQjtvQkFDRSxJQUFJLEVBQUUsYUFBYTtvQkFDbkIsUUFBUSxFQUFFLElBQUk7b0JBQ2QsR0FBRyxFQUFFLElBQUk7b0JBQ1QsU0FBUyxFQUFFLElBQUk7aUJBQ2hCO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUF6RUQsMENBeUVDO0FBRUQsTUFBTSx1QkFBd0IsU0FBUSxtQkFBbUI7SUFJdkQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFnQztRQUN4RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFaEMsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztJQUNuQyxDQUFDO0lBRU0sc0JBQXNCO1FBQzNCLGFBQWE7SUFDZixDQUFDO0lBRU0scUJBQXFCO1FBQzFCLGFBQWE7SUFDZixDQUFDO0NBQ0Y7QUFFRCxTQUFTLFdBQVcsQ0FBQyxPQUFlLEVBQUUsSUFBWTtJQUNoRCxPQUFPLE9BQU8sQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUMsQ0FBQyxDQUFDLEdBQUcsT0FBTyxJQUFJLElBQUksRUFBRSxDQUFDO0FBQ3ZFLENBQUM7QUFFWSxRQUFBLHNCQUFzQixHQUFzQztJQUN2RSxTQUFTLEVBQUUsQ0FBQztJQUNaLFNBQVMsRUFBRSxDQUFDO0lBQ1osVUFBVSxFQUFFLENBQUM7SUFDYixVQUFVLEVBQUUsQ0FBQztDQUNkLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEN1c3RvbUFjdGlvblJlZ2lzdHJhdGlvbiB9IGZyb20gJy4uL2N1c3RvbS1hY3Rpb24tcmVnaXN0cmF0aW9uJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElKZW5raW5zUHJvdmlkZXIgZXh0ZW5kcyBjZGsuSUNvbnN0cnVjdCB7XG4gIHJlYWRvbmx5IHByb3ZpZGVyTmFtZTogc3RyaW5nO1xuICByZWFkb25seSBzZXJ2ZXJVcmw6IHN0cmluZztcbiAgcmVhZG9ubHkgdmVyc2lvbjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZWdpc3RlcnMgYSBKZW5raW5zIFByb3ZpZGVyIGZvciB0aGUgYnVpbGQgY2F0ZWdvcnkuXG4gICAqIFRoaXMgbWV0aG9kIHdpbGwgYmUgYXV0b21hdGljYWxseSBjYWxsZWQgd2hlbiBjcmVhdGluZ1xuICAgKiBhIHtAbGluayBKZW5raW5zQWN0aW9ufSxcbiAgICogc28geW91IHNob3VsZCBuZXZlciBuZWVkIHRvIGNhbGwgaXQgZXhwbGljaXRseS5cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBfcmVnaXN0ZXJCdWlsZFByb3ZpZGVyKCk6IHZvaWQ7XG5cbiAgLyoqXG4gICAqIFJlZ2lzdGVycyBhIEplbmtpbnMgUHJvdmlkZXIgZm9yIHRoZSB0ZXN0IGNhdGVnb3J5LlxuICAgKiBUaGlzIG1ldGhvZCB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgY2FsbGVkIHdoZW4gY3JlYXRpbmdcbiAgICogYSB7QGxpbmsgSmVua2luc1Rlc3RBY3Rpb259LFxuICAgKiBzbyB5b3Ugc2hvdWxkIG5ldmVyIG5lZWQgdG8gY2FsbCBpdCBleHBsaWNpdGx5LlxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIF9yZWdpc3RlclRlc3RQcm92aWRlcigpOiB2b2lkO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBKZW5raW5zUHJvdmlkZXJBdHRyaWJ1dGVzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcm92aWRlck5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzZXJ2ZXJVcmw6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZlcnNpb24/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSmVua2luc1Byb3ZpZGVyUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb3ZpZGVyTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlcnZlclVybDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBmb3JCdWlsZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZm9yVGVzdD86IGJvb2xlYW47XG59XG5cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBCYXNlSmVua2luc1Byb3ZpZGVyIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCBpbXBsZW1lbnRzIElKZW5raW5zUHJvdmlkZXIge1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcHJvdmlkZXJOYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBzZXJ2ZXJVcmw6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHZlcnNpb246IHN0cmluZztcblxuICBwcm90ZWN0ZWQgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgdmVyc2lvbj86IHN0cmluZykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnZlcnNpb24gPSB2ZXJzaW9uIHx8ICcxJztcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCBfcmVnaXN0ZXJCdWlsZFByb3ZpZGVyKCk6IHZvaWQ7XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IF9yZWdpc3RlclRlc3RQcm92aWRlcigpOiB2b2lkO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBKZW5raW5zUHJvdmlkZXIgZXh0ZW5kcyBCYXNlSmVua2luc1Byb3ZpZGVyIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUplbmtpbnNQcm92aWRlckF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IEplbmtpbnNQcm92aWRlckF0dHJpYnV0ZXMpOiBJSmVua2luc1Byb3ZpZGVyIHtcbiAgICByZXR1cm4gbmV3IEltcG9ydGVkSmVua2luc1Byb3ZpZGVyKHNjb3BlLCBpZCwgYXR0cnMpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IHByb3ZpZGVyTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgc2VydmVyVXJsOiBzdHJpbmc7XG4gIHByaXZhdGUgYnVpbGRJbmNsdWRlZCA9IGZhbHNlO1xuICBwcml2YXRlIHRlc3RJbmNsdWRlZCA9IGZhbHNlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBKZW5raW5zUHJvdmlkZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMudmVyc2lvbik7XG5cbiAgICB0aGlzLnByb3ZpZGVyTmFtZSA9IHByb3BzLnByb3ZpZGVyTmFtZTtcbiAgICB0aGlzLnNlcnZlclVybCA9IHByb3BzLnNlcnZlclVybDtcblxuICAgIGlmIChwcm9wcy5mb3JCdWlsZCA9PT0gdHJ1ZSkge1xuICAgICAgdGhpcy5fcmVnaXN0ZXJCdWlsZFByb3ZpZGVyKCk7XG4gICAgfVxuICAgIGlmIChwcm9wcy5mb3JUZXN0ID09PSB0cnVlKSB7XG4gICAgICB0aGlzLl9yZWdpc3RlclRlc3RQcm92aWRlcigpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfcmVnaXN0ZXJCdWlsZFByb3ZpZGVyKCk6IHZvaWQge1xuICAgIGlmICh0aGlzLmJ1aWxkSW5jbHVkZWQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgdGhpcy5idWlsZEluY2x1ZGVkID0gdHJ1ZTtcbiAgICB0aGlzLnJlZ2lzdGVySmVua2luc0N1c3RvbUFjdGlvbignSmVua2luc0J1aWxkUHJvdmlkZXJSZXNvdXJjZScsIGNvZGVwaXBlbGluZS5BY3Rpb25DYXRlZ29yeS5CVUlMRCk7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3JlZ2lzdGVyVGVzdFByb3ZpZGVyKCk6IHZvaWQge1xuICAgIGlmICh0aGlzLnRlc3RJbmNsdWRlZCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICB0aGlzLnRlc3RJbmNsdWRlZCA9IHRydWU7XG4gICAgdGhpcy5yZWdpc3RlckplbmtpbnNDdXN0b21BY3Rpb24oJ0plbmtpbnNUZXN0UHJvdmlkZXJSZXNvdXJjZScsIGNvZGVwaXBlbGluZS5BY3Rpb25DYXRlZ29yeS5URVNUKTtcbiAgfVxuXG4gIHByaXZhdGUgcmVnaXN0ZXJKZW5raW5zQ3VzdG9tQWN0aW9uKGlkOiBzdHJpbmcsIGNhdGVnb3J5OiBjb2RlcGlwZWxpbmUuQWN0aW9uQ2F0ZWdvcnkpIHtcbiAgICBuZXcgQ3VzdG9tQWN0aW9uUmVnaXN0cmF0aW9uKHRoaXMsIGlkLCB7XG4gICAgICBjYXRlZ29yeSxcbiAgICAgIGFydGlmYWN0Qm91bmRzOiBqZW5raW5zQXJ0aWZhY3RzQm91bmRzLFxuICAgICAgcHJvdmlkZXI6IHRoaXMucHJvdmlkZXJOYW1lLFxuICAgICAgdmVyc2lvbjogdGhpcy52ZXJzaW9uLFxuICAgICAgZW50aXR5VXJsOiBhcHBlbmRUb1VybCh0aGlzLnNlcnZlclVybCwgJ2pvYi97Q29uZmlnOlByb2plY3ROYW1lfScpLFxuICAgICAgZXhlY3V0aW9uVXJsOiBhcHBlbmRUb1VybCh0aGlzLnNlcnZlclVybCwgJ2pvYi97Q29uZmlnOlByb2plY3ROYW1lfS97RXh0ZXJuYWxFeGVjdXRpb25JZH0nKSxcbiAgICAgIGFjdGlvblByb3BlcnRpZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6ICdQcm9qZWN0TmFtZScsXG4gICAgICAgICAgcmVxdWlyZWQ6IHRydWUsXG4gICAgICAgICAga2V5OiB0cnVlLFxuICAgICAgICAgIHF1ZXJ5YWJsZTogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH1cbn1cblxuY2xhc3MgSW1wb3J0ZWRKZW5raW5zUHJvdmlkZXIgZXh0ZW5kcyBCYXNlSmVua2luc1Byb3ZpZGVyIHtcbiAgcHVibGljIHJlYWRvbmx5IHByb3ZpZGVyTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgc2VydmVyVXJsOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEplbmtpbnNQcm92aWRlckF0dHJpYnV0ZXMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzLnZlcnNpb24pO1xuXG4gICAgdGhpcy5wcm92aWRlck5hbWUgPSBwcm9wcy5wcm92aWRlck5hbWU7XG4gICAgdGhpcy5zZXJ2ZXJVcmwgPSBwcm9wcy5zZXJ2ZXJVcmw7XG4gIH1cblxuICBwdWJsaWMgX3JlZ2lzdGVyQnVpbGRQcm92aWRlcigpOiB2b2lkIHtcbiAgICAvLyBkbyBub3RoaW5nXG4gIH1cblxuICBwdWJsaWMgX3JlZ2lzdGVyVGVzdFByb3ZpZGVyKCk6IHZvaWQge1xuICAgIC8vIGRvIG5vdGhpbmdcbiAgfVxufVxuXG5mdW5jdGlvbiBhcHBlbmRUb1VybChiYXNlVXJsOiBzdHJpbmcsIHBhdGg6IHN0cmluZyk6IHN0cmluZyB7XG4gIHJldHVybiBiYXNlVXJsLmVuZHNXaXRoKCcvJykgPyBiYXNlVXJsICsgcGF0aCA6IGAke2Jhc2VVcmx9LyR7cGF0aH1gO1xufVxuXG5leHBvcnQgY29uc3QgamVua2luc0FydGlmYWN0c0JvdW5kczogY29kZXBpcGVsaW5lLkFjdGlvbkFydGlmYWN0Qm91bmRzID0ge1xuICBtaW5JbnB1dHM6IDAsXG4gIG1heElucHV0czogNSxcbiAgbWluT3V0cHV0czogMCxcbiAgbWF4T3V0cHV0czogNSxcbn07XG4iXX0=