"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @stability stable
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.targetAccounts = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     * @stability stable
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new targets.
     *
     * No-op if target is undefined.
     *
     * @stability stable
     */
    addTarget(target) {
        var _b, _c, _d;
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = (_b = targetProps.role) === null || _b === void 0 ? void 0 : _b.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = ((_c = targetProps.targetResource.env) === null || _c === void 0 ? void 0 : _c.account) || targetStack.account;
            const targetRegion = ((_d = targetProps.targetResource.env) === null || _d === void 0 ? void 0 : _d.region) || targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            // if the target is in a different account or region and is defined in this CDK App
            // we can generate all the needed components:
            // - forwarding rule in the source stack (target: default event bus of the receiver region)
            // - eventbus permissions policy (creating an extra stack)
            // - receiver rule in the target stack (target: the actual target)
            if (!util_1.sameEnvDimension(sourceAccount, targetAccount) || !util_1.sameEnvDimension(sourceRegion, targetRegion)) {
                // cross-account and/or cross-region event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cross-account.html
                // for cross-account or cross-region events, we cannot create new components for an imported resource
                // because we don't have the target stack
                const isImportedResource = !util_1.sameEnvDimension(targetStack.account, targetAccount) || !util_1.sameEnvDimension(targetStack.region, targetRegion); //(targetAccount !== targetStack.account) || (targetRegion !== targetStack.region);
                if (isImportedResource) {
                    throw new Error('Cannot create a cross-account or cross-region rule with an imported resource');
                }
                // for cross-account or cross-region events, we require concrete accounts
                if (!targetAccount || core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account or cross-region events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account or cross-region events');
                }
                // and the target region has to be concrete as well
                if (!targetRegion || core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account or cross-region events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account per region
                let targetAccountExists = false;
                const accountKey = Object.keys(this.targetAccounts).find(account => account === targetAccount);
                if (accountKey) {
                    targetAccountExists = this.targetAccounts[accountKey].has(targetRegion);
                }
                if (!targetAccountExists) {
                    // add the current account-region pair to tracking structure
                    const regionsSet = this.targetAccounts[targetAccount];
                    if (!regionsSet) {
                        this.targetAccounts[targetAccount] = new Set();
                    }
                    this.targetAccounts[targetAccount].add(targetRegion);
                    const eventBusArn = targetStack.formatArn({
                        service: 'events',
                        resource: 'event-bus',
                        resourceName: 'default',
                        region: targetRegion,
                        account: targetAccount,
                    });
                    this.targets.push({
                        id,
                        arn: eventBusArn,
                        // for cross-region we now require a role with PutEvents permissions
                        roleArn: roleArn !== null && roleArn !== void 0 ? roleArn : this.singletonEventRole(this, [this.putEventStatement(eventBusArn)]).roleArn,
                    });
                }
                // Grant the source account in the source region permissions to publish events to the event bus of the target account in the target region.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (EventBridge verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account or cross-region targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account or cross-region event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                // if different accounts, we need to add the permissions to the target eventbus
                if (!util_1.sameEnvDimension(sourceAccount, targetAccount)) {
                    const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                    let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                    if (!eventBusPolicyStack) {
                        eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                            env: {
                                account: targetAccount,
                                region: targetRegion,
                            },
                            stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                        });
                        new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                            action: 'events:PutEvents',
                            statementId: `Allow-account-${sourceAccount}`,
                            principal: sourceAccount,
                        });
                    }
                    // deploy the event bus permissions before the source stack
                    sourceStack.addDependency(eventBusPolicyStack);
                }
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        // Here only if the target does not have a targetResource defined.
        // In such case we don't have to generate any extra component.
        // Note that this can also be an imported resource (i.e: EventBus target)
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            deadLetterConfig: targetProps.deadLetterConfig,
            retryPolicy: targetProps.retryPolicy,
            sqsParameters: targetProps.sqsParameters,
            httpParameters: targetProps.httpParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: (inputProps === null || inputProps === void 0 ? void 0 : inputProps.inputTemplate) !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule.
     *
     * If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *     {
     *       "resources": [ "r1" ],
     *       "detail": {
     *         "hello": [ 1 ]
     *       }
     *     }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *     {
     *       "resources": [ "r2" ],
     *       "detail": {
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * The resulting event pattern will be:
     *
     *     {
     *       "resources": [ "r1", "r2" ],
     *       "detail": {
     *         "hello": [ 1 ],
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * @stability stable
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        return util_1.renderEventPattern(this.eventPattern);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
    /**
   * Obtain the Role for the EventBridge event
   *
   * If a role already exists, it will be returned. This ensures that if multiple
   * events have the same target, they will share a role.
   * @internal
   */
    singletonEventRole(scope, policyStatements) {
        const id = 'EventsRole';
        const existing = scope.node.tryFindChild(id);
        if (existing) {
            return existing;
        }
        const role = new aws_iam_1.Role(scope, id, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new aws_iam_1.ServicePrincipal('events.amazonaws.com'),
        });
        policyStatements.forEach(role.addToPolicy.bind(role));
        return role;
    }
    putEventStatement(eventBusArn) {
        return new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [eventBusArn],
        });
    }
}
exports.Rule = Rule;
_a = JSII_RTTI_SYMBOL_1;
Rule[_a] = { fqn: "@aws-cdk/aws-events.Rule", version: "1.116.0" };
//# sourceMappingURL=data:application/json;base64,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