import os
import re
import numpy as np

from objectpath import *

import pandas as pd

# IR -> Nominal_rates
# RIR -> Real_rates
# EQ -> Equity
# RE -> Real_estate
# CRED -> Credit
# FX -> FX_rate
MODEL_DIR_NAMES = {'IR': 'Nominal_rates',
                   'RIR': 'Real_rates',
                   'EQ': 'Equity',
                   'RE': 'Real_estate',
                   'CRED': 'Credit',
                   'FX': 'FX_rate'}

FILE_MARK = "file::"


class Syntax:
    def __init__(self, expression, col, condition, value):
        self.expression = expression
        self.col = col
        self.condition = condition
        self.value = value


def extract_value_from_equal(param_string):
    if "=" not in param_string:
        raise SyntaxError("Incorrect syntax in param. Unable to find equal.")

    last_equal_position = param_string.rindex("=")
    syntax = param_string[:last_equal_position].strip('"')
    value = param_string[last_equal_position+1:]
    return syntax, value


def extract_target_column(param_string):
    param_string = param_string.strip('"')
    if ("[" in param_string) and (param_string.endswith("]")):
        right_quote_position = param_string.rindex("]")
        left_quote_position = param_string.rindex("[")
        syntax = param_string[:left_quote_position].strip('"')
        value = param_string[left_quote_position+1:right_quote_position]
        return syntax, value
    else:
        raise SyntaxError("Incorrect syntax in param. Unable to find square quote at the end of syntax.")


def parse_param(input_syntax):
    syntax = None
    param_expression = ""
    param_col = ''
    param_condition = ''
    param_value = ''
    param_string = str(input_syntax).strip()

    if FILE_MARK not in input_syntax:
        param_string, param_value = extract_value_from_equal(input_syntax)
        return Syntax(param_string, param_col, param_condition, param_value)

    param_string, param_value = extract_value_from_equal(input_syntax)

    if param_string.startswith(FILE_MARK):
        param_string = param_string[len(FILE_MARK):]
        # Checks if '.where' exists in param_string
        if ".where" in param_string:
            param_expression, param_condition = param_string.split(".where")
            # param_condition = param_condition.strip("()")
        else:
            # # print("Didn't find '.where' clause in {}".format(param_string))
            param_expression = param_string

        # Gets the column in the para_expressions
        param_expression, param_col = extract_target_column(param_expression)
        param_names = re.findall(r'\[.+?\]', param_expression)

        result = "$"
        for name in param_names:
            result = "".join([result, "..*[@.name is '{}']".format(name.strip('[]'))])

        result = "".join([result, re.split(r'\[.+?\]', param_expression)[-1].strip('[]'), '.filename'])
        param_expression = result
    else:
        pass
        # # print("file:: not found in param: {}, skipping".format(param_string))

    syntax = Syntax(param_expression, param_col, param_condition, param_value)
    return syntax


def query(data, expression):
    result = []
    if data and expression:
        try:
            tree = Tree(data)
            result = list(tree.execute(expression))

        except AttributeError as err:
            # print("In Tree(data): ", err.message)
            pass
        except StopIteration:
            # print("In tree.execute(expression): StopIteration")
            pass
        except SyntaxError:
            # print("In tree.execute(expression): SyntaxError")
            pass

    elif data is None:
        # print("Data is None")
        pass
    else:
        # # print("Expression is None")
        pass

    return result


def get_input_file_path(data, expression, env_dir):
    input_file_path = None
    filename = query(data, expression)
    folder_names = re.findall(r"(?<=')\w+(?=')", expression)

    for index in range(len(folder_names)):
        if folder_names[index] in MODEL_DIR_NAMES:
            folder_names[index] = MODEL_DIR_NAMES[folder_names[index]]

    if filename:
        local_filepath = "/".join(folder_names + filename)
        file_path = os.path.join(env_dir, 'resources/Central/RN_inputs', local_filepath).replace('\\', '/')

        if not os.path.exists(file_path):
            # print("The input file can't be found at {}".format(file_path))
            pass
        else:
            input_file_path = file_path

    return input_file_path


def get_selection_from_dataframe(selection, dataframe):
    if not dataframe.empty and selection.strip():
        col = selection.strip('[]')
        if col.count(',') == 1:
            column, row = col.split(',')
        elif col.count(',') == 0:
            column, row = col, None
        else:
            # print("Couldn't find a correct cols")
            column, row = None, None

        if column:
            try:
                column = column.strip()
                if column.startswith("'") and column.endswith("'"):
                    dataframe = dataframe[[column.strip("'")]]
                elif column.count("'") == 0:
                    if column == "*":
                        pass
                    elif column.isnumeric():
                        dataframe = dataframe[[dataframe.columns[int(column)]]]
                    else:
                        # TODO par Quincy: Throw error when incorrect column/row value
                        # print("{} is not correct".format(column))
                        pass
                else:
                    # print("{} is not correct".format(column))
                    pass
            except KeyError:
                # # print("{} is not in {} columns".format(column, dataframe))
                pass
            except IndexError:
                # # print("index {} is out of bounds for {} columns".format(column, dataframe))
                pass

        if row:
            try:
                row = row.strip()
                if row.startswith("'") and row.endswith("'"):
                    dataframe = dataframe.loc[[row.strip("'")]]
                elif row.count("'") == 0:
                    if row == "*":
                        pass
                    elif row.isnumeric():
                        dataframe = dataframe.iloc[[int(row)], :]
                    else:
                        # print("{} is not correct".format(row))
                        pass
                else:
                    # print("{} is not correct".format(row))
                    pass
            except KeyError:
                # # print("{} is not in {} rows".format(row, dataframe))
                pass
            except IndexError:
                # # print("index {} is out of bounds for {} rows".format(row, dataframe))
                pass

    return dataframe


def select_from_dataframe(condition, operation, dataframe):
    lvalue, rvalue = condition.split(operation)
    if lvalue:
        lvalue = lvalue.strip()
        selected = get_selection_from_dataframe(lvalue, dataframe)
        if not selected.empty:
            if rvalue:
                values = rvalue.strip().split(',')
                for index in range(len(values)):
                    values[index] = values[index].strip()
                    if values[index]:
                        try:
                            values[index] = int(values[index])
                        except ValueError:
                            try:
                                values[index] = float(values[index])
                            except ValueError:
                                if values[index].lower() in ["true", "false"]:
                                    values[index] = values[index].lower() == "true"
                                else:
                                    pass

                try:
                    if operation == "==":
                        dataframe = dataframe[selected.T.iloc[0].isin(values)]
                    elif operation == "!=":
                        dataframe = dataframe[~selected.T.iloc[0].isin(values)]
                    elif operation == ">=":
                        dataframe = dataframe[selected.T.iloc[0] >= values[0]]
                    elif operation == ">":
                        dataframe = dataframe[selected.T.iloc[0] > values[0]]
                    elif operation == "<=":
                        dataframe = dataframe[selected.T.iloc[0] <= values[0]]
                    elif operation == "<":
                        dataframe = dataframe[selected.T.iloc[0] < values[0]]
                    else:
                        # print("Unsupported Operation!")
                        pass
                except Exception:
                    dataframe = pd.DataFrame()
                    pass

                # print(dataframe)
                return dataframe

            else:
                # print('No rvalue found in {}'.format(condition))
                pass
    else:
        # print('No lvalue found in {}'.format(condition))
        pass


def interpret_condition(condition, dataframe):
    if condition.strip() and not dataframe.empty:
        condition = condition.strip()
        if condition.count('==') == 1:
            dataframe = select_from_dataframe(condition, "==", dataframe)
        elif condition.count('!=') == 1:
            dataframe = select_from_dataframe(condition, "!=", dataframe)
        elif condition.count('>=') == 1:
            dataframe = select_from_dataframe(condition, ">=", dataframe)
        elif condition.count('>') == 1:
            dataframe = select_from_dataframe(condition, ">", dataframe)
        elif condition.count('<=') == 1:
            dataframe = select_from_dataframe(condition, "<=", dataframe)
        elif condition.count('<') == 1:
            dataframe = select_from_dataframe(condition, "<", dataframe)
        else:
            # print("{} is not a correct condition".format(condition))
            pass

    return dataframe


def apply_value_to_selection(value, selected_dict):
    operation = None

    value = value.strip()
    value = value.strip('"')
    cond1 = value.startswith("(") and value.endswith(")")
    cond2 = not value.startswith("(") and not value.endswith(")")
    if cond1 or cond2:
        value = value.strip("()")

    if value:
        if value[0] in ('+', '-', '*', '/'):
            operation, value = value[0], value.split(value[0])[1]
        if value:
            try:
                value = int(value)
            except ValueError:
                try:
                    value = float(value)
                except ValueError:
                    if value.lower() in ["true", "false"]:
                        value = value.lower() == "true"
                    else:
                        pass

        if operation:
            if operation == '+':
                for column in selected_dict.keys():
                    selected_dict[column] = {k: v + value if not isinstance(value, (str)) and not isinstance(v, str) else value for k, v in selected_dict[column].items()}
            elif operation == '-':
                for column in selected_dict.keys():
                    selected_dict[column] = {k: v - value if not isinstance(value, (str)) and not isinstance(v, str) else -value for k, v in selected_dict[column].items()}
            elif operation == '*':
                for column in selected_dict.keys():
                    selected_dict[column] = {k: v * value if not isinstance(value, (str)) and not isinstance(v, str) else 0 for k, v in selected_dict[column].items()}
            elif operation == '/':
                for column in selected_dict.keys():
                    selected_dict[column] = {k: v / value if not isinstance(value, (str)) and not isinstance(v, str) else 0 for k, v in selected_dict[column].items()}
            else:
                # print("ERROR")
                pass
        else:
            for column in selected_dict.keys():
                selected_dict[column] = {k: value for k, v in selected_dict[column].items()}

    return selected_dict


def apply_syntax_to_file(input_path, syntax, settings_json):
    # read path to dataframe
    # apply syntax
    if input_path and syntax and settings_json:
        seps = settings_json.get('gen_param').get('input_format')
        if seps:
            dec_sep = seps['dec_sep']
            col_sep = seps['col_sep']

            if os.path.exists(input_path):
                input_df = pd.read_csv(input_path, sep=col_sep)

                selected_df = None
                if syntax.condition:
                    # # print(syntax.condition)
                    condition = syntax.condition.strip('()')
                    or_conditions = condition.split('||')
                    df_or_list = []
                    for or_cond in or_conditions:
                        if or_cond.strip():
                            and_conditions = or_cond.split('&&')
                            df_and_list = []
                            for and_cond in and_conditions:
                                selected_df = interpret_condition(and_cond, input_df)
                                df_and_list.append(selected_df)

                            # TODO by Quincy: CHANGE merge to using conditions and selection with pandas
                            df_merge = df_and_list[0]
                            for df in df_and_list[1:]:
                                df_merge = pd.merge(df_merge, df, how='inner')
                            df_or_list.append(df_merge)

                    # print("Condition: ", syntax.condition, end=", ")
                    df_concat = pd.concat(df_or_list)

                if syntax.col:
                    # print(syntax.col)
                    if syntax.condition:
                        selected_df = get_selection_from_dataframe(syntax.col, df_concat)
                    else:
                        selected_df = get_selection_from_dataframe(syntax.col, input_df)

                    # print("Column: ", syntax.col, end=" ")

                    # {"Nom_column": {"Num de ligne": "valeur associé"}}
                    selected_dict = selected_df.to_dict()
                    if syntax.value:
                        applied_dict = apply_value_to_selection(syntax.value, selected_dict)

                    # print("Value: ", syntax.value)
                    # # print(applied_dict)
                    # input_df = input_df.replace(applied_dict)

                    for column, indexes in applied_dict.items():
                        for index in indexes:
                            input_df.loc[index, column] = indexes[index]

                    # print(input_df)
                    os.remove(input_path)
                    input_df.to_csv(input_path, sep=col_sep, index=False)

                    return True

                else:
                    # print("No col found for selection")
                    pass
            else:
                # print("Couldn't find the input file at ", input_path)
                pass
        else:
            # print("Couldn't find the seps from settings.json")
            pass
    else:
        # print("Error in apply_syntax_to_file arguments:")
        # if not input_path: print("input_path is None")
        # if not syntax: print("syntax is None")
        # if not settings_json: print("settings_json is None")
        pass

    return False

