"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
const util_1 = require("./util");
/**
 * Defines a CloudWatch Event Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: core_1.Lazy.stringValue({ produce: () => this.scheduleExpression }),
            eventPattern: core_1.Lazy.anyValue({ produce: () => this.renderEventPattern() }),
            targets: core_1.Lazy.anyValue({ produce: () => this.renderTargets() }),
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.addEventPattern(props.eventPattern);
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    static fromEventRuleArn(scope, id, eventRuleArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target) {
        if (!target) {
            return;
        }
        const targetProps = target.bind(this);
        const id = sanitizeId(targetProps.id);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        // check if a target with this ID already exists
        if (this.targets.find(t => t.id === id)) {
            throw new Error('Duplicate event rule target with ID: ' + id);
        }
        const roleArn = targetProps.role ? targetProps.role.roleArn : undefined;
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: inputProps && inputProps.inputTemplate !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return [`Either 'eventPattern' or 'schedule' must be defined`];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
    renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
}
exports.Rule = Rule;
/**
 * Sanitize whatever is returned to make a valid ID
 *
 * Result must match regex [\.\-_A-Za-z0-9]+
 */
function sanitizeId(id) {
    const _id = id.replace(/[^\.\-_A-Za-z0-9]/g, '-');
    // cut to 64 chars to respect AWS::Events::Rule Target Id field specification
    return _id.substring(Math.max(_id.length - 64, 0), _id.length);
}
//# sourceMappingURL=data:application/json;base64,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