"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    bind() {
        return {
            id: 'T1', arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
        };
    }
}
module.exports = {
    'default rule'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "ScheduleExpression": "rate(10 minutes)",
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'rule with physical name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'MyRule', {
            ruleName: 'PhysicalName',
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(10)),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Name: 'PhysicalName'
        }));
        test.done();
    },
    'eventPattern is rendered properly'(test) {
        const stack = new cdk.Stack();
        new rule_1.Rule(stack, 'MyRule', {
            eventPattern: {
                account: ['account1', 'account2'],
                detail: {
                    foo: [1, 2],
                },
                detailType: ['detailType1'],
                id: ['id1', 'id2'],
                region: ['region1', 'region2', 'region3'],
                resources: ['r1'],
                source: ['src1', 'src2'],
                time: ['t1'],
                version: ['0']
            }
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "EventPattern": {
                            account: ['account1', 'account2'],
                            detail: { foo: [1, 2] },
                            'detail-type': ['detailType1'],
                            id: ['id1', 'id2'],
                            region: ['region1', 'region2', 'region3'],
                            resources: ['r1'],
                            source: ['src1', 'src2'],
                            time: ['t1'],
                            version: ['0']
                        },
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'fails synthesis if neither eventPattern nor scheudleExpression are specified'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        new rule_1.Rule(stack, 'Rule');
        test.throws(() => app.synth(), /Either 'eventPattern' or 'schedule' must be defined/);
        test.done();
    },
    'addEventPattern can be used to add filters'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'MyRule');
        rule.addEventPattern({
            account: ['12345'],
            detail: {
                foo: ['hello']
            }
        });
        rule.addEventPattern({
            source: ['aws.source'],
            detail: {
                foo: ['bar'],
                goo: {
                    hello: ['world']
                }
            }
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyRuleA44AB831": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "EventPattern": {
                            "account": [
                                "12345"
                            ],
                            "detail": {
                                "foo": [
                                    "hello",
                                    "bar"
                                ],
                                "goo": {
                                    "hello": [
                                        "world"
                                    ]
                                }
                            },
                            "source": [
                                "aws.source"
                            ]
                        },
                        "State": "ENABLED"
                    }
                }
            }
        });
        test.done();
    },
    'targets can be added via props or addTarget with input transformer'(test) {
        const stack = new cdk.Stack();
        const t1 = {
            bind: () => ({
                id: 'T1',
                arn: 'ARN1',
                kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
            })
        };
        const t2 = {
            bind: () => ({
                id: 'T2',
                arn: 'ARN2',
                input: lib_1.RuleTargetInput.fromText(`This is ${lib_1.EventField.fromPath('$.detail.bla')}`),
            })
        };
        const rule = new rule_1.Rule(stack, 'EventRule', {
            targets: [t1],
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(5)),
        });
        rule.addTarget(t2);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "EventRule5A491D2C": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "ScheduleExpression": "rate(5 minutes)",
                        "State": "ENABLED",
                        "Targets": [
                            {
                                "Arn": "ARN1",
                                "Id": "T1",
                                "KinesisParameters": {
                                    "PartitionKeyPath": "partitionKeyPath"
                                }
                            },
                            {
                                "Arn": "ARN2",
                                "Id": "T2",
                                "InputTransformer": {
                                    "InputPathsMap": {
                                        "detail-bla": "$.detail.bla"
                                    },
                                    "InputTemplate": "\"This is <detail-bla>\""
                                },
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'input template can contain tokens'(test) {
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: 'T2', arn: 'ARN2', input: lib_1.RuleTargetInput.fromText('Hello, "world"')
            })
        });
        // tokens are used here (FnConcat), but this is a text template so we
        // expect it to be wrapped in double quotes automatically for us.
        rule.addTarget({
            bind: () => ({
                id: 'T1', arn: 'ARN1', kinesisParameters: { partitionKeyPath: 'partitionKeyPath' },
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join('', ['a', 'b']).toString()),
            })
        });
        // jsonTemplate can be used to format JSON documents with replacements
        rule.addTarget({
            bind: () => ({
                id: 'T3', arn: 'ARN3',
                input: lib_1.RuleTargetInput.fromObject({ foo: lib_1.EventField.fromPath('$.detail.bar') }),
            })
        });
        // tokens can also used for JSON templates.
        rule.addTarget({
            bind: () => ({
                id: 'T4', arn: 'ARN4',
                input: lib_1.RuleTargetInput.fromText(cdk.Fn.join(' ', ['hello', '"world"']).toString()),
            })
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "EventRule5A491D2C": {
                    "Type": "AWS::Events::Rule",
                    "Properties": {
                        "State": "ENABLED",
                        "ScheduleExpression": "rate(1 minute)",
                        "Targets": [
                            {
                                "Arn": "ARN2",
                                "Id": "T2",
                                "Input": '"Hello, \\"world\\""',
                            },
                            {
                                "Arn": "ARN1",
                                "Id": "T1",
                                "Input": "\"ab\"",
                                "KinesisParameters": {
                                    "PartitionKeyPath": "partitionKeyPath"
                                }
                            },
                            {
                                "Arn": "ARN3",
                                "Id": "T3",
                                "InputTransformer": {
                                    "InputPathsMap": {
                                        "detail-bar": "$.detail.bar"
                                    },
                                    "InputTemplate": "{\"foo\":<detail-bar>}"
                                }
                            },
                            {
                                "Arn": "ARN4",
                                "Id": "T4",
                                "Input": '"hello \\"world\\""'
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'target can declare role which will be used'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'EventRule', {
            schedule: lib_1.Schedule.rate(cdk.Duration.minutes(1)),
        });
        const role = new iam.Role(stack, 'SomeRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('nobody')
        });
        // a plain string should just be stringified (i.e. double quotes added and escaped)
        rule.addTarget({
            bind: () => ({
                id: 'T2',
                arn: 'ARN2',
                role,
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            "Targets": [
                {
                    "Arn": "ARN2",
                    "Id": "T2",
                    "RoleArn": { "Fn::GetAtt": ["SomeRole6DDC54DD", "Arn"] }
                }
            ]
        }));
        test.done();
    },
    'asEventRuleTarget can use the ruleArn and a uniqueId of the rule'(test) {
        const stack = new cdk.Stack();
        let receivedRuleArn = 'FAIL';
        let receivedRuleId = 'FAIL';
        const t1 = {
            bind: (eventRule) => {
                receivedRuleArn = eventRule.ruleArn;
                receivedRuleId = eventRule.node.uniqueId;
                return {
                    id: 'T1',
                    arn: 'ARN1',
                    kinesisParameters: { partitionKeyPath: 'partitionKeyPath' }
                };
            }
        };
        const rule = new rule_1.Rule(stack, 'EventRule');
        rule.addTarget(t1);
        test.deepEqual(stack.resolve(receivedRuleArn), stack.resolve(rule.ruleArn));
        test.deepEqual(receivedRuleId, rule.node.uniqueId);
        test.done();
    },
    'fromEventRuleArn'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const importedRule = rule_1.Rule.fromEventRuleArn(stack, 'ImportedRule', 'arn:of:rule');
        // THEN
        test.deepEqual(importedRule.ruleArn, 'arn:of:rule');
        test.done();
    },
    'rule can be disabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "State": "DISABLED"
        }));
        test.done();
    },
    'fails if multiple targets with the same id are added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new rule_1.Rule(stack, 'Rule', {
            schedule: lib_1.Schedule.expression('foom'),
            enabled: false
        });
        rule.addTarget(new SomeTarget());
        // THEN
        test.throws(() => rule.addTarget(new SomeTarget()), /Duplicate event rule target with ID/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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