"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const route53 = require("../lib");
module.exports = {
    'with default ttl'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.RecordSet(stack, 'Basic', {
            zone,
            recordName: 'www',
            recordType: route53.RecordType.CNAME,
            target: route53.RecordTarget.fromValues('zzz')
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "CNAME",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                "zzz"
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'with custom ttl'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.RecordSet(stack, 'Basic', {
            zone,
            recordName: 'aa',
            recordType: route53.RecordType.CNAME,
            target: route53.RecordTarget.fromValues('bbb'),
            ttl: core_1.Duration.seconds(6077)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "aa.myzone.",
            Type: "CNAME",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                "bbb"
            ],
            TTL: "6077"
        }));
        test.done();
    },
    'defaults to zone root'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.RecordSet(stack, 'Basic', {
            zone,
            recordType: route53.RecordType.A,
            target: route53.RecordTarget.fromValues('1.2.3.4'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "myzone.",
            Type: "A",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                "1.2.3.4"
            ],
        }));
        test.done();
    },
    'A record with ip addresses'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.ARecord(stack, 'A', {
            zone,
            recordName: 'www',
            target: route53.RecordTarget.fromIpAddresses('1.2.3.4', '5.6.7.8'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "A",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                "1.2.3.4",
                "5.6.7.8"
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'A record with alias'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        const target = {
            bind: () => {
                return {
                    hostedZoneId: 'Z2P70J7EXAMPLE',
                    dnsName: 'foo.example.com'
                };
            }
        };
        // WHEN
        new route53.ARecord(zone, 'Alias', {
            zone,
            recordName: '_foo',
            target: route53.RecordTarget.fromAlias(target)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: '_foo.myzone.',
            HostedZoneId: {
                Ref: 'HostedZoneDB99F866'
            },
            Type: 'A',
            AliasTarget: {
                HostedZoneId: 'Z2P70J7EXAMPLE',
                DNSName: 'foo.example.com',
            }
        }));
        test.done();
    },
    'AAAA record with ip addresses'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.AaaaRecord(stack, 'AAAA', {
            zone,
            recordName: 'www',
            target: route53.RecordTarget.fromIpAddresses('2001:0db8:85a3:0000:0000:8a2e:0370:7334'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "AAAA",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                "2001:0db8:85a3:0000:0000:8a2e:0370:7334"
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'AAAA record with alias on zone root'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        const target = {
            bind: () => {
                return {
                    hostedZoneId: 'Z2P70J7EXAMPLE',
                    dnsName: 'foo.example.com'
                };
            }
        };
        // WHEN
        new route53.AaaaRecord(zone, 'Alias', {
            zone,
            target: route53.RecordTarget.fromAlias(target)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: 'myzone.',
            HostedZoneId: {
                Ref: 'HostedZoneDB99F866'
            },
            Type: 'AAAA',
            AliasTarget: {
                HostedZoneId: 'Z2P70J7EXAMPLE',
                DNSName: 'foo.example.com',
            }
        }));
        test.done();
    },
    'CNAME record'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.CnameRecord(stack, 'CNAME', {
            zone,
            recordName: 'www',
            domainName: 'hello',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "CNAME",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                "hello"
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'TXT record'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.TxtRecord(stack, 'TXT', {
            zone,
            recordName: 'www',
            values: ['should be enclosed with double quotes']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "TXT",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                '"should be enclosed with double quotes"'
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'SRV record'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.SrvRecord(stack, 'SRV', {
            zone,
            recordName: 'www',
            values: [{
                    hostName: 'aws.com',
                    port: 8080,
                    priority: 10,
                    weight: 5
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "SRV",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                '10 5 8080 aws.com'
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'CAA record'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.CaaRecord(stack, 'CAA', {
            zone,
            recordName: 'www',
            values: [{
                    flag: 0,
                    tag: route53.CaaTag.ISSUE,
                    value: 'ssl.com'
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "CAA",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                '0 issue "ssl.com"'
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'CAA Amazon record'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.CaaAmazonRecord(stack, 'CAAAmazon', {
            zone,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "myzone.",
            Type: "CAA",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                '0 issue "amazon.com"'
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'CAA Amazon record with record name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.CaaAmazonRecord(stack, 'CAAAmazon', {
            zone,
            recordName: 'www',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "www.myzone.",
            Type: "CAA",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                '0 issue "amazon.com"'
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'MX record'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.MxRecord(stack, 'MX', {
            zone,
            recordName: 'mail',
            values: [{
                    hostName: 'workmail.aws',
                    priority: 10
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "mail.myzone.",
            Type: "MX",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                '10 workmail.aws'
            ],
            TTL: "1800"
        }));
        test.done();
    },
    'Zone delegation record'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const zone = new route53.HostedZone(stack, 'HostedZone', {
            zoneName: 'myzone'
        });
        // WHEN
        new route53.ZoneDelegationRecord(stack, 'NS', {
            zone,
            recordName: 'foo',
            nameServers: ['ns-1777.awsdns-30.co.uk']
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Name: "foo.myzone.",
            Type: "NS",
            HostedZoneId: {
                Ref: "HostedZoneDB99F866"
            },
            ResourceRecords: [
                "ns-1777.awsdns-30.co.uk."
            ],
            TTL: "172800"
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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