"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
class TestApp {
    constructor() {
        const account = '123456789012';
        const region = 'bermuda-triangle';
        const context = {
            [`availability-zones:${account}:${region}`]: `${region}-1a`
        };
        this.app = new cdk.App({ context });
        this.stack = new cdk.Stack(this.app, 'MyStack', { env: { account, region } });
    }
}
module.exports = {
    'default properties': {
        'public hosted zone'(test) {
            const app = new TestApp();
            new lib_1.PublicHostedZone(app.stack, 'HostedZone', { zoneName: 'test.public' });
            assert_1.expect(app.stack).to(assert_1.exactlyMatchTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.public."
                        }
                    }
                }
            }));
            test.done();
        },
        'private hosted zone'(test) {
            const app = new TestApp();
            const vpcNetwork = new ec2.Vpc(app.stack, 'VPC');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetwork });
            assert_1.expect(app.stack).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.private.",
                            VPCs: [{
                                    VPCId: { Ref: 'VPCB9E5F0B4' },
                                    VPCRegion: 'bermuda-triangle'
                                }]
                        }
                    }
                }
            }));
            test.done();
        },
        'when specifying multiple VPCs'(test) {
            const app = new TestApp();
            const vpcNetworkA = new ec2.Vpc(app.stack, 'VPC1');
            const vpcNetworkB = new ec2.Vpc(app.stack, 'VPC2');
            new lib_1.PrivateHostedZone(app.stack, 'HostedZone', { zoneName: 'test.private', vpc: vpcNetworkA })
                .addVpc(vpcNetworkB);
            assert_1.expect(app.stack).to(assert_1.beASupersetOfTemplate({
                Resources: {
                    HostedZoneDB99F866: {
                        Type: "AWS::Route53::HostedZone",
                        Properties: {
                            Name: "test.private.",
                            VPCs: [{
                                    VPCId: { Ref: 'VPC17DE2CF87' },
                                    VPCRegion: 'bermuda-triangle'
                                },
                                {
                                    VPCId: { Ref: 'VPC2C1F0E711' },
                                    VPCRegion: 'bermuda-triangle'
                                }]
                        }
                    }
                }
            }));
            test.done();
        }
    },
    'exporting and importing works'(test) {
        const stack2 = new cdk.Stack();
        const importedZone = lib_1.HostedZone.fromHostedZoneAttributes(stack2, 'Imported', {
            hostedZoneId: 'hosted-zone-id',
            zoneName: 'cdk.local'
        });
        new lib_1.TxtRecord(importedZone, 'Record', {
            zone: importedZone,
            recordName: 'lookHere',
            values: ['SeeThere']
        });
        assert_1.expect(stack2).to(assert_1.haveResource("AWS::Route53::RecordSet", {
            HostedZoneId: "hosted-zone-id",
            Name: "lookHere.cdk.local.",
            ResourceRecords: ["\"SeeThere\""],
            Type: "TXT"
        }));
        test.done();
    },
    'adds period to name if not provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            Name: 'zonename.'
        }));
        test.done();
    },
    'fails if zone name ends with a trailing dot'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => new lib_1.HostedZone(stack, 'MyHostedZone', { zoneName: 'zonename.' }), /zone name must not end with a trailing dot/);
        test.done();
    },
    'a hosted zone can be assiciated with a VPC either upon creation or using "addVpc"'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc1 = new ec2.Vpc(stack, 'VPC1');
        const vpc2 = new ec2.Vpc(stack, 'VPC2');
        const vpc3 = new ec2.Vpc(stack, 'VPC3');
        // WHEN
        const zone = new lib_1.HostedZone(stack, 'MyHostedZone', {
            zoneName: 'zonename',
            vpcs: [vpc1, vpc2]
        });
        zone.addVpc(vpc3);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::HostedZone', {
            VPCs: [
                {
                    VPCId: {
                        Ref: "VPC17DE2CF87"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                },
                {
                    VPCId: {
                        Ref: "VPC2C1F0E711"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                },
                {
                    VPCId: {
                        Ref: "VPC3CB5FCDA8"
                    },
                    VPCRegion: {
                        Ref: "AWS::Region"
                    }
                }
            ]
        }));
        test.done();
    },
    'public zone cannot be associated with a vpc (runtime error)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'MyHostedZone', { zoneName: 'zonename' });
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => zone.addVpc(vpc), /Cannot associate public hosted zones with a VPC/);
        test.done();
    },
    'setting up zone delegation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const zone = new lib_1.PublicHostedZone(stack, 'TopZone', { zoneName: 'top.test' });
        const delegate = new lib_1.PublicHostedZone(stack, 'SubZone', { zoneName: 'sub.top.test' });
        // WHEN
        zone.addDelegation(delegate, { ttl: cdk.Duration.seconds(1337) });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Type: 'NS',
            Name: 'sub.top.test.',
            HostedZoneId: stack.resolve(zone.hostedZoneId),
            ResourceRecords: stack.resolve(delegate.hostedZoneNameServers),
            TTL: '1337',
        }));
        test.done();
    },
    'public hosted zone wiht caaAmazon set to true'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.PublicHostedZone(stack, 'MyHostedZone', {
            zoneName: 'protected.com',
            caaAmazon: true
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
            Type: 'CAA',
            Name: 'protected.com.',
            ResourceRecords: [
                '0 issue "amazon.com"'
            ]
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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