"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2022.11.1"
__checksum__ = "300ee3f4b5e6a3609ae1e2f93cbb2693a5951caa89bfda6009c73ccd0c643710"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'boo', b'channel', b'chrome', b'dad', b'day', b'dev', b'eat', b'esq', b'fly', b'foo', b'gle', b'gmail', b'google', b'hangout', b'hotmail', b'ing', b'insurance', b'meet', b'meme', b'mov', b'new', b'nexus', b'page', b'phd', b'play', b'prof', b'rsvp', b'search', b'youtube', b'zip'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 10), (21, 9), (30, 61), (91, 26), (117, 12), None, (129, 19), (148, 22), (170, 7), (177, 20), (197, 18), None, (215, 29), (244, 45), (289, 7), (296, 9), (305, 36), (341, 16), (357, 10), (367, 28), None, (395, 62), (457, 8), (465, 18), (483, 19), (502, 13), (515, 14), (529, 14), None, None, (543, 40), (583, 20), (603, 35), (638, 14), (652, 32), (684, 17), (701, 9), (710, 25), (735, 25), (760, 8), (768, 13), (781, 10), None, (791, 17), (808, 6), (814, 26), (840, 10), (850, 5), (855, 19), (874, 14), (888, 11), (899, 12), (911, 27), None, (938, 24), (962, 11), (973, 7), (980, 29), (1009, 18), (1027, 27), (1054, 46), (1100, 25), (1125, 16), (1141, 18), (1159, 5), (1164, 22), (1186, 18), None, (1204, 32), (1236, 15), (1251, 8), (1259, 11), None, (1270, 5), (1275, 16), (1291, 14), (1305, 18), None, (1323, 20), (1343, 26), (1369, 48), (1417, 19), (1436, 20), (1456, 59), (1515, 14), (1529, 14), (1543, 20), None, (1563, 10), (1573, 13), (1586, 20), (1606, 29), None, (1635, 13), (1648, 19), (1667, 11), (1678, 4), (1682, 22), (1704, 10), (1714, 13), (1727, 14), (1741, 28), (1769, 11), (1780, 21), (1801, 12), (1813, 32), None, (1845, 10), (1855, 14), (1869, 26), (1895, 45), (1940, 15), None, (1955, 11), (1966, 23), (1989, 21), (2010, 26), (2036, 6), (2042, 6), (2048, 7), (2055, 5), (2060, 30), (2090, 23), (2113, 35), (2148, 17), (2165, 15), (2180, 29), (2209, 12), (2221, 70), (2291, 55), (2346, 12), (2358, 23), (2381, 16), (2397, 38), (2435, 6), (2441, 20), (2461, 44), (2505, 6), (2511, 41), (2552, 22), (2574, 23), (2597, 36), (2633, 20), (2653, 8), (2661, 15), (2676, 12), (2688, 19), (2707, 25), (2732, 15), None, (2747, 35), (2782, 21), (2803, 17), (2820, 19), (2839, 26), (2865, 5), (2870, 37), (2907, 39), (2946, 16), (2962, 10), (2972, 17), (2989, 23), (3012, 14), (3026, 17), (3043, 8), (3051, 14), (3065, 7), (3072, 29), (3101, 6), (3107, 18), (3125, 32), (3157, 20), (3177, 17), (3194, 24), (3218, 12), (3230, 40), (3270, 40), (3310, 12), (3322, 48), (3370, 32), (3402, 17), None, (3419, 8), (3427, 25), (3452, 25), (3477, 6), (3483, 23), None, (3506, 36), (3542, 33), (3575, 14), (3589, 16), (3605, 27), None, (3632, 30), (3662, 41), (3703, 50), (3753, 15), (3768, 20), (3788, 25), (3813, 21), (3834, 32), (3866, 24), (3890, 20), (3910, 17), (3927, 60), (3987, 6), (3993, 9), (4002, 12), (4014, 12), (4026, 11), (4037, 10), (4047, 48), (4095, 42), None, (4137, 36), (4173, 28), None, (4201, 8), (4209, 8), (4217, 7), None, (4224, 25), (4249, 17), None, (4266, 21), (4287, 35), (4322, 21), (4343, 10), (4353, 41), (4394, 20), (4414, 31), (4445, 23), (4468, 19), (4487, 12), (4499, 5), (4504, 30), (4534, 29), (4563, 14), (4577, 14), (4591, 47), (4638, 52), None, None, (4690, 51), (4741, 42), None, (4783, 18), None, (4801, 15), (4816, 8), (4824, 21), (4845, 6), (4851, 16), (4867, 22)], [(4889, 9958), (14847, 10226), (25073, 10458), (35531, 9587), (45118, 9756), (54874, 9554), (64428, 10646), (75074, 9449), (84523, 10473), (94996, 9792), (104788, 10716), (115504, 9825), (125329, 10436), (135765, 11431), (147196, 10186), (157382, 10694), (168076, 11007), (179083, 10186), (189269, 10181), (199450, 9482), (208932, 10501), (219433, 10124), (229557, 10350), (239907, 10336), (250243, 10646), (260889, 10076), (270965, 10387), (281352, 10417), (291769, 9693), (301462, 10106), (311568, 10754), (322322, 10272), (332594, 10102), (342696, 10078), (352774, 9549), (362323, 10355), (372678, 10176), (382854, 10909), (393763, 10779), (404542, 10784), (415326, 11024), (426350, 9760), (436110, 9993), (446103, 10214), (456317, 10314), (466631, 10367), (476998, 10083), (487081, 11292), (498373, 10154), (508527, 9298), (517825, 10109), (527934, 10577), (538511, 10545), (549056, 10176), (559232, 10603), (569835, 10242), (580077, 10503), (590580, 10284), (600864, 10369), (611233, 8789), (620022, 10225), (630247, 10139), (640386, 10096), (650482, 10507), (660989, 10531), (671520, 10650), (682170, 10167), (692337, 10819), (703156, 10686), (713842, 10451), (724293, 10181), (734474, 9868), (744342, 9231), (753573, 10594), (764167, 10242), (774409, 10739), (785148, 9558), (794706, 11112), (805818, 10054), (815872, 9890), (825762, 10569), (836331, 9137), (845468, 9951), (855419, 10225), (865644, 9828), (875472, 10511), (885983, 11069), (897052, 10087), (907139, 10551), (917690, 10204), (927894, 11212), (939106, 9526), (948632, 10117), (958749, 9925), (968674, 10361), (979035, 10688), (989723, 10326), (1000049, 9906), (1009955, 10408), (1020363, 9809), (1030172, 9966), (1040138, 10249), (1050387, 10107), (1060494, 10002), (1070496, 10047), (1080543, 10346), (1090889, 10854), (1101743, 10582), (1112325, 11220), (1123545, 10431), (1133976, 10780), (1144756, 10429), (1155185, 9953), (1165138, 10212), (1175350, 10520), (1185870, 10294), (1196164, 10384), (1206548, 10238), (1216786, 9937), (1226723, 11215), (1237938, 11024), (1248962, 10379), (1259341, 10289), (1269630, 10674), (1280304, 11341), (1291645, 10238), (1301883, 9852), (1311735, 11154), (1322889, 10256), (1333145, 11820), (1344965, 11092), (1356057, 10033), (1366090, 10294), (1376384, 9801), (1386185, 10062), (1396247, 10401), (1406648, 9768), (1416416, 10786), (1427202, 9872), (1437074, 9914), (1446988, 10792), (1457780, 10734), (1468514, 9730), (1478244, 9990), (1488234, 10876), (1499110, 9559), (1508669, 10284), (1518953, 10052), (1529005, 9777), (1538782, 10682), (1549464, 10443), (1559907, 10609), (1570516, 10463), (1580979, 9634), (1590613, 10287), (1600900, 10382), (1611282, 10129), (1621411, 10506), (1631917, 9815), (1641732, 9507), (1651239, 9400), (1660639, 10362), (1671001, 10914), (1681915, 9617), (1691532, 10182), (1701714, 11291), (1713005, 10452), (1723457, 9900), (1733357, 10945), (1744302, 10093), (1754395, 9717), (1764112, 10050), (1774162, 11698), (1785860, 9570), (1795430, 9566), (1804996, 10834), (1815830, 10107), (1825937, 10703), (1836640, 10029), (1846669, 9716), (1856385, 12561), (1868946, 10308), (1879254, 10257), (1889511, 10205), (1899716, 10975), (1910691, 11223), (1921914, 9547), (1931461, 10765), (1942226, 10092), (1952318, 10064), (1962382, 11080), (1973462, 9677), (1983139, 10395), (1993534, 10015), (2003549, 10240), (2013789, 10234), (2024023, 10127), (2034150, 9563), (2043713, 10088), (2053801, 9771), (2063572, 10633), (2074205, 10248), (2084453, 11095), (2095548, 10102), (2105650, 10873), (2116523, 10952), (2127475, 9432), (2136907, 10552), (2147459, 10526), (2157985, 10426), (2168411, 10563), (2178974, 10586), (2189560, 10367), (2199927, 10125), (2210052, 10641), (2220693, 10175), (2230868, 9734), (2240602, 10247), (2250849, 10003), (2260852, 11054), (2271906, 9960), (2281866, 9556), (2291422, 11031), (2302453, 10351), (2312804, 10474), (2323278, 10082), (2333360, 10233), (2343593, 9607), (2353200, 10669), (2363869, 10386), (2374255, 11114), (2385369, 10198), (2395567, 9781), (2405348, 10649), (2415997, 10203), (2426200, 11229), (2437429, 9975), (2447404, 9599), (2457003, 9276), (2466279, 10812), (2477091, 10574), (2487665, 10735), (2498400, 10302), (2508702, 10386), (2519088, 9871), (2528959, 10882), (2539841, 10580), (2550421, 9392), (2559813, 10111), (2569924, 9897), (2579821, 10290), (2590111, 10716), (2600827, 10844), (2611671, 9977), (2621648, 10174), (2631822, 10371)], [(2642193, 1123), (2643316, 888), (2644204, 919), (2645123, 1148), (2646271, 857), (2647128, 1057), (2648185, 816), (2649001, 1216), (2650217, 861), (2651078, 994), (2652072, 698), (2652770, 788), (2653558, 1066), (2654624, 1143), (2655767, 1158), (2656925, 1235), (2658160, 1492), (2659652, 889), (2660541, 1086), (2661627, 1024), (2662651, 1158), (2663809, 963), (2664772, 1201), (2665973, 985), (2666958, 1079), (2668037, 920), (2668957, 1336), (2670293, 1553), (2671846, 1032), (2672878, 1056), (2673934, 1189), (2675123, 1135), (2676258, 857), (2677115, 983), (2678098, 1199), (2679297, 1040), (2680337, 940), (2681277, 1091), (2682368, 1049), (2683417, 1309), (2684726, 965), (2685691, 1198), (2686889, 972), (2687861, 994), (2688855, 1026), (2689881, 787), (2690668, 1190), (2691858, 1224), (2693082, 992), (2694074, 668), (2694742, 991), (2695733, 980), (2696713, 1079), (2697792, 1250), (2699042, 1589), (2700631, 858), (2701489, 940), (2702429, 953), (2703382, 857), (2704239, 1000), (2705239, 1021), (2706260, 1130), (2707390, 1326), (2708716, 1172), (2709888, 893), (2710781, 1089), (2711870, 999), (2712869, 721), (2713590, 905), (2714495, 1134), (2715629, 980), (2716609, 1061), (2717670, 797), (2718467, 1018), (2719485, 867), (2720352, 960), (2721312, 860), (2722172, 959), (2723131, 1033), (2724164, 780), (2724944, 1090), (2726034, 846), (2726880, 1161), (2728041, 847), (2728888, 1021), (2729909, 848), (2730757, 992), (2731749, 1095), (2732844, 1153), (2733997, 1073), (2735070, 1233), (2736303, 1360), (2737663, 1212), (2738875, 934), (2739809, 1032), (2740841, 724), (2741565, 1169), (2742734, 1023), (2743757, 833), (2744590, 891), (2745481, 1059), (2746540, 1192), (2747732, 1076), (2748808, 753), (2749561, 890), (2750451, 1070), (2751521, 722), (2752243, 724), (2752967, 1209), (2754176, 1306), (2755482, 1079), (2756561, 967), (2757528, 971), (2758499, 1066), (2759565, 1057), (2760622, 999), (2761621, 930), (2762551, 864), (2763415, 886), (2764301, 848), (2765149, 1340), (2766489, 985), (2767474, 1032), (2768506, 731), (2769237, 1033), (2770270, 1046), (2771316, 1002), (2772318, 1217), (2773535, 909), (2774444, 1247), (2775691, 1056), (2776747, 904), (2777651, 1080), (2778731, 908), (2779639, 1133), (2780772, 1084), (2781856, 984), (2782840, 968), (2783808, 1019), (2784827, 818), (2785645, 955), (2786600, 926), (2787526, 965), (2788491, 784), (2789275, 793), (2790068, 708), (2790776, 939), (2791715, 842), (2792557, 960), (2793517, 900), (2794417, 991), (2795408, 833), (2796241, 700), (2796941, 1174), (2798115, 1052), (2799167, 914), (2800081, 953), (2801034, 1205), (2802239, 980), (2803219, 859), (2804078, 1267), (2805345, 947), (2806292, 918), (2807210, 1083), (2808293, 1340), (2809633, 947), (2810580, 869), (2811449, 1092), (2812541, 967), (2813508, 895), (2814403, 1002), (2815405, 814), (2816219, 1261), (2817480, 1023), (2818503, 1107), (2819610, 1183), (2820793, 1063), (2821856, 838), (2822694, 1021), (2823715, 932), (2824647, 2147), (2826794, 805), (2827599, 1008), (2828607, 972), (2829579, 1289), (2830868, 890), (2831758, 975), (2832733, 871), (2833604, 780), (2834384, 1223), (2835607, 870), (2836477, 795), (2837272, 1022), (2838294, 1217), (2839511, 1181), (2840692, 1076), (2841768, 949), (2842717, 848), (2843565, 998), (2844563, 898), (2845461, 1063), (2846524, 866), (2847390, 1072), (2848462, 860), (2849322, 1001), (2850323, 814), (2851137, 1077), (2852214, 1242), (2853456, 961), (2854417, 1208), (2855625, 982), (2856607, 998), (2857605, 1187), (2858792, 1311), (2860103, 1035), (2861138, 947), (2862085, 1165), (2863250, 905), (2864155, 829), (2864984, 631), (2865615, 1011), (2866626, 1179), (2867805, 781), (2868586, 1297), (2869883, 819), (2870702, 1025), (2871727, 1001), (2872728, 1174), (2873902, 1162), (2875064, 956), (2876020, 1104), (2877124, 806), (2877930, 1143), (2879073, 881), (2879954, 779), (2880733, 868), (2881601, 903), (2882504, 530), (2883034, 1081), (2884115, 1237), (2885352, 1018), (2886370, 886), (2887256, 863), (2888119, 810), (2888929, 1131), (2890060, 792), (2890852, 802), (2891654, 1124), (2892778, 760), (2893538, 1154), (2894692, 2659), (2897351, 849), (2898200, 911), (2899111, 1184), (2900295, 1163), (2901458, 678)], [(2902136, 48), None, (2902184, 35), (2902219, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (2902261, 42), None, (2902303, 25), (2902328, 44), (2902372, 22), (2902394, 33), None, None, None, None, (2902427, 26), None, None, None, None, (2902453, 21), (2902474, 25), None, (2902499, 21), (2902520, 44), None, None, None, None, (2902564, 71), (2902635, 21), (2902656, 23), None, None, None, None, (2902679, 48), None, None, None, None, None, (2902727, 31), None, None, None, None, (2902758, 42), None, (2902800, 22), None, (2902822, 21), None, (2902843, 26), (2902869, 56), None, None, (2902925, 77), (2903002, 27), None, None, None, None, (2903029, 21), (2903050, 21), None, None, (2903071, 54), (2903125, 42), None, None, None, (2903167, 25), None, None, (2903192, 21), None, None, None, None, None, (2903213, 24), (2903237, 21), None, None, (2903258, 26), None, (2903284, 18), None, (2903302, 54), None, None, None, None, None, None, (2903356, 26), None, None, None, (2903382, 20), None, None, (2903402, 64), (2903466, 42), (2903508, 17), (2903525, 17), (2903542, 26), None, (2903568, 45), None, None, None, (2903613, 26), (2903639, 20), (2903659, 26), None, (2903685, 42), (2903727, 63), None, None, None, (2903790, 40), (2903830, 48), None, None, None, (2903878, 47), None, None, None, None, None, None, None, (2903925, 42), None, (2903967, 80), None, (2904047, 9), None, (2904056, 21), (2904077, 42), None, None, (2904119, 65), (2904184, 82), (2904266, 21), None, (2904287, 72), None, None, (2904359, 24), (2904383, 21), None, None, None, None, None, (2904404, 42), (2904446, 21), (2904467, 21), None, (2904488, 42), (2904530, 25), None, (2904555, 38), (2904593, 21), (2904614, 56), None, None, (2904670, 21), (2904691, 19), (2904710, 26), (2904736, 24), (2904760, 16), None, (2904776, 21), None, None, (2904797, 38), None, (2904835, 22), (2904857, 21), (2904878, 21), (2904899, 21), None, (2904920, 63), None, (2904983, 21), (2905004, 42), None, (2905046, 17), None, None, None, None, (2905063, 21), (2905084, 21), None, None, (2905105, 21), None, None, (2905126, 21), None, (2905147, 26), None, (2905173, 50), (2905223, 22), None, None, (2905245, 50), (2905295, 26), (2905321, 21), (2905342, 21), (2905363, 19), None, (2905382, 35), (2905417, 26), (2905443, 23), (2905466, 39), (2905505, 42), None, None, None, None, None, None, (2905547, 21), None, None, None, (2905568, 21), None, None, (2905589, 90), None, (2905679, 239), (2905918, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
