"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const common_substrings_1 = require("@udondan/common-substrings");
const RegexParser = require("regex-parser");
const access_level_1 = require("../access-level");
const _2_conditions_1 = require("./2-conditions");
/**
 * (experimental) Adds "action" functionality to the Policy Statement.
 *
 * @experimental
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        /**
         * @experimental
         */
        this.accessLevelList = {};
        this.useNotActions = false;
        /**
         * @experimental
         */
        this.actions = [];
        this.cdkActionsApplied = false;
        this.isCompact = false;
    }
    /**
     * (experimental) Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     *
     * @experimental
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotActions ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            if (this.isCompact) {
                this.compactActions();
            }
            const actions = this.actions
                .filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            })
                .sort();
            statement[mode] = actions.length > 1 ? actions : actions[0];
        }
        return statement;
    }
    /**
     * @experimental
     */
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotActions ? 'addNotActions' : 'addActions';
            const self = this;
            if (this.isCompact) {
                this.compactActions();
            }
            const uniqueActions = this.actions
                .filter((elem, pos) => {
                return self.actions.indexOf(elem) == pos;
            })
                .sort();
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * (experimental) Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * @experimental
     */
    notActions() {
        this.useNotActions = true;
        return this;
    }
    /**
     * (experimental) Checks weather actions have been applied to the policy.
     *
     * @experimental
     */
    hasActions() {
        return this.actions.length > 0;
    }
    /**
     * (experimental) Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotActions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notActions()`.
     *
     * If the action does not contain a colon, the action will be prefixed with the service prefix of the class, e.g. `ec2:`
     *
     * @param action Actions that will be added to the statement.
     * @experimental
     */
    to(action) {
        if (this.servicePrefix.length && action.indexOf(':') < 0) {
            action = this.servicePrefix + ':' + action;
        }
        this.actions.push(action);
        return this;
    }
    /**
     * (experimental) Adds all actions of the statement provider to the statement, e.g. `actions: 'ec2:*'`.
     *
     * @experimental
     */
    allActions() {
        if (this.servicePrefix.length) {
            this.to(`${this.servicePrefix}:*`);
        }
        else {
            this.to('*');
        }
        return this;
    }
    /**
     * (experimental) Adds all actions that match one of the given regular expressions.
     *
     * @param expressions One or more regular expressions.
     * @experimental
     */
    allMatchingActions(...expressions) {
        expressions.forEach((expression) => {
            for (const [_, actions] of Object.entries(this.accessLevelList)) {
                actions.forEach((action) => {
                    if (action.match(RegexParser(expression))) {
                        this.to(`${this.servicePrefix}:${action}`);
                    }
                });
            }
        });
        return this;
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) LIST to the statement.
     *
     * Permission to list resources within the service to determine whether an object exists.
     *
     * Actions with this level of access can list objects but cannot see the contents of a resource.
     *
     * For example, the Amazon S3 action `ListBucket` has the List access level.
     *
     * @experimental
     */
    allListActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.LIST);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) READ to the statement.
     *
     * Permission to read but not edit the contents and attributes of resources in the service.
     *
     * For example, the Amazon S3 actions `GetObject` and `GetBucketLocation` have the Read access level.
     *
     * @experimental
     */
    allReadActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.READ);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) WRITE to the statement.
     *
     * Permission to create, delete, or modify resources in the service.
     *
     * For example, the Amazon S3 actions `CreateBucket`, `DeleteBucket` and `PutObject` have the Write access level.
     *
     * Write actions might also allow modifying a resource tag. However, an action that allows only changes to tags has the Tagging access level.
     *
     * @experimental
     */
    allWriteActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.WRITE);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) PERMISSION MANAGEMENT to the statement.
     *
     * Permission to grant or modify resource permissions in the service.
     *
     * For example, most IAM and AWS Organizations actions, as well as actions like the Amazon S3 actions `PutBucketPolicy` and `DeleteBucketPolicy` have the Permissions management access level.
     *
     * @experimental
     */
    allPermissionManagementActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.PERMISSION_MANAGEMENT);
    }
    /**
     * (experimental) Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) TAGGING to the statement.
     *
     * Permission to perform actions that only change the state of resource tags.
     *
     * For example, the IAM actions `TagRole` and `UntagRole` have the Tagging access level because they allow only tagging or untagging a role. However, the `CreateRole` action allows tagging a role resource when you create that role. Because the action does not only add a tag, it has the Write access level.
     *
     * @experimental
     */
    allTaggingActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.TAGGING);
    }
    addAccessLevel(accessLevel) {
        if (accessLevel in this.accessLevelList) {
            this.accessLevelList[accessLevel].forEach((action) => {
                this.to(`${this.servicePrefix}:${action}`);
            });
        }
        return this;
    }
    /**
     * (experimental) Condense action list down to a list of patterns.
     *
     * Using this method can help to reduce the policy size.
     *
     * For example, all actions with access level `list` could be reduced to a small pattern `List*`.
     *
     * @experimental
     */
    compact() {
        this.isCompact = true;
        return this;
    }
    compactActions() {
        // actions that will be included, service prefix is removed
        const includeActions = this.actions.map((elem) => {
            return elem.substr(elem.indexOf(':') + 1);
        });
        // actions that will not be included, the opposite of includeActions
        const excludeActions = [];
        for (const [_, actions] of Object.entries(this.accessLevelList)) {
            actions.forEach((action) => {
                if (!includeActions.includes(action)) {
                    excludeActions.push(`^${action}$`);
                }
            });
        }
        // will contain the index of elements that are covered by substrings
        let covered = [];
        const subs = common_substrings_1.default(includeActions.map((action) => {
            return `^${action}$`;
        }), {
            minLength: 3,
            minOccurrence: 2,
        })
            .filter((sub) => {
            // remove all substrings, that match an action we have not selected
            for (let action of excludeActions) {
                if (action.includes(sub.name)) {
                    return false;
                }
            }
            return true;
        })
            .sort((a, b) => {
            // sort list by the number of matches
            if (a.source.length < b.source.length)
                return 1;
            if (a.source.length > b.source.length)
                return -1;
            return 0;
        })
            .filter((sub) => {
            // removes substrings, that have already been covered by other substrings
            const sources = sub.source.filter((source) => {
                return !covered.includes(source);
            });
            if (sources.length > 1) {
                //add list of sources to the global list of covered actions
                covered = covered.concat(sources);
                return true;
            }
            return false;
        });
        // stores the list of patterns
        const compactActionList = [];
        subs.forEach((sub) => {
            compactActionList.push(`${this.servicePrefix}:*${sub.name}*`
                .replace('*^', '')
                .replace('$*', ''));
            sub.source.forEach((source) => {
                includeActions[source] = ''; // invalidate, will be filtered later
            });
        });
        includeActions
            .filter((action) => {
            // remove elements that have been covered by patterns, we invalidated them above
            return action.length > 0;
        })
            .forEach((action) => {
            // add actions that have not been covered by patterns to the new action list
            compactActionList.push(`${this.servicePrefix}:${action}`);
        });
        // we're done. override action list
        this.actions = compactActionList;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiMy1hY3Rpb25zLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiMy1hY3Rpb25zLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLGtFQUFvRDtBQUNwRCw0Q0FBNkM7QUFFN0Msa0RBQThDO0FBRTlDLGtEQUE4RDs7Ozs7O0FBYzlELE1BQWEsMEJBQTJCLFNBQVEsNENBQTRCO0lBQTVFOzs7OztRQUNZLG9CQUFlLEdBQW9CLEVBQUUsQ0FBQztRQUN4QyxrQkFBYSxHQUFHLEtBQUssQ0FBQzs7OztRQUNwQixZQUFPLEdBQWEsRUFBRSxDQUFDO1FBQ3pCLHNCQUFpQixHQUFHLEtBQUssQ0FBQztRQUMxQixjQUFTLEdBQUcsS0FBSyxDQUFDO0lBK1I1QixDQUFDOzs7Ozs7OztJQXhSUSxNQUFNO1FBQ1gsa0RBQWtEO1FBQ2xELElBQUksT0FBTyxJQUFJLENBQUMsWUFBWSxJQUFJLFVBQVUsRUFBRTtZQUMxQyxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7WUFDdkIsT0FBTyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7U0FDdkI7UUFDRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUN6RCxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDakMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBRWxCLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFO1lBQ3JCLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtnQkFDbEIsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO2FBQ3ZCO1lBQ0QsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU87aUJBQ3pCLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsRUFBRTtnQkFDcEIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUM7WUFDM0MsQ0FBQyxDQUFDO2lCQUNELElBQUksRUFBRSxDQUFDO1lBQ1YsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLE9BQU8sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUM3RDtRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7Ozs7SUFFTSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztRQUN2QixrREFBa0Q7UUFDbEQsT0FBTyxLQUFLLENBQUMsZUFBZSxFQUFFLENBQUM7SUFDakMsQ0FBQztJQUVPLGVBQWU7UUFDckIsSUFBSSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtZQUMzQixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQztZQUNqRSxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7WUFDbEIsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO2dCQUNsQixJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7YUFDdkI7WUFDRCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsT0FBTztpQkFDL0IsTUFBTSxDQUFDLENBQUMsSUFBSSxFQUFFLEdBQUcsRUFBRSxFQUFFO2dCQUNwQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQztZQUMzQyxDQUFDLENBQUM7aUJBQ0QsSUFBSSxFQUFFLENBQUM7WUFDVixrREFBa0Q7WUFDbEQsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsYUFBYSxDQUFDLENBQUM7U0FDOUI7UUFDRCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDO0lBQ2hDLENBQUM7Ozs7OztJQUtNLFVBQVU7UUFDZixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQztRQUMxQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Ozs7OztJQUtNLFVBQVU7UUFDZixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztJQUNqQyxDQUFDOzs7Ozs7Ozs7Ozs7O0lBYU0sRUFBRSxDQUFDLE1BQWM7UUFDdEIsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUN4RCxNQUFNLEdBQUcsSUFBSSxDQUFDLGFBQWEsR0FBRyxHQUFHLEdBQUcsTUFBTSxDQUFDO1NBQzVDO1FBRUQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDMUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDOzs7Ozs7SUFLTSxVQUFVO1FBQ2YsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRTtZQUM3QixJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLGFBQWEsSUFBSSxDQUFDLENBQUM7U0FDcEM7YUFBTTtZQUNMLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDZDtRQUNELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQzs7Ozs7OztJQVdNLGtCQUFrQixDQUFDLEdBQUcsV0FBcUI7UUFDaEQsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFVBQVUsRUFBRSxFQUFFO1lBQ2pDLEtBQUssTUFBTSxDQUFDLENBQUMsRUFBRSxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRTtnQkFDL0QsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO29CQUN6QixJQUFJLE1BQU0sQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxDQUFDLEVBQUU7d0JBQ3pDLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsYUFBYSxJQUFJLE1BQU0sRUFBRSxDQUFDLENBQUM7cUJBQzVDO2dCQUNILENBQUMsQ0FBQyxDQUFDO2FBQ0o7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQzs7Ozs7Ozs7Ozs7O0lBV00sY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsMEJBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMvQyxDQUFDOzs7Ozs7Ozs7O0lBU00sY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsMEJBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMvQyxDQUFDOzs7Ozs7Ozs7Ozs7SUFXTSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQywwQkFBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2hELENBQUM7Ozs7Ozs7Ozs7SUFTTSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLDBCQUFXLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUNoRSxDQUFDOzs7Ozs7Ozs7O0lBU00saUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQywwQkFBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFTyxjQUFjLENBQUMsV0FBd0I7UUFDN0MsSUFBSSxXQUFXLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN2QyxJQUFJLENBQUMsZUFBZSxDQUFDLFdBQVcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO2dCQUNuRCxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLGFBQWEsSUFBSSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1lBQzdDLENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Ozs7Ozs7Ozs7SUFTTSxPQUFPO1FBQ1osSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7UUFDdEIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU8sY0FBYztRQUNwQiwyREFBMkQ7UUFDM0QsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtZQUMvQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUM1QyxDQUFDLENBQUMsQ0FBQztRQUVILG9FQUFvRTtRQUNwRSxNQUFNLGNBQWMsR0FBYSxFQUFFLENBQUM7UUFDcEMsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxFQUFFO1lBQy9ELE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDekIsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUU7b0JBQ3BDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxNQUFNLEdBQUcsQ0FBQyxDQUFDO2lCQUNwQztZQUNILENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxvRUFBb0U7UUFDcEUsSUFBSSxPQUFPLEdBQWEsRUFBRSxDQUFDO1FBRTNCLE1BQU0sSUFBSSxHQUFHLDJCQUFVLENBQ3JCLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUM1QixPQUFPLElBQUksTUFBTSxHQUFHLENBQUM7UUFDdkIsQ0FBQyxDQUFDLEVBQ0Y7WUFDRSxTQUFTLEVBQUUsQ0FBQztZQUNaLGFBQWEsRUFBRSxDQUFDO1NBQ2pCLENBQ0Y7YUFDRSxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtZQUNkLG1FQUFtRTtZQUNuRSxLQUFLLElBQUksTUFBTSxJQUFJLGNBQWMsRUFBRTtnQkFDakMsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRTtvQkFDN0IsT0FBTyxLQUFLLENBQUM7aUJBQ2Q7YUFDRjtZQUNELE9BQU8sSUFBSSxDQUFDO1FBQ2QsQ0FBQyxDQUFDO2FBQ0QsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQ2IscUNBQXFDO1lBQ3JDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNO2dCQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2hELElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNO2dCQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDakQsT0FBTyxDQUFDLENBQUM7UUFDWCxDQUFDLENBQUM7YUFDRCxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtZQUNkLHlFQUF5RTtZQUN6RSxNQUFNLE9BQU8sR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO2dCQUMzQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNuQyxDQUFDLENBQUMsQ0FBQztZQUNILElBQUksT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3RCLDJEQUEyRDtnQkFDM0QsT0FBTyxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQ2xDLE9BQU8sSUFBSSxDQUFDO2FBQ2I7WUFDRCxPQUFPLEtBQUssQ0FBQztRQUNmLENBQUMsQ0FBQyxDQUFDO1FBRUwsOEJBQThCO1FBQzlCLE1BQU0saUJBQWlCLEdBQWEsRUFBRSxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtZQUNuQixpQkFBaUIsQ0FBQyxJQUFJLENBQ3BCLEdBQUcsSUFBSSxDQUFDLGFBQWEsS0FBSyxHQUFHLENBQUMsSUFBSSxHQUFHO2lCQUNsQyxPQUFPLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQztpQkFDakIsT0FBTyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FDckIsQ0FBQztZQUNGLEdBQUcsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQzVCLGNBQWMsQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxxQ0FBcUM7WUFDcEUsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILGNBQWM7YUFDWCxNQUFNLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNqQixnRkFBZ0Y7WUFDaEYsT0FBTyxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztRQUMzQixDQUFDLENBQUM7YUFDRCxPQUFPLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNsQiw0RUFBNEU7WUFDNUUsaUJBQWlCLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLGFBQWEsSUFBSSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBQzVELENBQUMsQ0FBQyxDQUFDO1FBRUwsbUNBQW1DO1FBQ25DLElBQUksQ0FBQyxPQUFPLEdBQUcsaUJBQWlCLENBQUM7SUFDbkMsQ0FBQztDQUNGO0FBcFNELGdFQW9TQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBzdWJzdHJpbmdzIGZyb20gJ0B1ZG9uZGFuL2NvbW1vbi1zdWJzdHJpbmdzJztcbmltcG9ydCBSZWdleFBhcnNlciA9IHJlcXVpcmUoJ3JlZ2V4LXBhcnNlcicpO1xuXG5pbXBvcnQgeyBBY2Nlc3NMZXZlbCB9IGZyb20gJy4uL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50V2l0aENvbmRpdGlvbiB9IGZyb20gJy4vMi1jb25kaXRpb25zJztcblxuZXhwb3J0IGludGVyZmFjZSBBY3Rpb24ge1xuICB1cmw6IHN0cmluZztcbiAgZGVzY3JpcHRpb246IHN0cmluZztcbiAgYWNjZXNzTGV2ZWw6IHN0cmluZztcbiAgcmVzb3VyY2VUeXBlcz86IGFueTtcbiAgY29uZGl0aW9ucz86IHN0cmluZ1tdO1xuICBkZXBlbmRlbnRBY3Rpb25zPzogc3RyaW5nW107XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgUG9saWN5U3RhdGVtZW50V2l0aEFjdGlvbnMgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnRXaXRoQ29uZGl0aW9uIHtcbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge307XG4gIHByaXZhdGUgdXNlTm90QWN0aW9ucyA9IGZhbHNlO1xuICBwcm90ZWN0ZWQgYWN0aW9uczogc3RyaW5nW10gPSBbXTtcbiAgcHJpdmF0ZSBjZGtBY3Rpb25zQXBwbGllZCA9IGZhbHNlO1xuICBwcml2YXRlIGlzQ29tcGFjdCA9IGZhbHNlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyB0b0pTT04oKTogYW55IHtcbiAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgIGlmICh0eXBlb2YgdGhpcy5hZGRSZXNvdXJjZXMgPT0gJ2Z1bmN0aW9uJykge1xuICAgICAgdGhpcy5jZGtBcHBseUFjdGlvbnMoKTtcbiAgICAgIHJldHVybiBzdXBlci50b0pTT04oKTtcbiAgICB9XG4gICAgY29uc3QgbW9kZSA9IHRoaXMudXNlTm90QWN0aW9ucyA/ICdOb3RBY3Rpb24nIDogJ0FjdGlvbic7XG4gICAgY29uc3Qgc3RhdGVtZW50ID0gc3VwZXIudG9KU09OKCk7XG4gICAgY29uc3Qgc2VsZiA9IHRoaXM7XG5cbiAgICBpZiAodGhpcy5oYXNBY3Rpb25zKCkpIHtcbiAgICAgIGlmICh0aGlzLmlzQ29tcGFjdCkge1xuICAgICAgICB0aGlzLmNvbXBhY3RBY3Rpb25zKCk7XG4gICAgICB9XG4gICAgICBjb25zdCBhY3Rpb25zID0gdGhpcy5hY3Rpb25zXG4gICAgICAgIC5maWx0ZXIoKGVsZW0sIHBvcykgPT4ge1xuICAgICAgICAgIHJldHVybiBzZWxmLmFjdGlvbnMuaW5kZXhPZihlbGVtKSA9PSBwb3M7XG4gICAgICAgIH0pXG4gICAgICAgIC5zb3J0KCk7XG4gICAgICBzdGF0ZW1lbnRbbW9kZV0gPSBhY3Rpb25zLmxlbmd0aCA+IDEgPyBhY3Rpb25zIDogYWN0aW9uc1swXTtcbiAgICB9XG5cbiAgICByZXR1cm4gc3RhdGVtZW50O1xuICB9XG5cbiAgcHVibGljIHRvU3RhdGVtZW50SnNvbigpOiBhbnkge1xuICAgIHRoaXMuY2RrQXBwbHlBY3Rpb25zKCk7XG4gICAgLy8gQHRzLWlnbm9yZSBvbmx5IGF2YWlsYWJsZSBhZnRlciBzd2FwcGluZyAxLWJhc2VcbiAgICByZXR1cm4gc3VwZXIudG9TdGF0ZW1lbnRKc29uKCk7XG4gIH1cblxuICBwcml2YXRlIGNka0FwcGx5QWN0aW9ucygpIHtcbiAgICBpZiAoIXRoaXMuY2RrQWN0aW9uc0FwcGxpZWQpIHtcbiAgICAgIGNvbnN0IG1vZGUgPSB0aGlzLnVzZU5vdEFjdGlvbnMgPyAnYWRkTm90QWN0aW9ucycgOiAnYWRkQWN0aW9ucyc7XG4gICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgIGlmICh0aGlzLmlzQ29tcGFjdCkge1xuICAgICAgICB0aGlzLmNvbXBhY3RBY3Rpb25zKCk7XG4gICAgICB9XG4gICAgICBjb25zdCB1bmlxdWVBY3Rpb25zID0gdGhpcy5hY3Rpb25zXG4gICAgICAgIC5maWx0ZXIoKGVsZW0sIHBvcykgPT4ge1xuICAgICAgICAgIHJldHVybiBzZWxmLmFjdGlvbnMuaW5kZXhPZihlbGVtKSA9PSBwb3M7XG4gICAgICAgIH0pXG4gICAgICAgIC5zb3J0KCk7XG4gICAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgICAgdGhpc1ttb2RlXSguLi51bmlxdWVBY3Rpb25zKTtcbiAgICB9XG4gICAgdGhpcy5jZGtBY3Rpb25zQXBwbGllZCA9IHRydWU7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbm90QWN0aW9ucygpIHtcbiAgICB0aGlzLnVzZU5vdEFjdGlvbnMgPSB0cnVlO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgaGFzQWN0aW9ucygpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdGhpcy5hY3Rpb25zLmxlbmd0aCA+IDA7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHRvKGFjdGlvbjogc3RyaW5nKSB7XG4gICAgaWYgKHRoaXMuc2VydmljZVByZWZpeC5sZW5ndGggJiYgYWN0aW9uLmluZGV4T2YoJzonKSA8IDApIHtcbiAgICAgIGFjdGlvbiA9IHRoaXMuc2VydmljZVByZWZpeCArICc6JyArIGFjdGlvbjtcbiAgICB9XG5cbiAgICB0aGlzLmFjdGlvbnMucHVzaChhY3Rpb24pO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbGxBY3Rpb25zKCkge1xuICAgIGlmICh0aGlzLnNlcnZpY2VQcmVmaXgubGVuZ3RoKSB7XG4gICAgICB0aGlzLnRvKGAke3RoaXMuc2VydmljZVByZWZpeH06KmApO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnRvKCcqJyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWxsTWF0Y2hpbmdBY3Rpb25zKC4uLmV4cHJlc3Npb25zOiBzdHJpbmdbXSkge1xuICAgIGV4cHJlc3Npb25zLmZvckVhY2goKGV4cHJlc3Npb24pID0+IHtcbiAgICAgIGZvciAoY29uc3QgW18sIGFjdGlvbnNdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMuYWNjZXNzTGV2ZWxMaXN0KSkge1xuICAgICAgICBhY3Rpb25zLmZvckVhY2goKGFjdGlvbikgPT4ge1xuICAgICAgICAgIGlmIChhY3Rpb24ubWF0Y2goUmVnZXhQYXJzZXIoZXhwcmVzc2lvbikpKSB7XG4gICAgICAgICAgICB0aGlzLnRvKGAke3RoaXMuc2VydmljZVByZWZpeH06JHthY3Rpb259YCk7XG4gICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9KTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWxsTGlzdEFjdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkQWNjZXNzTGV2ZWwoQWNjZXNzTGV2ZWwuTElTVCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbGxSZWFkQWN0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy5hZGRBY2Nlc3NMZXZlbChBY2Nlc3NMZXZlbC5SRUFEKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhbGxXcml0ZUFjdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkQWNjZXNzTGV2ZWwoQWNjZXNzTGV2ZWwuV1JJVEUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWxsUGVybWlzc2lvbk1hbmFnZW1lbnRBY3Rpb25zKCkge1xuICAgIHJldHVybiB0aGlzLmFkZEFjY2Vzc0xldmVsKEFjY2Vzc0xldmVsLlBFUk1JU1NJT05fTUFOQUdFTUVOVCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFsbFRhZ2dpbmdBY3Rpb25zKCkge1xuICAgIHJldHVybiB0aGlzLmFkZEFjY2Vzc0xldmVsKEFjY2Vzc0xldmVsLlRBR0dJTkcpO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRBY2Nlc3NMZXZlbChhY2Nlc3NMZXZlbDogQWNjZXNzTGV2ZWwpIHtcbiAgICBpZiAoYWNjZXNzTGV2ZWwgaW4gdGhpcy5hY2Nlc3NMZXZlbExpc3QpIHtcbiAgICAgIHRoaXMuYWNjZXNzTGV2ZWxMaXN0W2FjY2Vzc0xldmVsXS5mb3JFYWNoKChhY3Rpb24pID0+IHtcbiAgICAgICAgdGhpcy50byhgJHt0aGlzLnNlcnZpY2VQcmVmaXh9OiR7YWN0aW9ufWApO1xuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgY29tcGFjdCgpIHtcbiAgICB0aGlzLmlzQ29tcGFjdCA9IHRydWU7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwcml2YXRlIGNvbXBhY3RBY3Rpb25zKCkge1xuICAgIC8vIGFjdGlvbnMgdGhhdCB3aWxsIGJlIGluY2x1ZGVkLCBzZXJ2aWNlIHByZWZpeCBpcyByZW1vdmVkXG4gICAgY29uc3QgaW5jbHVkZUFjdGlvbnMgPSB0aGlzLmFjdGlvbnMubWFwKChlbGVtKSA9PiB7XG4gICAgICByZXR1cm4gZWxlbS5zdWJzdHIoZWxlbS5pbmRleE9mKCc6JykgKyAxKTtcbiAgICB9KTtcblxuICAgIC8vIGFjdGlvbnMgdGhhdCB3aWxsIG5vdCBiZSBpbmNsdWRlZCwgdGhlIG9wcG9zaXRlIG9mIGluY2x1ZGVBY3Rpb25zXG4gICAgY29uc3QgZXhjbHVkZUFjdGlvbnM6IHN0cmluZ1tdID0gW107XG4gICAgZm9yIChjb25zdCBbXywgYWN0aW9uc10gb2YgT2JqZWN0LmVudHJpZXModGhpcy5hY2Nlc3NMZXZlbExpc3QpKSB7XG4gICAgICBhY3Rpb25zLmZvckVhY2goKGFjdGlvbikgPT4ge1xuICAgICAgICBpZiAoIWluY2x1ZGVBY3Rpb25zLmluY2x1ZGVzKGFjdGlvbikpIHtcbiAgICAgICAgICBleGNsdWRlQWN0aW9ucy5wdXNoKGBeJHthY3Rpb259JGApO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyB3aWxsIGNvbnRhaW4gdGhlIGluZGV4IG9mIGVsZW1lbnRzIHRoYXQgYXJlIGNvdmVyZWQgYnkgc3Vic3RyaW5nc1xuICAgIGxldCBjb3ZlcmVkOiBudW1iZXJbXSA9IFtdO1xuXG4gICAgY29uc3Qgc3VicyA9IHN1YnN0cmluZ3MoXG4gICAgICBpbmNsdWRlQWN0aW9ucy5tYXAoKGFjdGlvbikgPT4ge1xuICAgICAgICByZXR1cm4gYF4ke2FjdGlvbn0kYDtcbiAgICAgIH0pLFxuICAgICAge1xuICAgICAgICBtaW5MZW5ndGg6IDMsXG4gICAgICAgIG1pbk9jY3VycmVuY2U6IDIsXG4gICAgICB9XG4gICAgKVxuICAgICAgLmZpbHRlcigoc3ViKSA9PiB7XG4gICAgICAgIC8vIHJlbW92ZSBhbGwgc3Vic3RyaW5ncywgdGhhdCBtYXRjaCBhbiBhY3Rpb24gd2UgaGF2ZSBub3Qgc2VsZWN0ZWRcbiAgICAgICAgZm9yIChsZXQgYWN0aW9uIG9mIGV4Y2x1ZGVBY3Rpb25zKSB7XG4gICAgICAgICAgaWYgKGFjdGlvbi5pbmNsdWRlcyhzdWIubmFtZSkpIHtcbiAgICAgICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICB9KVxuICAgICAgLnNvcnQoKGEsIGIpID0+IHtcbiAgICAgICAgLy8gc29ydCBsaXN0IGJ5IHRoZSBudW1iZXIgb2YgbWF0Y2hlc1xuICAgICAgICBpZiAoYS5zb3VyY2UubGVuZ3RoIDwgYi5zb3VyY2UubGVuZ3RoKSByZXR1cm4gMTtcbiAgICAgICAgaWYgKGEuc291cmNlLmxlbmd0aCA+IGIuc291cmNlLmxlbmd0aCkgcmV0dXJuIC0xO1xuICAgICAgICByZXR1cm4gMDtcbiAgICAgIH0pXG4gICAgICAuZmlsdGVyKChzdWIpID0+IHtcbiAgICAgICAgLy8gcmVtb3ZlcyBzdWJzdHJpbmdzLCB0aGF0IGhhdmUgYWxyZWFkeSBiZWVuIGNvdmVyZWQgYnkgb3RoZXIgc3Vic3RyaW5nc1xuICAgICAgICBjb25zdCBzb3VyY2VzID0gc3ViLnNvdXJjZS5maWx0ZXIoKHNvdXJjZSkgPT4ge1xuICAgICAgICAgIHJldHVybiAhY292ZXJlZC5pbmNsdWRlcyhzb3VyY2UpO1xuICAgICAgICB9KTtcbiAgICAgICAgaWYgKHNvdXJjZXMubGVuZ3RoID4gMSkge1xuICAgICAgICAgIC8vYWRkIGxpc3Qgb2Ygc291cmNlcyB0byB0aGUgZ2xvYmFsIGxpc3Qgb2YgY292ZXJlZCBhY3Rpb25zXG4gICAgICAgICAgY292ZXJlZCA9IGNvdmVyZWQuY29uY2F0KHNvdXJjZXMpO1xuICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH0pO1xuXG4gICAgLy8gc3RvcmVzIHRoZSBsaXN0IG9mIHBhdHRlcm5zXG4gICAgY29uc3QgY29tcGFjdEFjdGlvbkxpc3Q6IHN0cmluZ1tdID0gW107XG4gICAgc3Vicy5mb3JFYWNoKChzdWIpID0+IHtcbiAgICAgIGNvbXBhY3RBY3Rpb25MaXN0LnB1c2goXG4gICAgICAgIGAke3RoaXMuc2VydmljZVByZWZpeH06KiR7c3ViLm5hbWV9KmBcbiAgICAgICAgICAucmVwbGFjZSgnKl4nLCAnJylcbiAgICAgICAgICAucmVwbGFjZSgnJConLCAnJylcbiAgICAgICk7XG4gICAgICBzdWIuc291cmNlLmZvckVhY2goKHNvdXJjZSkgPT4ge1xuICAgICAgICBpbmNsdWRlQWN0aW9uc1tzb3VyY2VdID0gJyc7IC8vIGludmFsaWRhdGUsIHdpbGwgYmUgZmlsdGVyZWQgbGF0ZXJcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgaW5jbHVkZUFjdGlvbnNcbiAgICAgIC5maWx0ZXIoKGFjdGlvbikgPT4ge1xuICAgICAgICAvLyByZW1vdmUgZWxlbWVudHMgdGhhdCBoYXZlIGJlZW4gY292ZXJlZCBieSBwYXR0ZXJucywgd2UgaW52YWxpZGF0ZWQgdGhlbSBhYm92ZVxuICAgICAgICByZXR1cm4gYWN0aW9uLmxlbmd0aCA+IDA7XG4gICAgICB9KVxuICAgICAgLmZvckVhY2goKGFjdGlvbikgPT4ge1xuICAgICAgICAvLyBhZGQgYWN0aW9ucyB0aGF0IGhhdmUgbm90IGJlZW4gY292ZXJlZCBieSBwYXR0ZXJucyB0byB0aGUgbmV3IGFjdGlvbiBsaXN0XG4gICAgICAgIGNvbXBhY3RBY3Rpb25MaXN0LnB1c2goYCR7dGhpcy5zZXJ2aWNlUHJlZml4fToke2FjdGlvbn1gKTtcbiAgICAgIH0pO1xuXG4gICAgLy8gd2UncmUgZG9uZS4gb3ZlcnJpZGUgYWN0aW9uIGxpc3RcbiAgICB0aGlzLmFjdGlvbnMgPSBjb21wYWN0QWN0aW9uTGlzdDtcbiAgfVxufVxuIl19