"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImagePipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const iam = require("aws-cdk-lib/aws-iam");
const imagebuilder = require("aws-cdk-lib/aws-imagebuilder");
const kms = require("aws-cdk-lib/aws-kms");
const sns = require("aws-cdk-lib/aws-sns");
const subscriptions = require("aws-cdk-lib/aws-sns-subscriptions");
const constructs_1 = require("constructs");
class ImagePipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let infrastructureConfig;
        let imageRecipe;
        this.imageRecipeComponents = [];
        // Construct code below
        const kmsKey = kms.Key.fromLookup(this, 'KmsKeyLookup', {
            aliasName: props.kmsKeyAlias,
        });
        const topic = new sns.Topic(this, 'ImageBuilderTopic', {
            displayName: 'Image Builder Notify',
            masterKey: kmsKey,
        });
        if (props.email != null) {
            topic.addSubscription(new subscriptions.EmailSubscription(props.email));
        }
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'IAM role used as part of an Image Builder pipeline',
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'));
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'));
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        if (typeof props.additionalPolicies !== 'undefined' && props.additionalPolicies.length >= 1) {
            for (const policy of props.additionalPolicies) {
                role.addManagedPolicy(policy);
            }
        }
        const profile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: props.profileName,
        });
        if (props.securityGroups == null || props.subnetId == null) {
            infrastructureConfig = new imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: props.profileName,
                name: props.infraConfigName,
                description: 'Example Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
            });
        }
        else {
            infrastructureConfig = new imagebuilder.CfnInfrastructureConfiguration(this, 'InfrastructureConfiguration', {
                instanceProfileName: props.profileName,
                name: props.infraConfigName,
                description: 'Example Infrastructure Configuration for Image Builder',
                instanceTypes: props.instanceTypes ?? ['t3.medium', 'm5.large', 'm5.xlarge'],
                snsTopicArn: topic.topicArn,
                securityGroupIds: props.securityGroups,
                subnetId: props.subnetId,
            });
        }
        infrastructureConfig.addDependency(profile);
        /**
         * Only add overridding UserData script if it is given
         */
        if (props.userDataScript?.trim() === '') {
            imageRecipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
                components: [],
                name: props.imageRecipe,
                parentImage: props.parentImage,
                version: props.imageRecipeVersion ?? '0.0.1',
            });
        }
        else {
            imageRecipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
                components: [],
                name: props.imageRecipe,
                parentImage: props.parentImage,
                version: props.imageRecipeVersion ?? '0.0.1',
                additionalInstanceConfiguration: {
                    userDataOverride: props.userDataScript,
                },
            });
        }
        props.componentDocuments.forEach((document, index) => {
            let component = new imagebuilder.CfnComponent(this, props.componentNames[index], {
                name: props.componentNames[index],
                platform: props.platform ?? 'Linux',
                version: props.componentVersions[index] ?? '0.0.1',
                data: fs_1.readFileSync(document).toString(),
            });
            // add the component to the Image Recipe
            this.imageRecipeComponents.push({ componentArn: component.attrArn });
            imageRecipe.components = this.imageRecipeComponents;
        });
        new imagebuilder.CfnImagePipeline(this, 'ImagePipeline', {
            infrastructureConfigurationArn: infrastructureConfig.attrArn,
            name: props.pipelineName,
            description: 'A sample image pipeline',
            imageRecipeArn: imageRecipe.attrArn,
        });
    }
}
exports.ImagePipeline = ImagePipeline;
_a = JSII_RTTI_SYMBOL_1;
ImagePipeline[_a] = { fqn: "cdk-image-pipeline.ImagePipeline", version: "0.1.46" };
//# sourceMappingURL=data:application/json;base64,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