"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Scheduler = exports.BaseOS = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const info_1 = require("./info");
/**
 * @experimental
 */
var BaseOS;
(function (BaseOS) {
    BaseOS["CENTOS_7"] = "centos7";
    BaseOS["RHEL_7"] = "rhel7";
    BaseOS["AMZN2"] = "amazonlinux2";
})(BaseOS = exports.BaseOS || (exports.BaseOS = {}));
/**
 * @experimental
 */
class Scheduler extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const urlsuffix = cdk.Stack.of(this).urlSuffix;
        const stack = cdk.Stack.of(this);
        const region = cdk.Stack.of(this).region;
        const s3InstallBucket = props.s3InstallBucket;
        const s3InstallFolder = props.s3InstallFolder;
        const baseOs = props.baseOs ? props.baseOs.toString() : BaseOS.AMZN2.toString();
        const clusterId = props.network.clusterId;
        const socaVersion = info_1.SocaInfo.Data.Version;
        const ldapUserName = props.ldapUserName;
        const ldapUserPassword = props.ldapUserPassword;
        const socaInstallAmi = props.customAmi ? ec2.MachineImage.genericLinux({
            [region]: props.customAmi,
        }) : ec2.MachineImage.genericLinux({
            [region]: info_1.RegionMap[region][baseOs],
        });
        const socaInstallAmiId = socaInstallAmi.getImage(this).imageId;
        // const bootscript: string = fs.readFileSync(path.join(__dirname, '../assets/user-data'), 'utf-8');
        const userData = ec2.UserData.forLinux();
        // userData.addCommands(bootscript)
        userData.addCommands(`
export PATH=$PATH:/usr/local/bin
# Deactivate shell to make sure users won't access the cluster if it's not ready
echo '
************* SOCA FIRST TIME CONFIGURATION *************
Hold on, cluster is not ready yet.
Please wait ~30 minutes as SOCA is being installed.
Once cluster is ready to use, this message will be replaced automatically and you will be able to SSH.
*********************************************************' > /etc/nologin

if [ "${baseOs}" == "amazonlinux2" ] || [ "${baseOs}" == "rhel7" ];
  then
    usermod --shell /usr/sbin/nologin ec2-user
fi

if [ "${baseOs}" == "centos7" ];
  then
    usermod --shell /usr/sbin/nologin centos
fi

# Install awscli
if [ "${baseOs}" == "centos7" ] || [ "${baseOs}" == "rhel7" ];
then
  EASY_INSTALL=$(which easy_install-2.7)
  $EASY_INSTALL pip
  PIP=$(which pip2.7)
  $PIP install awscli
fi

# Disable automatic motd update if using ALI
if [ "${baseOs}" == "amazonlinux2" ];
then
  /usr/sbin/update-motd --disable
  rm /etc/cron.d/update-motd
  rm -f /etc/update-motd.d/*
fi

AWS=$(which aws)
echo export "SOCA_BASE_OS=${baseOs}" >> /etc/environment
echo export "SOCA_CONFIGURATION=${clusterId}" >> /etc/environment
echo export "AWS_DEFAULT_REGION=${region} " >> /etc/environment
echo export "SOCA_INSTALL_BUCKET=${s3InstallBucket}" >> /etc/environment
echo export "SOCA_INSTALL_BUCKET_FOLDER=${s3InstallFolder}" >> /etc/environment
echo export "SOCA_VERSION=${socaVersion}" >> /etc/environment
echo export "SOCA_INSTALL_AMI=${socaInstallAmiId}" >> /etc/environment
source /etc/environment

# Tag EBS disks manually as CFN ASG does not support it
AWS_AVAIL_ZONE=$(curl http://169.254.169.254/latest/meta-data/placement/availability-zone)
AWS_REGION=$(echo \"$AWS_AVAIL_ZONE\" | sed "s/[a-z]$//")
AWS_INSTANCE_ID=$(curl http://169.254.169.254/latest/meta-data/instance-id)
EBS_IDS=$(aws ec2 describe-volumes --filters Name=attachment.instance-id,Values="$AWS_INSTANCE_ID" --region $AWS_REGION --query "Volumes[*].[VolumeId]" --out text | tr "\n" " ")
$AWS ec2 create-tags --resources $EBS_IDS --region $AWS_REGION --tags Key=Name,Value="${clusterId} Root Disk" "Key=soca:ClusterId,Value=${clusterId}"

# Tag Network Adapter for the Scheduler
ENI_IDS=$(aws ec2 describe-network-interfaces --filters Name=attachment.instance-id,Values="$AWS_INSTANCE_ID" --region $AWS_REGION --query "NetworkInterfaces[*].[NetworkInterfaceId]" --out text | tr "\n" " ")
$AWS ec2 create-tags --resources $ENI_IDS --region $AWS_REGION --tags Key=Name,Value="${clusterId} Scheduler Network Adapter" "Key=soca:ClusterId,Value=${clusterId}"

echo "@reboot /bin/aws s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/SchedulerPostReboot.sh /root && /bin/bash /root/SchedulerPostReboot.sh ${s3InstallBucket} ${s3InstallFolder} ${ldapUserName} '${ldapUserPassword}' >> /root/PostRebootConfig.log 2>&1" | crontab -
AWS=$(which aws)
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/config.cfg /root/
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/requirements.txt /root/
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/Scheduler.sh /root/
/bin/bash /root/Scheduler.sh ${props.storage.efsDataDns} ${props.storage.efsAppsDns} >> /root/Scheduler.sh.log 2>&1
`);
        const scheduler = new ec2.Instance(this, 'Scheduler', {
            vpc: props.network.vpc,
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : new ec2.InstanceType('m5.xlarge'),
            machineImage: socaInstallAmi,
            userData,
            securityGroup: props.schedulerSecurityGroup,
        });
        scheduler.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'pricing:GetProducts',
                'budgets:ViewBudget',
                'ec2:DescribeInstances',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
                'ec2:DescribeImages',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceTypes',
                'ec2:DescribeReservedInstances',
                'ec2:DescribeSpotInstanceRequests',
                'ec2:DescribeVpcClassicLink',
                'ec2:DescribeVolumes',
                'ec2:DescribePlacementGroups',
                'ec2:DescribeKeyPairs',
                'ec2:DescribeLaunchTemplates',
                'ec2:DescribeLaunchTemplateVersions',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DescribeSpotFleetRequests',
                'ec2:DescribeSpotFleetInstances',
                'fsx:DescribeFileSystems',
                'autoscaling:DescribeAutoScalingGroups',
                'autoscaling:DescribeScalingActivities',
                'autoscaling:DescribeLaunchConfigurations',
                'elasticloadbalancing:DescribeRules',
                'elasticloadbalancing:DescribeListeners',
                'elasticloadbalancing:DescribeTargetGroups',
                'savingsplans:DescribeSavingsPlans',
                'servicequotas:ListServiceQuotas',
            ],
            resources: ['*'],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
                'autoscaling:DeleteAutoScalingGroup',
                'autoscaling:CreateAutoScalingGroup',
                'autoscaling:DetachInstances',
                'ec2:DeleteLaunchTemplate',
                'ec2:CreateLaunchTemplate',
                'fsx:CreateDataRepositoryTask',
            ],
            resources: ['*'],
            conditions: {
                StringLikeIfExists: {
                    'autoscaling:LaunchConfigurationName': props.network.clusterId,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:CreateTags',
            ],
            resources: [
                stack.formatArn({ service: 'ec2', resource: 'volume' }),
                stack.formatArn({ service: 'ec2', resource: 'network-interface' }),
                stack.formatArn({ service: 'ec2', resource: 'instance' }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'cloudformation:CreateStack',
                'cloudformation:DeleteStack',
                'cloudformation:DescribeStacks',
            ],
            resources: ['*'],
            conditions: {
                'ForAllValues:StringEquals': {
                    'cloudformation:TemplateURL': `https://s3.${urlsuffix}/${s3InstallBucket}/${s3InstallFolder}/templates/ComputeNode.template`,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:RunInstances',
                'ec2:TerminateInstances',
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
            ],
            resources: [
                ...['subnet', 'key-pair', 'instance', 'launch-template', 'volume', 'security-group', 'placement-group', 'network-interface'].map(resource => stack.formatArn({ service: 'ec2', resource })).concat(['snapshot', 'image'].map(resource => stack.formatArn({ service: 'ec2', resource, account: '' }))),
            ],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'lambda:InvokeFunction',
            ],
            resources: [
                stack.formatArn({
                    service: 'lambda',
                    resource: 'function',
                    resourceName: `${props.network.clusterId}-Metrics`,
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'fsx:CreateFileSystem',
            ],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'fsx:DeleteFileSystem',
            ],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
            conditions: {
                StringLike: {
                    'aws:ResourceTag/soca:ClusterId': props.network.clusterId,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'iam:CreateServiceLinkedRole',
                'iam:AttachRolePolicy',
                'iam:PutRolePolicy',
            ],
            resources: [
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/s3.data-source.lustre.fsx.amazonaws.com/*',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/autoscaling.amazonaws.com/*',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/spotfleet.amazonaws.com/*',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ses:SendEmail',
            ],
            resources: [
                stack.formatArn({
                    service: 'ses',
                    resource: 'identity',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
                'ec2:RequestSpotFleet',
                'ec2:ModifySpotFleetRequest',
                'ec2:CancelSpotFleetRequests',
            ],
            resources: ['*'],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({
                        service: 'ec2',
                        resource: 'vpc',
                        resourceName: props.network.vpc.vpcId,
                    }),
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject',
                's3:ListBucket',
                's3:PutObject',
            ],
            resources: [
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: props.s3InstallBucket,
                }),
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `${props.s3InstallBucket}/*`,
                }),
            ],
        }));
        const eip = new ec2.CfnEIP(this, 'EIPScheduler', {
            instanceId: scheduler.instanceId,
            domain: props.network.vpc.vpcId,
        });
        new cdk.CfnOutput(this, 'SchedulerEIP', { value: eip.ref });
    }
}
exports.Scheduler = Scheduler;
//# sourceMappingURL=data:application/json;base64,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