"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'log retention construct'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lib_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: lib_1.RetentionDays.ONE_MONTH,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'logs:PutRetentionPolicy',
                            'logs:DeleteRetentionPolicy',
                        ],
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aServiceRoleDefaultPolicyADDA7DEB',
            'Roles': [
                {
                    'Ref': 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aServiceRole9741ECFB',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            'ServiceToken': {
                'Fn::GetAtt': [
                    'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                    'Arn',
                ],
            },
            'LogGroupName': 'group',
            'RetentionInDays': 30,
        }));
        test.done();
    },
    'with imported role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::123456789012:role/CoolRole');
        // WHEN
        new lib_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: lib_1.RetentionDays.ONE_MONTH,
            role,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'logs:PutRetentionPolicy',
                            'logs:DeleteRetentionPolicy',
                        ],
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'RolePolicy72E7D967',
            'Roles': [
                'CoolRole',
            ],
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::IAM::Role', 0));
        test.done();
    },
    'with RetentionPeriod set to Infinity'(test) {
        const stack = new cdk.Stack();
        new lib_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: lib_1.RetentionDays.INFINITE,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: assert_1.ABSENT,
        }));
        test.done();
    },
    'with LogGroupRegion specified'(test) {
        const stack = new cdk.Stack();
        new lib_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            logGroupRegion: 'us-east-1',
            retention: lib_1.RetentionDays.INFINITE,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupRegion: 'us-east-1',
        }));
        test.done();
    },
    'log group ARN is well formed and conforms'(test) {
        const stack = new cdk.Stack();
        const group = new lib_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: lib_1.RetentionDays.ONE_MONTH,
        });
        const logGroupArn = group.logGroupArn;
        test.ok(logGroupArn.indexOf('logs') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.indexOf('log-group') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.endsWith(':*'), 'log group ARN is not as expected');
        test.done();
    },
    'log group ARN is well formed and conforms when region is specified'(test) {
        const stack = new cdk.Stack();
        const group = new lib_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            logGroupRegion: 'us-west-2',
            retention: lib_1.RetentionDays.ONE_MONTH,
        });
        const logGroupArn = group.logGroupArn;
        test.ok(logGroupArn.indexOf('us-west-2') > -1, 'region of log group ARN is not as expected');
        test.ok(logGroupArn.indexOf('logs') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.indexOf('log-group') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.endsWith(':*'), 'log group ARN is not as expected');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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