"use strict";
const lib_1 = require("../lib");
module.exports = {
    'text patterns': {
        'simple text pattern'(test) {
            const pattern = lib_1.FilterPattern.allTerms('foo', 'bar', 'baz');
            test.equal('"foo" "bar" "baz"', pattern.logPatternString);
            test.done();
        },
        'quoted terms'(test) {
            const pattern = lib_1.FilterPattern.allTerms('"foo" he said');
            test.equal('"\\"foo\\" he said"', pattern.logPatternString);
            test.done();
        },
        'disjunction of conjunctions'(test) {
            const pattern = lib_1.FilterPattern.anyTermGroup(['foo', 'bar'], ['baz']);
            test.equal('?"foo" "bar" ?"baz"', pattern.logPatternString);
            test.done();
        },
        'dont prefix with ? if only one disjunction'(test) {
            const pattern = lib_1.FilterPattern.anyTermGroup(['foo', 'bar']);
            test.equal('"foo" "bar"', pattern.logPatternString);
            test.done();
        },
        'empty log pattern is empty string'(test) {
            const pattern = lib_1.FilterPattern.anyTermGroup();
            test.equal('', pattern.logPatternString);
            test.done();
        },
    },
    'json patterns': {
        'string value'(test) {
            const pattern = lib_1.FilterPattern.stringValue('$.field', '=', 'value');
            test.equal('{ $.field = "value" }', pattern.logPatternString);
            test.done();
        },
        'also recognize =='(test) {
            const pattern = lib_1.FilterPattern.stringValue('$.field', '==', 'value');
            test.equal('{ $.field = "value" }', pattern.logPatternString);
            test.done();
        },
        'number patterns'(test) {
            const pattern = lib_1.FilterPattern.numberValue('$.field', '<=', 300);
            test.equal('{ $.field <= 300 }', pattern.logPatternString);
            test.done();
        },
        'combining with AND or OR'(test) {
            const p1 = lib_1.FilterPattern.numberValue('$.field', '<=', 300);
            const p2 = lib_1.FilterPattern.stringValue('$.field', '=', 'value');
            const andPattern = lib_1.FilterPattern.all(p1, p2);
            test.equal('{ ($.field <= 300) && ($.field = "value") }', andPattern.logPatternString);
            const orPattern = lib_1.FilterPattern.any(p1, p2);
            test.equal('{ ($.field <= 300) || ($.field = "value") }', orPattern.logPatternString);
            test.done();
        },
        'single AND is not wrapped with parens'(test) {
            const p1 = lib_1.FilterPattern.stringValue('$.field', '=', 'value');
            const pattern = lib_1.FilterPattern.all(p1);
            test.equal('{ $.field = "value" }', pattern.logPatternString);
            test.done();
        },
        'empty AND is rejected'(test) {
            test.throws(() => {
                lib_1.FilterPattern.all();
            });
            test.done();
        },
        'invalid string operators are rejected'(test) {
            test.throws(() => {
                lib_1.FilterPattern.stringValue('$.field', '<=', 'hello');
            });
            test.done();
        },
        'can test boolean value'(test) {
            const pattern = lib_1.FilterPattern.booleanValue('$.field', false);
            test.equal('{ $.field IS FALSE }', pattern.logPatternString);
            test.done();
        },
    },
    'table patterns': {
        'simple model'(test) {
            const pattern = lib_1.FilterPattern.spaceDelimited('...', 'status_code', 'bytes');
            test.equal('[..., status_code, bytes]', pattern.logPatternString);
            test.done();
        },
        'add restrictions'(test) {
            const pattern = lib_1.FilterPattern.spaceDelimited('...', 'status_code', 'bytes')
                .whereString('status_code', '=', '4*')
                .whereNumber('status_code', '!=', 403);
            test.equal('[..., status_code = "4*" && status_code != 403, bytes]', pattern.logPatternString);
            test.done();
        },
        'cant use more than one ellipsis'(test) {
            test.throws(() => {
                lib_1.FilterPattern.spaceDelimited('...', 'status_code', '...');
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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