# vim: ts=4:sts=4:sw=4
#
# @author: <sylvain.herledan@oceandatalab.com>
# @date: 2016-11-04
#
# This file is part of SEAScope, a 3D visualisation and analysis application
# for satellite, in-situ and numerical model data.
#
# Copyright (C) 2014-2023 OceanDataLab
#
# SEAScope is free software: you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by the Free
# Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# SEAScope is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with SEAScope. If not, see <https://www.gnu.org/licenses/>.

"""
This module handles the serialization of the getCatalogues command and the
deserialization of the result sent by SEAScope
"""

import logging
import flatbuffers
import SEAScope.API.GetCataloguesResponse
import SEAScope.types.source

logger = logging.getLogger(__name__)


def serialize(builder):
    """
    Serialize a getCatalogues command using FlatBuffers.

    Parameters
    ----------
    builder : flatbuffers.builder.Builder
        The FlatBuffers builder instance which serializes data. If this
        parameter is None, then a new builder will be created

    Returns
    -------
    tuple(flatbuffers.builder.Builder, int)
        A tuple which contains two elements:

        - the :obj:`flatbuffers.builder.Builder` instance which has been used
          to serialize data
        - an :obj:`int` which is the address/offset of the serialized object
          in the builder buffer
    """
    if builder is None:
        builder = flatbuffers.Builder(0)

    opcode = SEAScope.API.OpCode.OpCode().getCatalogues
    SEAScope.API.Command.CommandStart(builder)
    SEAScope.API.Command.CommandAddOpcode(builder, opcode)
    cmd = SEAScope.API.Command.CommandEnd(builder)

    return builder, cmd


def deserialize(buf):
    """
    Deserialize the response that SEAScope sends after executing the
    getCatalogues command

    Parameters
    ----------
    buf : bytearray
        The buffer which contains the result of the getCatalogues command
        serialized with FlatBuffers

    Returns
    -------
    dict
        The sources and collections (including variables) currently available
        in SEAScope. The dict contains two keys:

        - ``sources`` : a :obj:`list` of :obj:`dict` generated by the
          :func:`SEAScope.types.source.deserialize` method

        - ``collections`` : o :obj:`list` of :obj:`dict` generated by the
          :func:`SEAScope.types.collection.deserialize` method
    """
    cls = SEAScope.API.GetCataloguesResponse.GetCataloguesResponse
    res = cls.GetRootAsGetCataloguesResponse(buf, 0)

    result = {'sources': [], 'collections': []}

    srcs_count = res.SourcesLength()
    for i in range(0, srcs_count):
        src = SEAScope.types.source.deserialize(res.Sources(i))
        result['sources'].append(src)

    collections_count = res.CollectionsLength()
    for i in range(0, collections_count):
        collection = SEAScope.types.collection.deserialize(res.Collections(i))
        result['collections'].append(collection)

    return result
