"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateClientProjects = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const languages_1 = require("../languages");
const generated_java_client_project_1 = require("./generated-java-client-project");
const generated_python_client_project_1 = require("./generated-python-client-project");
const generated_typescript_client_project_1 = require("./generated-typescript-client-project");
/**
 * Returns a generated client project for the given language
 */
const generateClientProject = (language, options) => {
    switch (language) {
        case languages_1.ClientLanguage.TYPESCRIPT:
            return new generated_typescript_client_project_1.GeneratedTypescriptClientProject({
                parent: options.parent,
                // Ensure kebab-case for typescript
                name: `${options.parentPackageName}-${languages_1.ClientLanguage.TYPESCRIPT}`.replace(/_/g, "-"),
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.TYPESCRIPT),
                specPath: options.parsedSpecPath,
                ...options.typescriptOptions,
            });
        case languages_1.ClientLanguage.PYTHON:
            // Ensure snake_case for python
            const moduleName = `${options.parentPackageName}_${languages_1.ClientLanguage.PYTHON}`
                .replace(/@/g, "")
                .replace(/[\-/]/g, "_");
            return new generated_python_client_project_1.GeneratedPythonClientProject({
                parent: options.parent,
                // Use dashes in project name since distributable's PKG-INFO always converts _ to -
                // https://stackoverflow.com/questions/36300788/python-package-wheel-pkg-info-name
                name: moduleName.replace(/_/g, "-"),
                moduleName,
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.PYTHON),
                specPath: options.parsedSpecPath,
                ...options.pythonOptions,
            });
        case languages_1.ClientLanguage.JAVA:
            // Ensure no dashes/underscores since name is used in package name
            const javaProjectName = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[\-/_]/g, "");
            const artifactId = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[/_]/g, "-");
            return new generated_java_client_project_1.GeneratedJavaClientProject({
                parent: options.parent,
                name: javaProjectName,
                artifactId,
                groupId: "com.generated.api",
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.JAVA),
                specPath: options.parsedSpecPath,
                ...options.javaOptions,
            });
        default:
            throw new Error(`Unknown client language ${language}`);
    }
};
/**
 * Generate API clients in the given languages
 * @param languages the languages to generate clients for
 * @param options options for the projects to be created
 */
exports.generateClientProjects = (languages, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Clients",
            "",
            "This directory contains generated client code based on your OpenAPI Specification file (spec.yaml).",
            "",
            "Like other `projen` managed files, this directory should be checked in to source control, but should not be edited manually.",
        ],
        readonly: true,
    });
    return Object.fromEntries([...languages].map((language) => [
        language,
        generateClientProject(language, options),
    ]));
};
//# sourceMappingURL=data:application/json;base64,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