#!/usr/bin/env python
# coding: utf-8

"""Matching raw log messages and its templates
that is generated by external tools."""

import re
from collections import defaultdict

# shortest match
REPLACER_REGEX_ESCAPED = re.compile(r"\\\*[A-Z]*?\\\*")


def add_esc_external(tpl):
    """Add escape sequence for imported external templates.
    It fails if the template has some statement that cannot
    be distinguished with variable replacers (e.g., *****).
    In that case, use option log_template_import.import_format_ext_esc
    and add escape sequences manually (or with another way).
    """
    from amulog import strutil
    from amulog import lt_common
    l_wild = lt_common.REPLACER_REGEX.findall(tpl)
    l_others = [strutil.add_esc(tmp)
                for tmp in lt_common.REPLACER_REGEX.split(tpl)]
    return "".join([other + wild for other, wild in zip(l_others, l_wild + [""])])


def generate_regex(tpl):
    d_name = defaultdict(list)

    def _replace_wildcard(matchobj):
        name = matchobj.group(0).strip("\\*")
        v = len(d_name[name])
        vname = "v" + name + str(v)
        d_name[name].append(vname)
        # shortest match
        regexstr = r"(?P<" + vname + r">[^*]*)"
        return regexstr

    regex_base = r"^" + re.escape(tpl) + r"$"
    tmp = REPLACER_REGEX_ESCAPED.sub(_replace_wildcard, regex_base, count=0)
    return re.compile(tmp)


def match_line(parsed_line, l_regex):
    for rid, regex in enumerate(l_regex):
        m = regex.match(parsed_line["message"])
        if m:
            return rid, m
    else:
        return None

