"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNameOfCloudFormationIntrinsic = exports.minimalCloudFormationJoin = exports.CLOUDFORMATION_TOKEN_RESOLVER = exports.CloudFormationLang = void 0;
const lazy_1 = require("../lazy");
const resolvable_1 = require("../resolvable");
const token_1 = require("../token");
const resolve_1 = require("./resolve");
/**
 * Routines that know how to do operations at the CloudFormation document language level
 */
class CloudFormationLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     * @param space Indentation to use (default: no pretty-printing)
     */
    static toJSON(obj, space) {
        return lazy_1.Lazy.uncachedString({
            // We used to do this by hooking into `JSON.stringify()` by adding in objects
            // with custom `toJSON()` functions, but it's ultimately simpler just to
            // reimplement the `stringify()` function from scratch.
            produce: (ctx) => tokenAwareStringify(obj, space !== null && space !== void 0 ? space : 0, ctx),
        });
    }
    /**
     * Produce a CloudFormation expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return '';
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 && typeof parts[0] === 'string' && typeof parts[1] === 'string') {
            return parts[0] + parts[1];
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return fnJoinConcat(parts);
    }
}
exports.CloudFormationLang = CloudFormationLang;
/**
 * Return a CFN intrinsic mass concatting any number of CloudFormation expressions
 */
function fnJoinConcat(parts) {
    return { 'Fn::Join': ['', minimalCloudFormationJoin('', parts)] };
}
/**
 * Perform a JSON.stringify()-like operation, except aware of Tokens and CloudFormation intrincics
 *
 * Tokens will be resolved and if any resolve to CloudFormation intrinsics, the intrinsics
 * will be lifted to the top of a giant `{ Fn::Join }` expression.
 *
 * If Tokens resolve to primitive types (for example, by using Lazies), we'll
 * use the primitive type to determine how to encode the value into the JSON.
 *
 * If Tokens resolve to CloudFormation intrinsics, we'll use the type of the encoded
 * value as a type hint to determine how to encode the value into the JSON. The difference
 * is that we add quotes (") around strings, and don't add anything around non-strings.
 *
 * The following structure:
 *
 *    { SomeAttr: resource.someAttr }
 *
 * Will JSONify to either:
 *
 *    '{ "SomeAttr": "' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ '" }'
 * or '{ "SomeAttr": ' ++ { Fn::GetAtt: [Resource, SomeAttr] } ++ ' }'
 *
 * Depending on whether `someAttr` is type-hinted to be a string or not.
 *
 * (Where ++ is the CloudFormation string-concat operation (`{ Fn::Join }`).
 *
 * -----------------------
 *
 * This work requires 2 features from the `resolve()` function:
 *
 * - INTRINSICS TYPE HINTS: intrinsics are represented by values like
 *   `{ Ref: 'XYZ' }`. These values can reference either a string or a list/number at
 *   deploy time, and from the value alone there's no way to know which. We need
 *   to know the type to know whether to JSONify this reference to:
 *
 *      '{ "referencedValue": "' ++ { Ref: XYZ } ++ '"}'
 *   or '{ "referencedValue": ' ++ { Ref: XYZ } ++ '}'
 *
 *   I.e., whether or not we need to enclose the reference in quotes or not.
 *
 *   We COULD have done this by resolving one token at a time, and looking at the
 *   type of the encoded token we were resolving to obtain a type hint. However,
 *   the `resolve()` and Token system resist a level-at-a-time resolve
 *   operation: because of the existence of post-processors, we must have done a
 *   complete recursive resolution of a token before we can look at its result
 *   (after which any type information about the sources of nested resolved
 *   values is lost).
 *
 *   To fix this, "type hints" have been added to the `resolve()` function,
 *   giving an idea of the type of the source value for compplex result values.
 *   This only works for objects (not strings and numbers) but fortunately
 *   we only care about the types of intrinsics, which are always complex values.
 *
 *   Type hinting could have been added to the `IResolvable` protocol as well,
 *   but for now we just use the type of an encoded value as a type hint. That way
 *   we don't need to annotate anything more at the L1 level--we will use the type
 *   encodings added by construct authors at the L2 levels. L1 users can escape the
 *   default decision of "string" by using `Token.asList()`.
 *
 * - COMPLEX KEYS: since tokens can be string-encoded, we can use string-encoded tokens
 *   as the keys in JavaScript objects. However, after resolution, those string-encoded
 *   tokens could resolve to intrinsics (`{ Ref: ... }`), which CANNOT be stored in
 *   JavaScript objects anymore.
 *
 *   We therefore need a protocol to store the resolved values somewhere in the JavaScript
 *   type model,  which can be returned by `resolve()`, and interpreted by `tokenAwareStringify()`
 *   to produce the correct JSON.
 *
 *   And example will quickly show the point:
 *
 *    User writes:
 *       { [resource.resourceName]: 'SomeValue' }
 *    ------ string actually looks like ------>
 *       { '${Token[1234]}': 'SomeValue' }
 *    ------ resolve ------->
 *       { '$IntrinsicKey$0': [ {Ref: Resource}, 'SomeValue' ] }
 *    ------ tokenAwareStringify ------->
 *       '{ "' ++ { Ref: Resource } ++ '": "SomeValue" }'
 */
function tokenAwareStringify(root, space, ctx) {
    let indent = 0;
    const ret = new Array();
    // First completely resolve the tree, then encode to JSON while respecting the type
    // hints we got for the resolved intrinsics.
    recurse(ctx.resolve(root, { allowIntrinsicKeys: true }));
    switch (ret.length) {
        case 0: return undefined;
        case 1: return renderSegment(ret[0]);
        default:
            return fnJoinConcat(ret.map(renderSegment));
    }
    /**
     * Stringify a JSON element
     */
    function recurse(obj) {
        if (obj === undefined) {
            return;
        }
        if (token_1.Token.isUnresolved(obj)) {
            throw new Error('This shouldnt happen anymore');
        }
        if (Array.isArray(obj)) {
            return renderCollection('[', ']', obj, recurse);
        }
        if (typeof obj === 'object' && obj != null && !(obj instanceof Date)) {
            // Treat as an intrinsic if this LOOKS like a CFN intrinsic (`{ Ref: ... }`)
            // AND it's the result of a token resolution. Otherwise, we just treat this
            // value as a regular old JSON object (that happens to look a lot like an intrinsic).
            if (isIntrinsic(obj) && resolve_1.resolvedTypeHint(obj)) {
                return renderIntrinsic(obj);
            }
            return renderCollection('{', '}', definedEntries(obj), ([key, value]) => {
                if (key.startsWith(resolve_1.INTRINSIC_KEY_PREFIX)) {
                    [key, value] = value;
                }
                recurse(key);
                pushLiteral(prettyPunctuation(':'));
                recurse(value);
            });
        }
        // Otherwise we have a scalar, defer to JSON.stringify()s serialization
        pushLiteral(JSON.stringify(obj));
    }
    /**
     * Render an object or list
     */
    function renderCollection(pre, post, xs, each) {
        pushLiteral(pre);
        indent += space;
        let atLeastOne = false;
        for (const [comma, item] of sepIter(xs)) {
            if (comma) {
                pushLiteral(',');
            }
            pushLineBreak();
            each(item);
            atLeastOne = true;
        }
        indent -= space;
        if (atLeastOne) {
            pushLineBreak();
        }
        pushLiteral(post);
    }
    function renderIntrinsic(intrinsic) {
        switch (resolve_1.resolvedTypeHint(intrinsic)) {
            case resolve_1.ResolutionTypeHint.STRING:
                pushLiteral('"');
                pushIntrinsic(deepQuoteStringLiterals(intrinsic));
                pushLiteral('"');
                break;
            default:
                pushIntrinsic(intrinsic);
                break;
        }
    }
    /**
     * Push a literal onto the current segment if it's also a literal, otherwise open a new Segment
     */
    function pushLiteral(lit) {
        let last = ret[ret.length - 1];
        if ((last === null || last === void 0 ? void 0 : last.type) !== 'literal') {
            last = { type: 'literal', parts: [] };
            ret.push(last);
        }
        last.parts.push(lit);
    }
    /**
     * Add a new intrinsic segment
     */
    function pushIntrinsic(intrinsic) {
        ret.push({ type: 'intrinsic', intrinsic });
    }
    /**
     * Push a line break if we are pretty-printing, otherwise don't
     */
    function pushLineBreak() {
        if (space > 0) {
            pushLiteral(`\n${' '.repeat(indent)}`);
        }
    }
    /**
     * Add a space after the punctuation if we are pretty-printing, no space if not
     */
    function prettyPunctuation(punc) {
        return space > 0 ? `${punc} ` : punc;
    }
}
/**
 * Render a segment
 */
function renderSegment(s) {
    switch (s.type) {
        case 'literal': return s.parts.join('');
        case 'intrinsic': return s.intrinsic;
    }
}
const CLOUDFORMATION_CONCAT = {
    join(left, right) {
        return CloudFormationLang.concat(left, right);
    },
};
/**
 * Default Token resolver for CloudFormation templates
 */
exports.CLOUDFORMATION_TOKEN_RESOLVER = new resolvable_1.DefaultTokenResolver(CLOUDFORMATION_CONCAT);
/**
 * Do an intelligent CloudFormation join on the given values, producing a minimal expression
 */
function minimalCloudFormationJoin(delimiter, values) {
    let i = 0;
    while (i < values.length) {
        const el = values[i];
        if (isSplicableFnJoinIntrinsic(el)) {
            values.splice(i, 1, ...el['Fn::Join'][1]);
        }
        else if (i > 0 && isPlainString(values[i - 1]) && isPlainString(values[i])) {
            values[i - 1] += delimiter + values[i];
            values.splice(i, 1);
        }
        else {
            i += 1;
        }
    }
    return values;
    function isPlainString(obj) {
        return typeof obj === 'string' && !token_1.Token.isUnresolved(obj);
    }
    function isSplicableFnJoinIntrinsic(obj) {
        if (!isIntrinsic(obj)) {
            return false;
        }
        if (Object.keys(obj)[0] !== 'Fn::Join') {
            return false;
        }
        const [delim, list] = obj['Fn::Join'];
        if (delim !== delimiter) {
            return false;
        }
        if (token_1.Token.isUnresolved(list)) {
            return false;
        }
        if (!Array.isArray(list)) {
            return false;
        }
        return true;
    }
}
exports.minimalCloudFormationJoin = minimalCloudFormationJoin;
/**
 * Return whether the given value represents a CloudFormation intrinsic
 */
function isIntrinsic(x) {
    if (Array.isArray(x) || x === null || typeof x !== 'object') {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || isNameOfCloudFormationIntrinsic(keys[0]);
}
function isNameOfCloudFormationIntrinsic(name) {
    if (!name.startsWith('Fn::')) {
        return false;
    }
    // these are 'fake' intrinsics, only usable inside the parameter overrides of a CFN CodePipeline Action
    return name !== 'Fn::GetArtifactAtt' && name !== 'Fn::GetParam';
}
exports.isNameOfCloudFormationIntrinsic = isNameOfCloudFormationIntrinsic;
/**
 * Separated iterator
 */
function* sepIter(xs) {
    let comma = false;
    for (const item of xs) {
        yield [comma, item];
        comma = true;
    }
}
/**
 * Object.entries() but skipping undefined values
 */
function* definedEntries(xs) {
    for (const [key, value] of Object.entries(xs)) {
        if (value !== undefined) {
            yield [key, value];
        }
    }
}
/**
 * Quote string literals inside an intrinsic
 *
 * Formally, this should only match string literals that will be interpreted as
 * string literals. Fortunately, the strings that should NOT be quoted are
 * Logical IDs and attribute names, which cannot contain quotes anyway. Hence,
 * we can get away not caring about the distinction and just quoting everything.
 */
function deepQuoteStringLiterals(x) {
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringLiterals);
    }
    if (typeof x === 'object' && x != null) {
        const ret = {};
        for (const [key, value] of Object.entries(x)) {
            ret[deepQuoteStringLiterals(key)] = deepQuoteStringLiterals(value);
        }
        return ret;
    }
    if (typeof x === 'string') {
        return quoteString(x);
    }
    return x;
}
/**
 * Quote the characters inside a string, for use inside toJSON
 */
function quoteString(s) {
    s = JSON.stringify(s);
    return s.substring(1, s.length - 1);
}
//# sourceMappingURL=data:application/json;base64,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