from typing import TYPE_CHECKING, List, Tuple, Type
import logging

import PySide2.QtWidgets
from PySide2.QtWidgets import QGraphicsSimpleTextItem
from PySide2.QtGui import QColor, QPen
from PySide2.QtCore import Qt, QRectF

from angr.analyses.proximity_graph import BaseProxiNode, FunctionProxiNode, CallProxiNode, StringProxiNode, \
    IntegerProxiNode, UnknownProxiNode

from ...config import Conf
from .qgraph_object import QCachedGraphicsItem

if TYPE_CHECKING:
    from angrmanagement.ui.views.proximity_view import ProximityView



_l = logging.getLogger(__name__)


class QProximityGraphBlock(QCachedGraphicsItem):

    HORIZONTAL_PADDING = 5
    VERTICAL_PADDING = 5
    LINE_MARGIN = 3

    #
    # Colors
    #

    FUNCTION_NODE_TEXT_COLOR = Qt.blue
    STRING_NODE_TEXT_COLOR = Qt.darkGreen
    INTEGER_NODE_TEXT_COLOR = Qt.black
    CALL_NODE_TEXT_COLOR = Qt.darkBlue
    CALL_NODE_TEXT_COLOR_PLT = Qt.darkMagenta
    CALL_NODE_TEXT_COLOR_SIMPROC = Qt.darkMagenta

    def __init__(self, is_selected, proximity_view: 'ProximityView', node: 'BaseProxiNode'):
        super().__init__()

        self._proximity_view = proximity_view
        self._workspace = self._proximity_view.workspace
        self._config = Conf

        self.selected = is_selected

        self._node = node

        self._init_widgets()
        self._update_size()

        self.setAcceptHoverEvents(True)

    def _init_widgets(self):
        raise NotImplementedError()

    def refresh(self):
        self._update_size()

    #
    # Event handlers
    #

    def mousePressEvent(self, event): #pylint: disable=no-self-use
        super().mousePressEvent(event)

    def mouseReleaseEvent(self, event):
        if event.button() == Qt.LeftButton:
            self.selected = not self.selected
            self._proximity_view.redraw_graph()
            event.accept()

        super().mouseReleaseEvent(event)

    def mouseDoubleClickEvent(self, event):
        # Jump to the reference address of the node
        if event.button() == Qt.LeftButton:
            if self._node.ref_at:
                self._workspace.viz(next(iter(self._node.ref_at)))
            event.accept()
            return

        super().mouseDoubleClickEvent(event)

    def hoverEnterEvent(self, event: PySide2.QtWidgets.QGraphicsSceneHoverEvent):
        self._proximity_view.hover_enter_block(self)

    def hoverLeaveEvent(self, event: PySide2.QtWidgets.QGraphicsSceneHoverEvent):
        self._proximity_view.hover_leave_block(self)

    def _paint_boundary(self, painter):
        painter.setFont(Conf.symexec_font)
        normal_background = QColor(0xfa, 0xfa, 0xfa)
        selected_background = QColor(0xcc, 0xcc, 0xcc)

        # The node background
        if self.selected:
            painter.setBrush(selected_background)
        else:
            painter.setBrush(normal_background)
        painter.setPen(QPen(QColor(0xf0, 0xf0, 0xf0), 1.5))
        painter.drawRect(0, 0, self.width, self.height)

    def paint(self, painter, option, widget): #pylint: disable=unused-argument
        """
        Paint a state block on the scene.

        :param painter:
        :return: None
        """

        self._paint_boundary(painter)

        x = 0
        y = 0

        x += self.HORIZONTAL_PADDING
        y += self.VERTICAL_PADDING

        # text
        text_label_x = x
        painter.setPen(Qt.gray)
        painter.drawText(text_label_x, y + self._config.symexec_font_ascent, "Unknown block")

    def _boundingRect(self):
        return QRectF(0, 0, self._width, self._height)

    #
    # Private methods
    #

    def _update_size(self):
        self._width = 100
        self._height = 50
        self.recalculate_size()


class QProximityGraphFunctionBlock(QProximityGraphBlock):

    def __init__(self, is_selected, proximity_view: 'ProximityView', node: FunctionProxiNode):
        self._text = None
        self._text_item: QGraphicsSimpleTextItem = None
        super().__init__(is_selected, proximity_view, node)

    def _init_widgets(self):
        self._text = "Function %s" % self._node.func.name
        self._text_item = QGraphicsSimpleTextItem(self._text, self)
        self._text_item.setFont(Conf.symexec_font)
        self._text_item.setBrush(self.FUNCTION_NODE_TEXT_COLOR)
        self._text_item.setPos(self.HORIZONTAL_PADDING, self.VERTICAL_PADDING)

    def mouseDoubleClickEvent(self, event):
        if event.button() == Qt.LeftButton and (event.modifiers() & Qt.ControlModifier) == Qt.ControlModifier:
            # ctrl + double click to collapse a function call
            event.accept()
            self._proximity_view.collapse_function(self._node.func)
            return

        super().mouseDoubleClickEvent(event)

    def paint(self, painter, option, widget):
        self._paint_boundary(painter)

    def _update_size(self):
        width_candidates = [
            self.HORIZONTAL_PADDING * 2 + self._text_item.boundingRect().width(),
        ]

        self._width = max(width_candidates)
        self._height = self.VERTICAL_PADDING * 2 + self._text_item.boundingRect().height()

        self._width = max(30, self._width)
        self._height = max(10, self._height)
        self.recalculate_size()


class QProximityGraphCallBlock(QProximityGraphBlock):

    def __init__(self, is_selected, proximity_view: 'ProximityView', node: CallProxiNode):
        self._func_name: str = None
        self._args: List[Tuple[Type,str]] = None

        self._func_name_item: QGraphicsSimpleTextItem = None
        self._left_parenthesis_item: QGraphicsSimpleTextItem = None
        self._args_list: List[QGraphicsSimpleTextItem] = None
        self._right_parenthesis_item: QGraphicsSimpleTextItem = None

        super().__init__(is_selected, proximity_view, node)

    def _init_widgets(self):
        self._node: CallProxiNode
        self._func_name = self._node.callee.name
        if self._node.args is not None:
            self._args = [ self._argument_text(arg) for arg in self._node.args ]
        else:
            self._args = [ ]

        # func name
        self._func_name_item = QGraphicsSimpleTextItem(self._func_name, self)
        if self._node.callee.is_simprocedure:
            pen_color = self.CALL_NODE_TEXT_COLOR_SIMPROC
        elif self._node.callee.is_plt:
            pen_color = self.CALL_NODE_TEXT_COLOR_SIMPROC
        else:
            pen_color = self.CALL_NODE_TEXT_COLOR
        self._func_name_item.setBrush(pen_color)
        self._func_name_item.setFont(Conf.symexec_font)
        self._func_name_item.setPos(self.HORIZONTAL_PADDING, self.VERTICAL_PADDING)

        x = self.HORIZONTAL_PADDING + self._func_name_item.boundingRect().width()
        y = self.VERTICAL_PADDING
        # left parenthesis
        self._left_parenthesis_item = QGraphicsSimpleTextItem("(", self)
        self._left_parenthesis_item.setBrush(pen_color)
        self._left_parenthesis_item.setFont(Conf.symexec_font)
        self._left_parenthesis_item.setPos(x, y)

        x += self._left_parenthesis_item.boundingRect().width()

        # arguments
        self._args_list = [ ]
        for i, (type_, arg) in enumerate(self._args):
            if type_ is str:
                color = self.STRING_NODE_TEXT_COLOR
            elif type_ is int:
                color = self.INTEGER_NODE_TEXT_COLOR
            else:
                color = self.CALL_NODE_TEXT_COLOR
            o = QGraphicsSimpleTextItem(arg, self)
            o.setBrush(color)
            o.setFont(Conf.symexec_font)
            o.setPos(x, y)
            self._args_list.append(o)
            x += o.boundingRect().width()

            # comma
            if i != len(self._args) - 1:
                comma = QGraphicsSimpleTextItem(", ", self)
                comma.setBrush(pen_color)
                comma.setFont(Conf.symexec_font)
                comma.setPos(x, y)
                self._args_list.append(comma)
                x += comma.boundingRect().width()

        # right parenthesis
        self._right_parenthesis_item = QGraphicsSimpleTextItem(")", self)
        self._right_parenthesis_item.setBrush(pen_color)
        self._right_parenthesis_item.setFont(Conf.symexec_font)
        self._right_parenthesis_item.setPos(x, y)

    def _argument_text(self, arg) -> Tuple[Type,str]:
        if isinstance(arg, StringProxiNode):
            return str, '"' + arg.content.decode("utf-8") + '"'
        elif isinstance(arg, IntegerProxiNode):
            return int, str(arg.value)
        elif isinstance(arg, UnknownProxiNode):
            return object, str(arg.dummy_value)
        return object, "Unknown"

    def mouseDoubleClickEvent(self, event):
        if event.button() == Qt.LeftButton and (event.modifiers() & Qt.ControlModifier) == Qt.ControlModifier:
            # ctrl + double click to expand a function call
            event.accept()
            self._proximity_view.expand_function(self._node.callee)
            return

        super().mouseDoubleClickEvent(event)

    def paint(self, painter, option, widget):
        self._paint_boundary(painter)

    def _update_size(self):
        width_candidates = [
            self.HORIZONTAL_PADDING * 2 +
            self._func_name_item.boundingRect().width() +
            self._left_parenthesis_item.boundingRect().width() +
            sum(map(lambda x: x.boundingRect().width(), self._args_list)) +
            self._right_parenthesis_item.boundingRect().width()
        ]

        self._width = max(width_candidates)
        self._height = self.VERTICAL_PADDING * 2 + self._func_name_item.boundingRect().height()

        self._width = max(30, self._width)
        self._height = max(10, self._height)
        self.recalculate_size()


class QProximityGraphStringBlock(QProximityGraphBlock):

    def __init__(self, is_selected, proximity_view: 'ProximityView', node: StringProxiNode):
        self._text = None
        self._text_item: QGraphicsSimpleTextItem = None
        super().__init__(is_selected, proximity_view, node)

    def _init_widgets(self):
        self._text = '"' + self._node.content.decode("utf-8") + '"'

        self._text_item = QGraphicsSimpleTextItem(self._text, self)
        self._text_item.setBrush(self.STRING_NODE_TEXT_COLOR)
        self._text_item.setFont(Conf.symexec_font)
        self._text_item.setPos(self.HORIZONTAL_PADDING, self.VERTICAL_PADDING)

    def paint(self, painter, option, widget):
        self._paint_boundary(painter)

    def _update_size(self):
        width_candidates = [
            self.HORIZONTAL_PADDING * 2 + self._text_item.boundingRect().width(),
        ]

        self._width = max(width_candidates)
        self._height = self.VERTICAL_PADDING * 2 + self._text_item.boundingRect().height()

        self._width = max(30, self._width)
        self._height = max(10, self._height)
        self.recalculate_size()
