# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/API_User.ipynb (unless otherwise specified).

__all__ = ["User"]

# Cell

from typing import *

# Internal Cell

import os

import pandas as pd
from fastcore.foundation import patch

from ..logger import get_logger, set_level
from ..helper import get_data, post_data, delete_data, generate_df

from .client import Client

# Internal Cell

logger = get_logger(__name__)

# Cell


class User:
    """A class to encapsulate all the user management routes

    All the user management routes are only accessable to super users. If a non-super user tried to access
    the user management routes they will get a permission error.
    """

    @staticmethod
    def create(
        *,
        username: str,
        first_name: str,
        last_name: str,
        email: str,
        password: str,
        super_user: bool = False,
        quota: bool = False,
    ) -> pd.DataFrame:
        """A method to create a new user in airt service

        Args:
            username: The username for the new user. If the passed username is already available in airt service, then an error message
                will be diplayed indicating the same.
            first_name: The first name for the new user.
            last_name: The last name for the new user.
            email: The email for the new user. If the passed email is already available in airt service, then an error message
                will be diplayed indicating the same.
            password: The password for the new user.
            super_user: If set to **True**, then the new user will have super user privilages. If **None**, then the default value **False** will be used.
            quota: If set to True, then the user will have access to all the routes. If **None**, then the default value **False** will be used and the user
                will have access only to `DataSource` routes.

        Returns:
                A pandas dataframe with the details of the newly created user

        Raises:
            ConnectionError: If the server address is invalid or not reachable.
            ValueError: If the username or email is already present in the airt service.

        An minimal example for calling the `create` method:

        ```python
            new_user = User.create(
                username="example_username",
                first_name="example_first_name",
                last_name="example_last_name",
                email="example@example.com",
                password="example_password",
            )
        ```
        """
        req_data = dict(
            username=username,
            first_name=first_name,
            last_name=last_name,
            email=email,
            quota=quota,
            super_user=super_user,
            password=password,
        )

        response = Client.post_data(relative_url=f"/user/", data=req_data)

        columns = User._get_columns()

        return pd.DataFrame(response, index=[0])[columns]

    @classmethod
    def _get_columns(cls) -> list:
        """Returns the list of columns to be added to the DataFrame."""

        return [
            "id",
            "username",
            "super_user",
            "disabled",
            "created",
            "quota",
            "first_name",
            "last_name",
            "email",
        ]

    @staticmethod
    def ls(
        offset: int = 0,
        limit: int = 100,
        disabled: bool = False,
    ) -> pd.DataFrame:
        """Display the list of available users.

        Args:
            offset: The number of rows to offset at the beginning of the users
                list from the server.If **None**, then the default value **0** will be used.
            limit: The maximum number of rows to return from the server. If **None**,
                then the default value **100** will be used.
            disabled: If set to **True**, then both active and disabled users will be displayed. If **None**,
                then the default value **False** will be used which will display only the list of active users.

        Returns:
            A pandas dataframe with the list of available users.

        Raises:
            ConnectionError: If the server address is invalid or not reachable.

        An example to list the available users:

        ```python
        User.ls()
        ```
        """
        users = Client.get_data(
            relative_url=f"/user/?disabled={disabled}&offset={offset}&limit={limit}"
        )

        columns = User._get_columns()

        return generate_df(users, columns)

    @staticmethod
    def disable(user_id: int) -> pd.DataFrame:
        """Disable a user in airt service.

        Args:
            user_id: The user id to disabled in airt service.

        Returns:
            A pandas dataframe with the details of the disabled user.

        Raises:
            ConnectionError: If the server address is invalid or not reachable.

        An example to disable a user (id=1) in airt service:

        ```python
        User.disable(user_id=1)
        ```
        """

        response = Client.delete_data(relative_url=f"/user/{user_id}")

        columns = User._get_columns()

        return pd.DataFrame(response, index=[0])[columns]

    @staticmethod
    def enable(user_id: int) -> pd.DataFrame:
        """Enable a disabled user in airt service.

        Args:
            user_id: The id of the disabled user in airt service.

        Returns:
            A pandas dataframe with the details of the enabled user.

        Raises:
            ConnectionError: If the server address is invalid or not reachable.

        An example to enable a disabled user (id=1) in airt service:

        ```python
        User.enable(user_id=1)
        ```
        """

        response = Client.get_data(relative_url=f"/user/{user_id}/enable")

        columns = User._get_columns()

        return pd.DataFrame(response, index=[0])[columns]

    @staticmethod
    def update(
        user_id: int,
        username: Optional[str] = None,
        first_name: Optional[str] = None,
        last_name: Optional[str] = None,
        email: Optional[str] = None,
        password: Optional[str] = None,
    ) -> pd.DataFrame:
        """Update user details in airt service.

        Args:
            user_id: The id of the user in airt service.
            username: New **username** for the user_id in airt service. If **None** (default value), then the username won't be updated for the user.
            first_name: New **first_name** for the user_id in airt service. If **None** (default value), then the first_name won't be updated for the user.
            last_name: New **last_name** for the user_id in airt service. If **None** (default value), then the last_name won't be updated for the user.
            email: New **email** for the user_id in airt service. If **None** (default value), then the email won't be updated for the user.
            password: New **password** for the user_id in airt service. If **None** (default value), then the password won't be updated for the user.

        Returns:
            A pandas dataframe with the updated user details.

        Raises:
            ConnectionError: If the server address is invalid or not reachable.

        An example to update the username for a user (id=1) in airt service:

        ```python
        User.update(user_id=1, username="new_username")
        ```
        """

        body = dict(
            username=username,
            first_name=first_name,
            last_name=last_name,
            email=email,
            password=password,
        )

        response = Client.post_data(relative_url=f"/user/{user_id}/update", data=body)

        columns = User._get_columns()

        return pd.DataFrame(response, index=[0])[columns]
