# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['miom']

package_data = \
{'': ['*']}

install_requires = \
['PICOS>=2.2.52,<3.0.0',
 'mip>=1.13.0,<2.0.0',
 'numpy>=1.19.2,<2.0.0',
 'swiglpk>=5.0.3,<6.0.0']

setup_kwargs = {
    'name': 'miom',
    'version': '0.9.0a4',
    'description': 'Mixed Integer Optimization for Metabolism',
    'long_description': '# Introduction\n[![Run In Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/drive/1JAOEHLlRCW8GziIpBqkFwJL2ha3OEOWJ?usp=sharing)\n[![PyPI version](https://badge.fury.io/py/miom.svg)](https://badge.fury.io/py/miom)\n![Tests](https://github.com/metexplore/miom/actions/workflows/unit-tests.yml/badge.svg)\n\nhttps://metexplore.github.io/miom\n\n__MIOM__ (Mixed Integer Optimization for Metabolism) is a python library for creating and solving complex optimization problems using genome-scale metabolic networks, in just a few lines. \n\nMIOM offers a high-level API that leverages the power of modern Mixed Integer Optimization (MIO) solvers to easily define steady-state metabolic optimization problems, from simple Flux Balance Analysis (FBA) simulations, to more complex problems, such as sparse FBA or context-specific reconstruction problems, and solve them the __required level of optimality__.\n\nMost of the time, algorithms formulated as Mixed Integer Optimization problems with MIOM can be solved faster and with better quality than currently existing alternatives that are approximations of the original problem. By using the MIO formulation, you can get also an estimation of how close to optimality a solution is, so you don\'t need to waste more time than needed.\n\nMIOM uses the [PICOS](https://picos-api.gitlab.io/picos/) and the [Python-MIP](https://www.python-mip.com/) libraries to build and solve the optimization problems using many commercial, academic and free solvers. It is also compatible and complementary to [cobrapy](https://opencobra.github.io/cobrapy/).\n\n> NOTE: This library is functional but still in a very early stage. API is still not stable and might be changed in future versions.\n\n## Installation\n\nBy default, MIOM comes with support for COIN-OR CBC solver and GLPK using the swiglpk wrapper. To install MIOM with minimal dependencies, run:\n\n```\npip install miom\n```\n\nYou can also install it with the following command to include the interfaces for [Gurobi](https://www.gurobi.com/downloads) and [Mosek](https://www.mosek.com/downloads/):\n\n```\npip install miom[all]\n```\n\nCPLEX is also supported, but requires a license. To install MIOM with CPLEX support, follow the instructions on the [CPLEX page](https://www.ibm.com/docs/en/icos/12.8.0.0?topic=cplex-setting-up-python-api).\n\n\n## A quick example\n\nHere is an example of how to load a metabolic network and maximize the flux through a target reaction using FBA, and then how to modify the original problem to implement the sparse FBA problem adding only a few lines to the original problem:\n\n```python\nfrom miom import miom, load_gem, Solvers\n\n# Load a genome-scale metabolic network using the miom format. \n# You can load SMBL or Matlab metabolic networks as well using \n# the same method, but it requires to have the cobratoolbox python library\n# installed (and scipy for mat files). To install these dependencies, run:\n# $ pip install cobra scipy\nnetwork = load_gem("https://github.com/pablormier/miom-gems/raw/main/gems/mus_musculus_iMM1865.miom")\ntarget_rxn = "BIOMASS_reaction"\n\n# Create the optimization problem with miom and solve\nmodel = (miom(network)\n        .steady_state()\n        .set_rxn_objective(target_rxn)\n        .solve(verbosity=1))\n\nprint("Optimal flux:", model.get_fluxes(target_rxn), "mmol/(h·gDW)")\n\n# Show reactions with non-zero flux\nV, _ = model.get_values()\nprint("Number of reactions active reactions:", sum(abs(V) > 1e-8))\n```\n\n```\nOptimal flux: 798.8110517749975 mmol/(h·gDW)\nNumber of active reactions: 2549\n```\n\nNow, modify the original problem to solve the sparse FBA problem, minimizing the number of reactions with non-zero flux that can lead to the optimal possible flux through the target reaction. This can be easily done by transforming the FBA problem into a subset selection problem, where each reaction has a negative weight and the goal is to remove as many negative weighted reactions as possible. Note that since each reaction has the same weight (-1), all reactions are equally important in the optimization problem:\n\n> Note: To better understand the meaning of each step, please read the documentation of the [BaseModel class](https://metexplore.github.io/miom/references/miom/#miom.miom.BaseModel), and the complete example in [examples/sparse_fba.py](https://metexplore.github.io/miom/examples/sparse_fba).    \n   \n\n```python\nV, X = (model\n        # Set the MIP Gap tolerance to 5%, using the default solver\n        # (COIN-OR CBC included with the Python-MIP lib)\n        .setup(opt_tol=0.05, verbosity=1)\n        # Set the fluxes to the optimal value found\n        .set_fluxes_for(\'BIOMASS_reaction\')\n        # Transform to a subset selection problem\n        # adding a negative weight to all reactions\n        # (to remove them from the final solution)\n        .subset_selection(-1)\n        # Solve the MIO problem\n        .solve()\n        # Get continuos vars (fluxes) and binary vars\n        .get_values())\n\n# Show reactions with non-zero flux\nprint("Number of active reactions:", sum(abs(V) > 1e-8))\n```\n\n```\nNumber of active reactions: 404\n```\n\nSolving this problem with default COIN-OR CBC solver returns a solution with 404 active reactions (much less than the 2549 reactions obtained with FBA, and less than the 433 reactions returned by the CappedL1 approximation in the [sparseFBA](https://opencobra.github.io/cobratoolbox/stable/modules/analysis/sparseFBA/index.html) implementation in Matlab), with a relative gap between the lower and upper objective bound below 5% (as indicated in the setup method):\n\n```\nCbc0011I Exiting as integer gap of 122.04538 less than 1e-10 or 5%\nCbc0001I Search completed - best objective -10208, took 0 iterations and 0 nodes (28.34 seconds)\nCbc0035I Maximum depth 0, 0 variables fixed on reduced cost\nTotal time (CPU seconds):       60.79   (Wallclock seconds):       60.79\n```\n\nThe concise API provided by MIOM makes everything explicit: the sparse FBA problem can be implemented as a best subset selection problem of reactions (minimize the number of reactions with non-zero flux)\nsubject to the steady-state constraint and the optimality constraint of the flux for the target\nreaction (in this case the `BIOMASS_reaction`). Using this formulation, you can take advantage of\nmodern solvers like CPLEX, GUROBI, MOSEK, COIN-OR CBC (among others) to obtain an optimal or an\napproximate solution, controlled by the `opt_tol` parameter.\n\nTo use other solvers, you only need to provide the specific solver to the `miom` method, for example:\n\n```python\nmodel = (miom(network, solver=Solvers.GLPK)\n        .steady_state()\n        .set_rxn_objective(target_rxn)\n        .solve(verbosity=1))\n```\n\n## Advantages\n\n* __It\'s easy to use:__ MIOM uses the [PICOS](https://picos-api.gitlab.io/picos/) and the [Python-MIP](https://www.python-mip.com/) libraries, which means you can use any solver supported by those libraries.\n* __It\'s easy to extend:__ MIOM is written in pure python, so you can easily extend it to solve more complex optimization problems.\n* __It makes the problem explicit:__ MIOM uses a declarative way to express the problem, so you can easily read and understand what you are solving and differences between algorithms.\n* __It\'s fast:__ MIOM leverages the power of MIO solvers to solve complex optimization problems. You can control the quality and speed of the solutions for your problem and get better solutions that the approximations (LP) of the original problem available in other constraint-based modeling libraries.\n* __It\'s lightweight:__ The library has a small number of dependencies, so it\'s easy to install and distribute also in HPC environments.\n* __It includes a compressed GEM format__: MIOM can load and save the minimal information of the metabolic networks required for performing simulations into a compressed file compatible with numpy. The small size of the files allows you to quickly run online experiments so other people can reproduce your results. It also supports SBML and matlab formats if `cobratoolbox` is installed.\n* __It\'s open-source:__ MIOM is open-source and free to use. You can contribute to the development of MIOM by forking the repository and sending pull requests.\n\n## Documentation\n\nThe documentation of the library can be found at https://metexplore.github.io/miom/\n\n## How to cite\n\nManuscript in progress\n\n## License\n\nGNU General Public License v3.0\n',
    'author': 'Pablo R. Mier',
    'author_email': 'pablo.rodriguez.mier@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/MetExplore/miom/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
