"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service synthetics
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchsynthetics.html
 */
class Synthetics extends shared_1.PolicyStatement {
    /**
     * Action provider for service synthetics
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchsynthetics.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'synthetics';
        this.actions = {
            "CreateCanary": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CreateCanary.html",
                "description": "Create a canary.",
                "accessLevel": "Write"
            },
            "DeleteCanary": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DeleteCanary.html",
                "description": "Deletes a canary. Amazon Synthetics deletes all the resources except for the Lambda function and the CloudWatch Alarms if you created one.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "canary": {
                        "required": true
                    }
                }
            },
            "DescribeCanaries": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanaries.html",
                "description": "Returns information of all canaries.",
                "accessLevel": "Read"
            },
            "DescribeCanariesLastRun": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanariesLastRun.html",
                "description": "Returns information about the last test run associated with all canaries.",
                "accessLevel": "Read"
            },
            "GetCanaryRuns": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_GetCanaryRuns.html",
                "description": "Returns information about all the test runs associated with a canary.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "canary": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_ListTagsForResource.html",
                "description": "Returns a list of all tags and values associated with a canary.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "canary": {
                        "required": false
                    }
                }
            },
            "StartCanary": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StartCanary.html",
                "description": "Starts a canary, so that Amazon Synthetics starts monitoring a website.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "canary": {
                        "required": true
                    }
                }
            },
            "StopCanary": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StopCanary.html",
                "description": "Stops a canary.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "canary": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_TagResource.html",
                "description": "Adds one or more tags to a canary.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "canary": {
                        "required": false
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UntagResource.html",
                "description": "Removes one or more tags from a canary.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "canary": {
                        "required": false
                    }
                }
            },
            "UpdateCanary": {
                "url": "https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UpdateCanary.html",
                "description": "Updates a canary.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "canary": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "canary": {
                "name": "canary",
                "url": "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries.html",
                "arn": "arn:${Partition}:synthetics::${Account}:canary:${CanaryName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Create a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CreateCanary.html
     */
    createCanary() {
        this.add('synthetics:CreateCanary');
        return this;
    }
    /**
     * Deletes a canary. Amazon Synthetics deletes all the resources except for the Lambda function and the CloudWatch Alarms if you created one.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DeleteCanary.html
     */
    deleteCanary() {
        this.add('synthetics:DeleteCanary');
        return this;
    }
    /**
     * Returns information of all canaries.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanaries.html
     */
    describeCanaries() {
        this.add('synthetics:DescribeCanaries');
        return this;
    }
    /**
     * Returns information about the last test run associated with all canaries.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanariesLastRun.html
     */
    describeCanariesLastRun() {
        this.add('synthetics:DescribeCanariesLastRun');
        return this;
    }
    /**
     * Returns information about all the test runs associated with a canary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_GetCanaryRuns.html
     */
    getCanaryRuns() {
        this.add('synthetics:GetCanaryRuns');
        return this;
    }
    /**
     * Returns a list of all tags and values associated with a canary.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('synthetics:ListTagsForResource');
        return this;
    }
    /**
     * Starts a canary, so that Amazon Synthetics starts monitoring a website.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StartCanary.html
     */
    startCanary() {
        this.add('synthetics:StartCanary');
        return this;
    }
    /**
     * Stops a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StopCanary.html
     */
    stopCanary() {
        this.add('synthetics:StopCanary');
        return this;
    }
    /**
     * Adds one or more tags to a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('synthetics:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('synthetics:UntagResource');
        return this;
    }
    /**
     * Updates a canary.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UpdateCanary.html
     */
    updateCanary() {
        this.add('synthetics:UpdateCanary');
        return this;
    }
    /**
     * Adds a resource of type canary to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries.html
     *
     * @param canaryName - Identifier for the canaryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCanary(canaryName, account, partition) {
        var arn = 'arn:${Partition}:synthetics::${Account}:canary:${CanaryName}';
        arn = arn.replace('${CanaryName}', canaryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Synthetics = Synthetics;
//# sourceMappingURL=data:application/json;base64,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