"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service codeguru-reviewer
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodegurureviewer.html
 */
class CodeguruReviewer extends shared_1.PolicyStatement {
    /**
     * Action provider for service codeguru-reviewer
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodegurureviewer.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codeguru-reviewer';
        this.actions = {
            "AssociateRepository": {
                "url": "",
                "description": "Grants permission to associates a repository with Amazon CodeGuru Reviewer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": false
                    }
                }
            },
            "CreateConnectionToken": {
                "url": "",
                "description": "Grants permission to perform webbased oauth handshake for 3rd party providers.",
                "accessLevel": "Read"
            },
            "DescribeCodeReview": {
                "url": "",
                "description": "Grants permission to describe a code review.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "codereview": {
                        "required": true
                    }
                }
            },
            "DescribeRecommendationFeedback": {
                "url": "",
                "description": "Grants permission to describe a recommendation feedback on a code review.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "codereview": {
                        "required": true
                    }
                }
            },
            "DescribeRepositoryAssociation": {
                "url": "",
                "description": "Grants permission to describe a repository association.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "association": {
                        "required": true
                    }
                }
            },
            "DisassociateRepository": {
                "url": "",
                "description": "Grants permission to disassociate a repository with Amazon CodeGuru Reviewer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "association": {
                        "required": true
                    }
                }
            },
            "GetMetricsData": {
                "url": "",
                "description": "Grants permission to view pull request metrics in console.",
                "accessLevel": "Read"
            },
            "ListCodeReviews": {
                "url": "",
                "description": "Grants permission to list summary of code reviews.",
                "accessLevel": "List"
            },
            "ListRecommendationFeedback": {
                "url": "",
                "description": "Grants permission to list summary of recommendation feedback on a code review.",
                "accessLevel": "List",
                "resourceTypes": {
                    "codereview": {
                        "required": true
                    }
                }
            },
            "ListRecommendations": {
                "url": "",
                "description": "Grants permission to list summary of recommendations on a code review",
                "accessLevel": "List",
                "resourceTypes": {
                    "codereview": {
                        "required": true
                    }
                }
            },
            "ListRepositoryAssociations": {
                "url": "",
                "description": "Grants permission to list summary of repository associations.",
                "accessLevel": "List"
            },
            "ListThirdPartyRepositories": {
                "url": "",
                "description": "Grants permission to list 3rd party providers repositories in console.",
                "accessLevel": "Read"
            },
            "PutRecommendationFeedback": {
                "url": "",
                "description": "Grants permission to put feedback for a recommendation on a code review.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "codereview": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "association": {
                "name": "association",
                "url": "",
                "arn": "arn:${Partition}:codeguru-reviewer::${Account}:association:${ResourceId}",
                "conditionKeys": []
            },
            "codereview": {
                "name": "codereview",
                "url": "",
                "arn": "arn:${Partition}:codeguru-reviewer:${Region}:${Account}:code-review:${CodeReviewUuid}",
                "conditionKeys": []
            },
            "repository": {
                "name": "repository",
                "url": "https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associates a repository with Amazon CodeGuru Reviewer.
     *
     * Access Level: Write
     */
    associateRepository() {
        this.add('codeguru-reviewer:AssociateRepository');
        return this;
    }
    /**
     * Grants permission to perform webbased oauth handshake for 3rd party providers.
     *
     * Access Level: Read
     */
    createConnectionToken() {
        this.add('codeguru-reviewer:CreateConnectionToken');
        return this;
    }
    /**
     * Grants permission to describe a code review.
     *
     * Access Level: Read
     */
    describeCodeReview() {
        this.add('codeguru-reviewer:DescribeCodeReview');
        return this;
    }
    /**
     * Grants permission to describe a recommendation feedback on a code review.
     *
     * Access Level: Read
     */
    describeRecommendationFeedback() {
        this.add('codeguru-reviewer:DescribeRecommendationFeedback');
        return this;
    }
    /**
     * Grants permission to describe a repository association.
     *
     * Access Level: Read
     */
    describeRepositoryAssociation() {
        this.add('codeguru-reviewer:DescribeRepositoryAssociation');
        return this;
    }
    /**
     * Grants permission to disassociate a repository with Amazon CodeGuru Reviewer.
     *
     * Access Level: Write
     */
    disassociateRepository() {
        this.add('codeguru-reviewer:DisassociateRepository');
        return this;
    }
    /**
     * Grants permission to view pull request metrics in console.
     *
     * Access Level: Read
     */
    getMetricsData() {
        this.add('codeguru-reviewer:GetMetricsData');
        return this;
    }
    /**
     * Grants permission to list summary of code reviews.
     *
     * Access Level: List
     */
    listCodeReviews() {
        this.add('codeguru-reviewer:ListCodeReviews');
        return this;
    }
    /**
     * Grants permission to list summary of recommendation feedback on a code review.
     *
     * Access Level: List
     */
    listRecommendationFeedback() {
        this.add('codeguru-reviewer:ListRecommendationFeedback');
        return this;
    }
    /**
     * Grants permission to list summary of recommendations on a code review
     *
     * Access Level: List
     */
    listRecommendations() {
        this.add('codeguru-reviewer:ListRecommendations');
        return this;
    }
    /**
     * Grants permission to list summary of repository associations.
     *
     * Access Level: List
     */
    listRepositoryAssociations() {
        this.add('codeguru-reviewer:ListRepositoryAssociations');
        return this;
    }
    /**
     * Grants permission to list 3rd party providers repositories in console.
     *
     * Access Level: Read
     */
    listThirdPartyRepositories() {
        this.add('codeguru-reviewer:ListThirdPartyRepositories');
        return this;
    }
    /**
     * Grants permission to put feedback for a recommendation on a code review.
     *
     * Access Level: Write
     */
    putRecommendationFeedback() {
        this.add('codeguru-reviewer:PutRecommendationFeedback');
        return this;
    }
    /**
     * Adds a resource of type association to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAssociation(resourceId, account, partition) {
        var arn = 'arn:${Partition}:codeguru-reviewer::${Account}:association:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type codereview to the statement
     *
     * @param codeReviewUuid - Identifier for the codeReviewUuid.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCodereview(codeReviewUuid, account, region, partition) {
        var arn = 'arn:${Partition}:codeguru-reviewer:${Region}:${Account}:code-review:${CodeReviewUuid}';
        arn = arn.replace('${CodeReviewUuid}', codeReviewUuid);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRepository(repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}';
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CodeguruReviewer = CodeguruReviewer;
//# sourceMappingURL=data:application/json;base64,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