"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service comprehend
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html
 */
class Comprehend extends shared_1.PolicyStatement {
    /**
     * Action provider for service comprehend
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'comprehend';
        this.actions = {
            "BatchDetectDominantLanguage": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectDominantLanguage.html",
                "description": "Detects the language or languages present in the list of text documents.",
                "accessLevel": "Read"
            },
            "BatchDetectEntities": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectEntities.html",
                "description": "Detects the named entities (\"People\", \"Places\", \"Locations\", etc) within the given list of text documents.",
                "accessLevel": "Read"
            },
            "BatchDetectKeyPhrases": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectKeyPhrases.html",
                "description": "Detects the phrases in the list of text documents that are most indicative of the content.",
                "accessLevel": "Read"
            },
            "BatchDetectSentiment": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSentiment.html",
                "description": "Detects the sentiment of a text in the list of documents (Positive, Negative, Neutral, or Mixed).",
                "accessLevel": "Read"
            },
            "BatchDetectSyntax": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSyntax.html",
                "description": "Detects syntactic information (like Part of Speech, Tokens) in a list of text documents.",
                "accessLevel": "Read"
            },
            "ClassifyDocument": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ClassifyDocument.html",
                "description": "Creates a new document classification request to analyze a single document in real-time, using a previously created and trained custom model and an endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    }
                }
            },
            "CreateDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateDocumentClassifier.html",
                "description": "Creates a new document classifier that you can use to categorize documents.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEndpoint.html",
                "description": "Creates a model-specific endpoint for synchronous inference for a previously trained custom model",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEntityRecognizer.html",
                "description": "Creates an entity recognizer using submitted files.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteDocumentClassifier.html",
                "description": "Deletes a previously created document classifier.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEndpoint.html",
                "description": "Deletes a model-specific endpoint for a previously-trained custom model. All endpoints must be deleted in order for the model to be deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEntityRecognizer.html",
                "description": "Deletes a submitted entity recognizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "entity-recognizer": {
                        "required": true
                    }
                }
            },
            "DescribeDocumentClassificationJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassificationJob.html",
                "description": "Gets the properties associated with a document classification job.",
                "accessLevel": "Read"
            },
            "DescribeDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassifier.html",
                "description": "Gets the properties associated with a document classifier.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "DescribeDominantLanguageDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDominantLanguageDetectionJob.html",
                "description": "Gets the properties associated with a dominant language detection job.",
                "accessLevel": "Read"
            },
            "DescribeEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEndpoint.html",
                "description": "Gets the properties associated with a specific endpoint. Use this operation to get the status of an endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeEntitiesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntitiesDetectionJob.html",
                "description": "Gets the properties associated with an entities detection job.",
                "accessLevel": "Read"
            },
            "DescribeEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntityRecognizer.html",
                "description": "Provides details about an entity recognizer including status, S3 buckets containing training data, recognizer metadata, metrics, and so on.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "entity-recognizer": {
                        "required": true
                    }
                }
            },
            "DescribeKeyPhrasesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeKeyPhrasesDetectionJob.html",
                "description": "Gets the properties associated with a key phrases detection job.",
                "accessLevel": "Read"
            },
            "DescribeSentimentDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeSentimentDetectionJob.html",
                "description": "Gets the properties associated with a sentiment detection job.",
                "accessLevel": "Read"
            },
            "DescribeTopicsDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeTopicsDetectionJob.html",
                "description": "Gets the properties associated with a topic detection job.",
                "accessLevel": "Read"
            },
            "DetectDominantLanguage": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html",
                "description": "Detects the language or languages present in the text.",
                "accessLevel": "Read"
            },
            "DetectEntities": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectEntities.html",
                "description": "Detects the named entities (\"People\", \"Places\", \"Locations\", etc) within the given text document.",
                "accessLevel": "Read"
            },
            "DetectKeyPhrases": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectKeyPhrases.html",
                "description": "Detects the phrases in the text that are most indicative of the content.",
                "accessLevel": "Read"
            },
            "DetectSentiment": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSentiment.html",
                "description": "Detects the sentiment of a text in a document (Positive, Negative, Neutral, or Mixed).",
                "accessLevel": "Read"
            },
            "DetectSyntax": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSyntax.html",
                "description": "Detects syntactic information (like Part of Speech, Tokens) in a text document.",
                "accessLevel": "Read"
            },
            "ListDocumentClassificationJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassificationJobs.html",
                "description": "Gets a list of the document classification jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListDocumentClassifiers": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassifiers.html",
                "description": "Gets a list of the document classifiers that you have created.",
                "accessLevel": "List"
            },
            "ListDominantLanguageDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDominantLanguageDetectionJobs.html",
                "description": "Gets a list of the dominant language detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListEndpoints": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEndpoints.html",
                "description": "Gets a list of all existing endpoints that you've created.",
                "accessLevel": "List"
            },
            "ListEntitiesDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntitiesDetectionJobs.html",
                "description": "Gets a list of the entity detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListEntityRecognizers": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntityRecognizers.html",
                "description": "Gets a list of the properties of all entity recognizers that you created, including recognizers currently in training.",
                "accessLevel": "List"
            },
            "ListKeyPhrasesDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListKeyPhrasesDetectionJobs.html",
                "description": "Get a list of key phrase detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListSentimentDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListSentimentDetectionJobs.html",
                "description": "Gets a list of sentiment detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTagsForResource.html",
                "description": "Lists tags for a resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "document-classifier": {
                        "required": false
                    },
                    "document-classifier-endpoint": {
                        "required": false
                    },
                    "entity-recognizer": {
                        "required": false
                    }
                }
            },
            "ListTopicsDetectionJobs": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTopicsDetectionJobs.html",
                "description": "Gets a list of the topic detection jobs that you have submitted.",
                "accessLevel": "List"
            },
            "StartDocumentClassificationJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDocumentClassificationJob.html",
                "description": "Starts an asynchronous document classification job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "StartDominantLanguageDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDominantLanguageDetectionJob.html",
                "description": "Starts an asynchronous dominant language detection job for a collection of documents.",
                "accessLevel": "Write"
            },
            "StartEntitiesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartEntitiesDetectionJob.html",
                "description": "Starts an asynchronous entity detection job for a collection of documents.",
                "accessLevel": "Write"
            },
            "StartKeyPhrasesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartKeyPhrasesDetectionJob.html",
                "description": "Starts an asynchronous key phrase detection job for a collection of documents.",
                "accessLevel": "Write"
            },
            "StartSentimentDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartSentimentDetectionJob.html",
                "description": "Starts an asynchronous sentiment detection job for a collection of documents.",
                "accessLevel": "Write"
            },
            "StartTopicsDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StartTopicsDetectionJob.html",
                "description": "Starts an asynchronous job to detect the most common topics in the collection of documents and the phrases associated with each topic.",
                "accessLevel": "Write"
            },
            "StopDominantLanguageDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopDominantLanguageDetectionJob.html",
                "description": "Stops a dominant language detection job.",
                "accessLevel": "Write"
            },
            "StopEntitiesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopEntitiesDetectionJob.html",
                "description": "Stops an entity detection job.",
                "accessLevel": "Write"
            },
            "StopKeyPhrasesDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopKeyPhrasesDetectionJob.html",
                "description": "Stops a key phrase detection job.",
                "accessLevel": "Write"
            },
            "StopSentimentDetectionJob": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopSentimentDetectionJob.html",
                "description": "Stops a sentiment detection job.",
                "accessLevel": "Write"
            },
            "StopTrainingDocumentClassifier": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingDocumentClassifier.html",
                "description": "Stop a previously created document classifier training job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier": {
                        "required": true
                    }
                }
            },
            "StopTrainingEntityRecognizer": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingEntityRecognizer.html",
                "description": "Stop a previously created entity recognizer training job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "entity-recognizer": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_TagResource.html",
                "description": "Tags a resource with given key value pairs",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "document-classifier": {
                        "required": false
                    },
                    "document-classifier-endpoint": {
                        "required": false
                    },
                    "entity-recognizer": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_UntagResource.html",
                "description": "Untags a resource with given key",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "document-classifier": {
                        "required": false
                    },
                    "document-classifier-endpoint": {
                        "required": false
                    },
                    "entity-recognizer": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateEndpoint": {
                "url": "https://docs.aws.amazon.com/comprehend/latest/dg/API_UpdateEndpoint.html",
                "description": "Updates information about the specified endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "document-classifier-endpoint": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "document-classifier": {
                "name": "document-classifier",
                "url": "",
                "arn": "arn:${Partition}:comprehend:${Region}:${Account}:document-classifier/${DocumentClassifierName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "entity-recognizer": {
                "name": "entity-recognizer",
                "url": "",
                "arn": "arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer/${EntityRecognizerName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "document-classifier-endpoint": {
                "name": "document-classifier-endpoint",
                "url": "",
                "arn": "arn:${Partition}:comprehend:${Region}:${Account}:document-classifier-endpoint/${DocumentClassifierEndpointName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Detects the language or languages present in the list of text documents.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectDominantLanguage.html
     */
    batchDetectDominantLanguage() {
        this.add('comprehend:BatchDetectDominantLanguage');
        return this;
    }
    /**
     * Detects the named entities ("People", "Places", "Locations", etc) within the given list of text documents.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectEntities.html
     */
    batchDetectEntities() {
        this.add('comprehend:BatchDetectEntities');
        return this;
    }
    /**
     * Detects the phrases in the list of text documents that are most indicative of the content.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectKeyPhrases.html
     */
    batchDetectKeyPhrases() {
        this.add('comprehend:BatchDetectKeyPhrases');
        return this;
    }
    /**
     * Detects the sentiment of a text in the list of documents (Positive, Negative, Neutral, or Mixed).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSentiment.html
     */
    batchDetectSentiment() {
        this.add('comprehend:BatchDetectSentiment');
        return this;
    }
    /**
     * Detects syntactic information (like Part of Speech, Tokens) in a list of text documents.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_BatchDetectSyntax.html
     */
    batchDetectSyntax() {
        this.add('comprehend:BatchDetectSyntax');
        return this;
    }
    /**
     * Creates a new document classification request to analyze a single document in real-time, using a previously created and trained custom model and an endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ClassifyDocument.html
     */
    classifyDocument() {
        this.add('comprehend:ClassifyDocument');
        return this;
    }
    /**
     * Creates a new document classifier that you can use to categorize documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateDocumentClassifier.html
     */
    createDocumentClassifier() {
        this.add('comprehend:CreateDocumentClassifier');
        return this;
    }
    /**
     * Creates a model-specific endpoint for synchronous inference for a previously trained custom model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEndpoint.html
     */
    createEndpoint() {
        this.add('comprehend:CreateEndpoint');
        return this;
    }
    /**
     * Creates an entity recognizer using submitted files.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_CreateEntityRecognizer.html
     */
    createEntityRecognizer() {
        this.add('comprehend:CreateEntityRecognizer');
        return this;
    }
    /**
     * Deletes a previously created document classifier.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteDocumentClassifier.html
     */
    deleteDocumentClassifier() {
        this.add('comprehend:DeleteDocumentClassifier');
        return this;
    }
    /**
     * Deletes a model-specific endpoint for a previously-trained custom model. All endpoints must be deleted in order for the model to be deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEndpoint.html
     */
    deleteEndpoint() {
        this.add('comprehend:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes a submitted entity recognizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DeleteEntityRecognizer.html
     */
    deleteEntityRecognizer() {
        this.add('comprehend:DeleteEntityRecognizer');
        return this;
    }
    /**
     * Gets the properties associated with a document classification job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassificationJob.html
     */
    describeDocumentClassificationJob() {
        this.add('comprehend:DescribeDocumentClassificationJob');
        return this;
    }
    /**
     * Gets the properties associated with a document classifier.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDocumentClassifier.html
     */
    describeDocumentClassifier() {
        this.add('comprehend:DescribeDocumentClassifier');
        return this;
    }
    /**
     * Gets the properties associated with a dominant language detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeDominantLanguageDetectionJob.html
     */
    describeDominantLanguageDetectionJob() {
        this.add('comprehend:DescribeDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Gets the properties associated with a specific endpoint. Use this operation to get the status of an endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEndpoint.html
     */
    describeEndpoint() {
        this.add('comprehend:DescribeEndpoint');
        return this;
    }
    /**
     * Gets the properties associated with an entities detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntitiesDetectionJob.html
     */
    describeEntitiesDetectionJob() {
        this.add('comprehend:DescribeEntitiesDetectionJob');
        return this;
    }
    /**
     * Provides details about an entity recognizer including status, S3 buckets containing training data, recognizer metadata, metrics, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeEntityRecognizer.html
     */
    describeEntityRecognizer() {
        this.add('comprehend:DescribeEntityRecognizer');
        return this;
    }
    /**
     * Gets the properties associated with a key phrases detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeKeyPhrasesDetectionJob.html
     */
    describeKeyPhrasesDetectionJob() {
        this.add('comprehend:DescribeKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Gets the properties associated with a sentiment detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeSentimentDetectionJob.html
     */
    describeSentimentDetectionJob() {
        this.add('comprehend:DescribeSentimentDetectionJob');
        return this;
    }
    /**
     * Gets the properties associated with a topic detection job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DescribeTopicsDetectionJob.html
     */
    describeTopicsDetectionJob() {
        this.add('comprehend:DescribeTopicsDetectionJob');
        return this;
    }
    /**
     * Detects the language or languages present in the text.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectDominantLanguage.html
     */
    detectDominantLanguage() {
        this.add('comprehend:DetectDominantLanguage');
        return this;
    }
    /**
     * Detects the named entities ("People", "Places", "Locations", etc) within the given text document.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectEntities.html
     */
    detectEntities() {
        this.add('comprehend:DetectEntities');
        return this;
    }
    /**
     * Detects the phrases in the text that are most indicative of the content.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectKeyPhrases.html
     */
    detectKeyPhrases() {
        this.add('comprehend:DetectKeyPhrases');
        return this;
    }
    /**
     * Detects the sentiment of a text in a document (Positive, Negative, Neutral, or Mixed).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSentiment.html
     */
    detectSentiment() {
        this.add('comprehend:DetectSentiment');
        return this;
    }
    /**
     * Detects syntactic information (like Part of Speech, Tokens) in a text document.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_DetectSyntax.html
     */
    detectSyntax() {
        this.add('comprehend:DetectSyntax');
        return this;
    }
    /**
     * Gets a list of the document classification jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassificationJobs.html
     */
    listDocumentClassificationJobs() {
        this.add('comprehend:ListDocumentClassificationJobs');
        return this;
    }
    /**
     * Gets a list of the document classifiers that you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDocumentClassifiers.html
     */
    listDocumentClassifiers() {
        this.add('comprehend:ListDocumentClassifiers');
        return this;
    }
    /**
     * Gets a list of the dominant language detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListDominantLanguageDetectionJobs.html
     */
    listDominantLanguageDetectionJobs() {
        this.add('comprehend:ListDominantLanguageDetectionJobs');
        return this;
    }
    /**
     * Gets a list of all existing endpoints that you've created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEndpoints.html
     */
    listEndpoints() {
        this.add('comprehend:ListEndpoints');
        return this;
    }
    /**
     * Gets a list of the entity detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntitiesDetectionJobs.html
     */
    listEntitiesDetectionJobs() {
        this.add('comprehend:ListEntitiesDetectionJobs');
        return this;
    }
    /**
     * Gets a list of the properties of all entity recognizers that you created, including recognizers currently in training.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListEntityRecognizers.html
     */
    listEntityRecognizers() {
        this.add('comprehend:ListEntityRecognizers');
        return this;
    }
    /**
     * Get a list of key phrase detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListKeyPhrasesDetectionJobs.html
     */
    listKeyPhrasesDetectionJobs() {
        this.add('comprehend:ListKeyPhrasesDetectionJobs');
        return this;
    }
    /**
     * Gets a list of sentiment detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListSentimentDetectionJobs.html
     */
    listSentimentDetectionJobs() {
        this.add('comprehend:ListSentimentDetectionJobs');
        return this;
    }
    /**
     * Lists tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('comprehend:ListTagsForResource');
        return this;
    }
    /**
     * Gets a list of the topic detection jobs that you have submitted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_ListTopicsDetectionJobs.html
     */
    listTopicsDetectionJobs() {
        this.add('comprehend:ListTopicsDetectionJobs');
        return this;
    }
    /**
     * Starts an asynchronous document classification job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDocumentClassificationJob.html
     */
    startDocumentClassificationJob() {
        this.add('comprehend:StartDocumentClassificationJob');
        return this;
    }
    /**
     * Starts an asynchronous dominant language detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartDominantLanguageDetectionJob.html
     */
    startDominantLanguageDetectionJob() {
        this.add('comprehend:StartDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous entity detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartEntitiesDetectionJob.html
     */
    startEntitiesDetectionJob() {
        this.add('comprehend:StartEntitiesDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous key phrase detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartKeyPhrasesDetectionJob.html
     */
    startKeyPhrasesDetectionJob() {
        this.add('comprehend:StartKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous sentiment detection job for a collection of documents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartSentimentDetectionJob.html
     */
    startSentimentDetectionJob() {
        this.add('comprehend:StartSentimentDetectionJob');
        return this;
    }
    /**
     * Starts an asynchronous job to detect the most common topics in the collection of documents and the phrases associated with each topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StartTopicsDetectionJob.html
     */
    startTopicsDetectionJob() {
        this.add('comprehend:StartTopicsDetectionJob');
        return this;
    }
    /**
     * Stops a dominant language detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopDominantLanguageDetectionJob.html
     */
    stopDominantLanguageDetectionJob() {
        this.add('comprehend:StopDominantLanguageDetectionJob');
        return this;
    }
    /**
     * Stops an entity detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopEntitiesDetectionJob.html
     */
    stopEntitiesDetectionJob() {
        this.add('comprehend:StopEntitiesDetectionJob');
        return this;
    }
    /**
     * Stops a key phrase detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopKeyPhrasesDetectionJob.html
     */
    stopKeyPhrasesDetectionJob() {
        this.add('comprehend:StopKeyPhrasesDetectionJob');
        return this;
    }
    /**
     * Stops a sentiment detection job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopSentimentDetectionJob.html
     */
    stopSentimentDetectionJob() {
        this.add('comprehend:StopSentimentDetectionJob');
        return this;
    }
    /**
     * Stop a previously created document classifier training job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingDocumentClassifier.html
     */
    stopTrainingDocumentClassifier() {
        this.add('comprehend:StopTrainingDocumentClassifier');
        return this;
    }
    /**
     * Stop a previously created entity recognizer training job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_StopTrainingEntityRecognizer.html
     */
    stopTrainingEntityRecognizer() {
        this.add('comprehend:StopTrainingEntityRecognizer');
        return this;
    }
    /**
     * Tags a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_TagResource.html
     */
    tagResource() {
        this.add('comprehend:TagResource');
        return this;
    }
    /**
     * Untags a resource with given key
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UntagResource.html
     */
    untagResource() {
        this.add('comprehend:UntagResource');
        return this;
    }
    /**
     * Updates information about the specified endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/API_UpdateEndpoint.html
     */
    updateEndpoint() {
        this.add('comprehend:UpdateEndpoint');
        return this;
    }
    /**
     * Adds a resource of type document-classifier to the statement
     *
     * @param documentClassifierName - Identifier for the documentClassifierName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDocumentClassifier(documentClassifierName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier/${DocumentClassifierName}';
        arn = arn.replace('${DocumentClassifierName}', documentClassifierName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-recognizer to the statement
     *
     * @param entityRecognizerName - Identifier for the entityRecognizerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onEntityRecognizer(entityRecognizerName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:entity-recognizer/${EntityRecognizerName}';
        arn = arn.replace('${EntityRecognizerName}', entityRecognizerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type document-classifier-endpoint to the statement
     *
     * @param documentClassifierEndpointName - Identifier for the documentClassifierEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDocumentClassifierEndpoint(documentClassifierEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:comprehend:${Region}:${Account}:document-classifier-endpoint/${DocumentClassifierEndpointName}';
        arn = arn.replace('${DocumentClassifierEndpointName}', documentClassifierEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Comprehend = Comprehend;
//# sourceMappingURL=data:application/json;base64,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