"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service honeycode
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonhoneycode.html
 */
class Honeycode extends shared_1.PolicyStatement {
    /**
     * Action provider for service honeycode
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonhoneycode.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'honeycode';
        this.actions = {
            "ApproveTeamAssociation": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#approve-team-association",
                "description": "Grants permission to approve a team association request for your AWS Account",
                "accessLevel": "Write"
            },
            "GetScreenData": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_GetScreenData.html",
                "description": "Grants permission to load the data from a screen",
                "accessLevel": "Read",
                "resourceTypes": {
                    "screen": {
                        "required": true
                    }
                }
            },
            "InvokeScreenAutomation": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_InvokeScreenAutomation.html",
                "description": "Grants permission to invoke a screen automation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "screen-automation": {
                        "required": true
                    }
                }
            },
            "ListTeamAssociations": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#list-team-associations",
                "description": "Grants permission to list all pending and approved team associations with your AWS Account",
                "accessLevel": "List"
            },
            "RejectTeamAssociation": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#reject-team-association",
                "description": "Grants permission to reject a team association request for your AWS Account",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "screen": {
                "name": "screen",
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen.html",
                "arn": "arn:${Partition}:honeycode:${Region}:${Account}:screen:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}",
                "conditionKeys": []
            },
            "screen-automation": {
                "name": "screen-automation",
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen-automation.html",
                "arn": "arn:${Partition}:honeycode:${Region}:${Account}:screen-automation:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}/automation/${AutomationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to approve a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#approve-team-association
     */
    approveTeamAssociation() {
        this.add('honeycode:ApproveTeamAssociation');
        return this;
    }
    /**
     * Grants permission to load the data from a screen
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_GetScreenData.html
     */
    getScreenData() {
        this.add('honeycode:GetScreenData');
        return this;
    }
    /**
     * Grants permission to invoke a screen automation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_InvokeScreenAutomation.html
     */
    invokeScreenAutomation() {
        this.add('honeycode:InvokeScreenAutomation');
        return this;
    }
    /**
     * Grants permission to list all pending and approved team associations with your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#list-team-associations
     */
    listTeamAssociations() {
        this.add('honeycode:ListTeamAssociations');
        return this;
    }
    /**
     * Grants permission to reject a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#reject-team-association
     */
    rejectTeamAssociation() {
        this.add('honeycode:RejectTeamAssociation');
        return this;
    }
    /**
     * Adds a resource of type screen to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreen(workbookId, appId, screenId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type screen-automation to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen-automation.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param automationId - Identifier for the automationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreenAutomation(workbookId, appId, screenId, automationId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen-automation:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}/automation/${AutomationId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${AutomationId}', automationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Honeycode = Honeycode;
//# sourceMappingURL=data:application/json;base64,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