"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service mobiletargeting
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpinpoint.html
 */
class Mobiletargeting extends shared_1.PolicyStatement {
    /**
     * Action provider for service mobiletargeting
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonpinpoint.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mobiletargeting';
        this.actions = {
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Create an app.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods",
                "description": "Create a campaign for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Create an email template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateExportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Create an export job that exports endpoint definitions to Amazon S3.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "CreateImportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods",
                "description": "Import endpoint definitions from to create a segment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "CreateJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Create a Journey for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreatePushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Create a push notification template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#CreateRecommenderConfiguration",
                "description": "Create an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Write"
            },
            "CreateSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods",
                "description": "Create a segment that is based on endpoint data reported to Pinpoint by your app. To allow a user to create a segment by importing endpoint data from outside of Pinpoint, allow the mobiletargeting:CreateImportJob action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Create an sms message template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "CreateVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Create a voice message template.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteAdmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods",
                "description": "Delete the ADM channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods",
                "description": "Delete the APNs channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods",
                "description": "Delete the APNs sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsVoipChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods",
                "description": "Delete the APNs VoIP channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApnsVoipSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods",
                "description": "Delete the APNs VoIP sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Delete a specific campaign.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteBaiduChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods",
                "description": "Delete the Baidu channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods",
                "description": "Delete a specific campaign.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "DeleteEmailChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods",
                "description": "Delete the email channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Delete an email template or an email template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods",
                "description": "Delete an endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteEventStream": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods",
                "description": "Delete the event stream for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteGcmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods",
                "description": "Delete the GCM channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Delete a specific journey.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                }
            },
            "DeletePushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Delete a push notification template or a push notification template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "DeleteRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#DeleteRecommenderConfiguration",
                "description": "Delete an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recommenders": {
                        "required": true
                    }
                }
            },
            "DeleteSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods",
                "description": "Delete a specific segment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "DeleteSmsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods",
                "description": "Delete the SMS channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Delete an sms message template or an sms message template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "DeleteUserEndpoints": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods",
                "description": "Delete all of the endpoints that are associated with a user ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteVoiceChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods",
                "description": "Delete the Voice channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "DeleteVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Delete a voice message template or a voice message template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetAdmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods",
                "description": "Retrieve information about the Amazon Device Messaging (ADM) channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods",
                "description": "Retrieve information about the APNs channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods",
                "description": "Retrieve information about the APNs sandbox channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsVoipChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods",
                "description": "Retrieve information about the APNs VoIP channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApnsVoipSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods",
                "description": "Retrieve information about the APNs VoIP sandbox channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApp": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Retrieve information about a specific app in your Amazon Pinpoint account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApplicationSettings": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods",
                "description": "Retrieve the default settings for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetApps": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apps.html#rest-api-apps-methods",
                "description": "Retrieve a list of apps in your Amazon Pinpoint account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetBaiduChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods",
                "description": "Retrieve information about the Baidu channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods",
                "description": "Retrieve information about a specific campaign.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaignActivities": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-activities.html#rest-api-campaign-activities-methods",
                "description": "Retrieve information about the activities performed by a campaign.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaignVersion": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-version.html#rest-api-campaign-version-methods",
                "description": "Retrieve information about a specific campaign version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaignVersions": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-versions.html#rest-api-campaign-versions-methods",
                "description": "Retrieve information about the current and prior versions of a campaign.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                }
            },
            "GetCampaigns": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods",
                "description": "Retrieve information about all campaigns for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetChannels": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-channels.html#rest-api-channels-methods",
                "description": "Get all channels information for your app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetEmailChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods",
                "description": "Obtain information about the email channel in an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Retrieve information about a specific or the active version of an email template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetEndpoint": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods",
                "description": "Retrieve information about a specific endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetEventStream": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods",
                "description": "Retrieve information about the event stream for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetExportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Obtain information about a specific export job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetExportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Retrieve a list of all of the export jobs for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetGcmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods",
                "description": "Retrieve information about the GCM channel for an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetImportJob": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-job.html#rest-api-import-job-methods",
                "description": "Retrieve information about a specific import job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetImportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods",
                "description": "Retrieve information about all import jobs for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Retrieve information about a specific journey.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                }
            },
            "GetPushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Retrieve information about a specific or the active version of an push notification template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#GetRecommenderConfiguration",
                "description": "Retrieve information about an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recommenders": {
                        "required": true
                    }
                }
            },
            "GetRecommenderConfigurations": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#GetRecommenderConfigurations",
                "description": "Retrieve information about all the recommender model configurations that are associated with an Amazon Pinpoint account.",
                "accessLevel": "List"
            },
            "GetSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods",
                "description": "Retrieve information about a specific segment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentExportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods",
                "description": "Retrieve information about jobs that export endpoint definitions from segments to Amazon S3.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentImportJobs": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods",
                "description": "Retrieve information about jobs that create segments by importing endpoint definitions from .",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentVersion": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-version.html#rest-api-segment-version-methods",
                "description": "Retrieve information about a specific segment version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegmentVersions": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-versions.html#rest-api-segment-versions-methods",
                "description": "Retrieve information about the current and prior versions of a segment.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                }
            },
            "GetSegments": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods",
                "description": "Retrieve information about the segments for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetSmsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods",
                "description": "Obtain information about the SMS channel in an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Retrieve information about a specific or the active version of an sms message template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "GetUserEndpoints": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods",
                "description": "Retrieve information about the endpoints that are associated with a user ID.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetVoiceChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods",
                "description": "Obtain information about the Voice channel in an app.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "GetVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Retrieve information about a specific or the active version of a voice message template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "ListJourneys": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html#apps-application-id-journeys-http-methods",
                "description": "Retrieve information about all journeys for an app.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-get",
                "description": "List tags for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "campaigns": {
                        "required": false
                    },
                    "segments": {
                        "required": false
                    }
                }
            },
            "ListTemplateVersions": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods",
                "description": "Retrieve all versions about a specific template.",
                "accessLevel": "List",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "ListTemplates": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html#templates-http-methods",
                "description": "Retrieve metadata about the queried templates.",
                "accessLevel": "List",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "PhoneNumberValidate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-phone-number-validate.html#rest-api-phone-number-validate-methods",
                "description": "Obtain metadata for a phone number, such as the number type (mobile, landline, or VoIP), location, and provider.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "PutEventStream": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods",
                "description": "Create or update an event stream for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "PutEvents": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-events.html#rest-api-events-methods",
                "description": "Create or update events for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "RemoveAttributes": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods",
                "description": "Used to remove the attributes for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "SendMessages": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-messages.html#rest-api-messages-methods",
                "description": "Send an SMS message or push notification to specific endpoints.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "SendUsersMessages": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-users-messages.html#rest-api-users-messages-methods",
                "description": "Send an SMS message or push notification to all endpoints that are associated with a specific user ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-post",
                "description": "Adds tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "campaigns": {
                        "required": false
                    },
                    "segments": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-delete",
                "description": "Removes tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "apps": {
                        "required": false
                    },
                    "campaigns": {
                        "required": false
                    },
                    "segments": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateAdmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods",
                "description": "Update the Amazon Device Messaging (ADM) channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsVoipChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) VoIP channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApnsVoipSandboxChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods",
                "description": "Update the Apple Push Notification service (APNs) VoIP sandbox channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateApplicationSettings": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods",
                "description": "Update the default settings for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateBaiduChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods",
                "description": "Update the Baidu channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateCampaign": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods",
                "description": "Update a specific campaign.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "campaigns": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateEmailChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods",
                "description": "Update the email channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateEmailTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods",
                "description": "Update a specific email template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateEndpoint": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods",
                "description": "Create an endpoint or update the information for an endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateEndpointsBatch": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoints.html#rest-api-endpoints-methods",
                "description": "Create or update endpoints as a batch operation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateGcmChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods",
                "description": "Update the Firebase Cloud Messaging (FCM) or Google Cloud Messaging (GCM) API key that allows to send push notifications to your Android app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateJourney": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods",
                "description": "Update a specific journey.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateJourneyState": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-state.html#apps-application-id-journeys-journey-id-state-http-methods",
                "description": "Update a specific journey state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "journeys": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdatePushTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods",
                "description": "Update a specific push notification template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateRecommenderConfiguration": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#UpdateRecommenderConfiguration",
                "description": "Update an Amazon Pinpoint configuration for a recommender model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recommenders": {
                        "required": true
                    }
                }
            },
            "UpdateSegment": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods",
                "description": "Update a specific segment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    },
                    "segments": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateSmsChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods",
                "description": "Update the SMS channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateSmsTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods",
                "description": "Update a specific sms message template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateTemplateActiveVersion": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods",
                "description": "Upate the active version parameter of a specific template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                }
            },
            "UpdateVoiceChannel": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods",
                "description": "Update the Voice channel for an app.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "apps": {
                        "required": true
                    }
                }
            },
            "UpdateVoiceTemplate": {
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods",
                "description": "Update a specific voice message template under the same version or generate a new version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "templates": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "apps": {
                "name": "apps",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/developerguide/gettingstarted.html#gettingstarted-addapp",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "campaigns": {
                "name": "campaigns",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/campaigns/${CampaignId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "journeys": {
                "name": "journeys",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/journeys/${JourneyId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "segments": {
                "name": "segments",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/segments/${SegmentId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "templates": {
                "name": "templates",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:templates/${TemplateName}/${ChannelType}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "recommenders": {
                "name": "recommenders",
                "url": "https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html",
                "arn": "arn:${Partition}:mobiletargeting:${Region}:${Account}:recommenders/${RecommenderId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Create an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    createApp() {
        this.add('mobiletargeting:CreateApp');
        return this;
    }
    /**
     * Create a campaign for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods
     */
    createCampaign() {
        this.add('mobiletargeting:CreateCampaign');
        return this;
    }
    /**
     * Create an email template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    createEmailTemplate() {
        this.add('mobiletargeting:CreateEmailTemplate');
        return this;
    }
    /**
     * Create an export job that exports endpoint definitions to Amazon S3.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    createExportJob() {
        this.add('mobiletargeting:CreateExportJob');
        return this;
    }
    /**
     * Import endpoint definitions from to create a segment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    createImportJob() {
        this.add('mobiletargeting:CreateImportJob');
        return this;
    }
    /**
     * Create a Journey for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    createJourney() {
        this.add('mobiletargeting:CreateJourney');
        return this;
    }
    /**
     * Create a push notification template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    createPushTemplate() {
        this.add('mobiletargeting:CreatePushTemplate');
        return this;
    }
    /**
     * Create an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#CreateRecommenderConfiguration
     */
    createRecommenderConfiguration() {
        this.add('mobiletargeting:CreateRecommenderConfiguration');
        return this;
    }
    /**
     * Create a segment that is based on endpoint data reported to Pinpoint by your app. To allow a user to create a segment by importing endpoint data from outside of Pinpoint, allow the mobiletargeting:CreateImportJob action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods
     */
    createSegment() {
        this.add('mobiletargeting:CreateSegment');
        return this;
    }
    /**
     * Create an sms message template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    createSmsTemplate() {
        this.add('mobiletargeting:CreateSmsTemplate');
        return this;
    }
    /**
     * Create a voice message template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    createVoiceTemplate() {
        this.add('mobiletargeting:CreateVoiceTemplate');
        return this;
    }
    /**
     * Delete the ADM channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    deleteAdmChannel() {
        this.add('mobiletargeting:DeleteAdmChannel');
        return this;
    }
    /**
     * Delete the APNs channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    deleteApnsChannel() {
        this.add('mobiletargeting:DeleteApnsChannel');
        return this;
    }
    /**
     * Delete the APNs sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    deleteApnsSandboxChannel() {
        this.add('mobiletargeting:DeleteApnsSandboxChannel');
        return this;
    }
    /**
     * Delete the APNs VoIP channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    deleteApnsVoipChannel() {
        this.add('mobiletargeting:DeleteApnsVoipChannel');
        return this;
    }
    /**
     * Delete the APNs VoIP sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    deleteApnsVoipSandboxChannel() {
        this.add('mobiletargeting:DeleteApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Delete a specific campaign.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    deleteApp() {
        this.add('mobiletargeting:DeleteApp');
        return this;
    }
    /**
     * Delete the Baidu channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    deleteBaiduChannel() {
        this.add('mobiletargeting:DeleteBaiduChannel');
        return this;
    }
    /**
     * Delete a specific campaign.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    deleteCampaign() {
        this.add('mobiletargeting:DeleteCampaign');
        return this;
    }
    /**
     * Delete the email channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    deleteEmailChannel() {
        this.add('mobiletargeting:DeleteEmailChannel');
        return this;
    }
    /**
     * Delete an email template or an email template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    deleteEmailTemplate() {
        this.add('mobiletargeting:DeleteEmailTemplate');
        return this;
    }
    /**
     * Delete an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    deleteEndpoint() {
        this.add('mobiletargeting:DeleteEndpoint');
        return this;
    }
    /**
     * Delete the event stream for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    deleteEventStream() {
        this.add('mobiletargeting:DeleteEventStream');
        return this;
    }
    /**
     * Delete the GCM channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    deleteGcmChannel() {
        this.add('mobiletargeting:DeleteGcmChannel');
        return this;
    }
    /**
     * Delete a specific journey.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    deleteJourney() {
        this.add('mobiletargeting:DeleteJourney');
        return this;
    }
    /**
     * Delete a push notification template or a push notification template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    deletePushTemplate() {
        this.add('mobiletargeting:DeletePushTemplate');
        return this;
    }
    /**
     * Delete an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#DeleteRecommenderConfiguration
     */
    deleteRecommenderConfiguration() {
        this.add('mobiletargeting:DeleteRecommenderConfiguration');
        return this;
    }
    /**
     * Delete a specific segment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    deleteSegment() {
        this.add('mobiletargeting:DeleteSegment');
        return this;
    }
    /**
     * Delete the SMS channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    deleteSmsChannel() {
        this.add('mobiletargeting:DeleteSmsChannel');
        return this;
    }
    /**
     * Delete an sms message template or an sms message template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    deleteSmsTemplate() {
        this.add('mobiletargeting:DeleteSmsTemplate');
        return this;
    }
    /**
     * Delete all of the endpoints that are associated with a user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods
     */
    deleteUserEndpoints() {
        this.add('mobiletargeting:DeleteUserEndpoints');
        return this;
    }
    /**
     * Delete the Voice channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    deleteVoiceChannel() {
        this.add('mobiletargeting:DeleteVoiceChannel');
        return this;
    }
    /**
     * Delete a voice message template or a voice message template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    deleteVoiceTemplate() {
        this.add('mobiletargeting:DeleteVoiceTemplate');
        return this;
    }
    /**
     * Retrieve information about the Amazon Device Messaging (ADM) channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    getAdmChannel() {
        this.add('mobiletargeting:GetAdmChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    getApnsChannel() {
        this.add('mobiletargeting:GetApnsChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs sandbox channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    getApnsSandboxChannel() {
        this.add('mobiletargeting:GetApnsSandboxChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs VoIP channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    getApnsVoipChannel() {
        this.add('mobiletargeting:GetApnsVoipChannel');
        return this;
    }
    /**
     * Retrieve information about the APNs VoIP sandbox channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    getApnsVoipSandboxChannel() {
        this.add('mobiletargeting:GetApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Retrieve information about a specific app in your Amazon Pinpoint account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    getApp() {
        this.add('mobiletargeting:GetApp');
        return this;
    }
    /**
     * Retrieve the default settings for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods
     */
    getApplicationSettings() {
        this.add('mobiletargeting:GetApplicationSettings');
        return this;
    }
    /**
     * Retrieve a list of apps in your Amazon Pinpoint account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apps.html#rest-api-apps-methods
     */
    getApps() {
        this.add('mobiletargeting:GetApps');
        return this;
    }
    /**
     * Retrieve information about the Baidu channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    getBaiduChannel() {
        this.add('mobiletargeting:GetBaiduChannel');
        return this;
    }
    /**
     * Retrieve information about a specific campaign.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    getCampaign() {
        this.add('mobiletargeting:GetCampaign');
        return this;
    }
    /**
     * Retrieve information about the activities performed by a campaign.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-activities.html#rest-api-campaign-activities-methods
     */
    getCampaignActivities() {
        this.add('mobiletargeting:GetCampaignActivities');
        return this;
    }
    /**
     * Retrieve information about a specific campaign version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-version.html#rest-api-campaign-version-methods
     */
    getCampaignVersion() {
        this.add('mobiletargeting:GetCampaignVersion');
        return this;
    }
    /**
     * Retrieve information about the current and prior versions of a campaign.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign-versions.html#rest-api-campaign-versions-methods
     */
    getCampaignVersions() {
        this.add('mobiletargeting:GetCampaignVersions');
        return this;
    }
    /**
     * Retrieve information about all campaigns for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html#rest-api-campaigns-methods
     */
    getCampaigns() {
        this.add('mobiletargeting:GetCampaigns');
        return this;
    }
    /**
     * Get all channels information for your app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-channels.html#rest-api-channels-methods
     */
    getChannels() {
        this.add('mobiletargeting:GetChannels');
        return this;
    }
    /**
     * Obtain information about the email channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    getEmailChannel() {
        this.add('mobiletargeting:GetEmailChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an email template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    getEmailTemplate() {
        this.add('mobiletargeting:GetEmailTemplate');
        return this;
    }
    /**
     * Retrieve information about a specific endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    getEndpoint() {
        this.add('mobiletargeting:GetEndpoint');
        return this;
    }
    /**
     * Retrieve information about the event stream for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    getEventStream() {
        this.add('mobiletargeting:GetEventStream');
        return this;
    }
    /**
     * Obtain information about a specific export job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    getExportJob() {
        this.add('mobiletargeting:GetExportJob');
        return this;
    }
    /**
     * Retrieve a list of all of the export jobs for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    getExportJobs() {
        this.add('mobiletargeting:GetExportJobs');
        return this;
    }
    /**
     * Retrieve information about the GCM channel for an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    getGcmChannel() {
        this.add('mobiletargeting:GetGcmChannel');
        return this;
    }
    /**
     * Retrieve information about a specific import job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-job.html#rest-api-import-job-methods
     */
    getImportJob() {
        this.add('mobiletargeting:GetImportJob');
        return this;
    }
    /**
     * Retrieve information about all import jobs for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    getImportJobs() {
        this.add('mobiletargeting:GetImportJobs');
        return this;
    }
    /**
     * Retrieve information about a specific journey.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    getJourney() {
        this.add('mobiletargeting:GetJourney');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an push notification template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    getPushTemplate() {
        this.add('mobiletargeting:GetPushTemplate');
        return this;
    }
    /**
     * Retrieve information about an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#GetRecommenderConfiguration
     */
    getRecommenderConfiguration() {
        this.add('mobiletargeting:GetRecommenderConfiguration');
        return this;
    }
    /**
     * Retrieve information about all the recommender model configurations that are associated with an Amazon Pinpoint account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html#GetRecommenderConfigurations
     */
    getRecommenderConfigurations() {
        this.add('mobiletargeting:GetRecommenderConfigurations');
        return this;
    }
    /**
     * Retrieve information about a specific segment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    getSegment() {
        this.add('mobiletargeting:GetSegment');
        return this;
    }
    /**
     * Retrieve information about jobs that export endpoint definitions from segments to Amazon S3.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-export-jobs.html#rest-api-export-jobs-methods
     */
    getSegmentExportJobs() {
        this.add('mobiletargeting:GetSegmentExportJobs');
        return this;
    }
    /**
     * Retrieve information about jobs that create segments by importing endpoint definitions from .
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-import-jobs.html#rest-api-import-jobs-methods
     */
    getSegmentImportJobs() {
        this.add('mobiletargeting:GetSegmentImportJobs');
        return this;
    }
    /**
     * Retrieve information about a specific segment version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-version.html#rest-api-segment-version-methods
     */
    getSegmentVersion() {
        this.add('mobiletargeting:GetSegmentVersion');
        return this;
    }
    /**
     * Retrieve information about the current and prior versions of a segment.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment-versions.html#rest-api-segment-versions-methods
     */
    getSegmentVersions() {
        this.add('mobiletargeting:GetSegmentVersions');
        return this;
    }
    /**
     * Retrieve information about the segments for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html#rest-api-segments-methods
     */
    getSegments() {
        this.add('mobiletargeting:GetSegments');
        return this;
    }
    /**
     * Obtain information about the SMS channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    getSmsChannel() {
        this.add('mobiletargeting:GetSmsChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of an sms message template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    getSmsTemplate() {
        this.add('mobiletargeting:GetSmsTemplate');
        return this;
    }
    /**
     * Retrieve information about the endpoints that are associated with a user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-user.html#rest-api-user-methods
     */
    getUserEndpoints() {
        this.add('mobiletargeting:GetUserEndpoints');
        return this;
    }
    /**
     * Obtain information about the Voice channel in an app.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    getVoiceChannel() {
        this.add('mobiletargeting:GetVoiceChannel');
        return this;
    }
    /**
     * Retrieve information about a specific or the active version of a voice message template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    getVoiceTemplate() {
        this.add('mobiletargeting:GetVoiceTemplate');
        return this;
    }
    /**
     * Retrieve information about all journeys for an app.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html#apps-application-id-journeys-http-methods
     */
    listJourneys() {
        this.add('mobiletargeting:ListJourneys');
        return this;
    }
    /**
     * List tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-get
     */
    listTagsForResource() {
        this.add('mobiletargeting:ListTagsForResource');
        return this;
    }
    /**
     * Retrieve all versions about a specific template.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods
     */
    listTemplateVersions() {
        this.add('mobiletargeting:ListTemplateVersions');
        return this;
    }
    /**
     * Retrieve metadata about the queried templates.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html#templates-http-methods
     */
    listTemplates() {
        this.add('mobiletargeting:ListTemplates');
        return this;
    }
    /**
     * Obtain metadata for a phone number, such as the number type (mobile, landline, or VoIP), location, and provider.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-phone-number-validate.html#rest-api-phone-number-validate-methods
     */
    phoneNumberValidate() {
        this.add('mobiletargeting:PhoneNumberValidate');
        return this;
    }
    /**
     * Create or update an event stream for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-event-stream.html#rest-api-event-stream-methods
     */
    putEventStream() {
        this.add('mobiletargeting:PutEventStream');
        return this;
    }
    /**
     * Create or update events for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-events.html#rest-api-events-methods
     */
    putEvents() {
        this.add('mobiletargeting:PutEvents');
        return this;
    }
    /**
     * Used to remove the attributes for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-app.html#rest-api-app-methods
     */
    removeAttributes() {
        this.add('mobiletargeting:RemoveAttributes');
        return this;
    }
    /**
     * Send an SMS message or push notification to specific endpoints.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-messages.html#rest-api-messages-methods
     */
    sendMessages() {
        this.add('mobiletargeting:SendMessages');
        return this;
    }
    /**
     * Send an SMS message or push notification to all endpoints that are associated with a specific user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-users-messages.html#rest-api-users-messages-methods
     */
    sendUsersMessages() {
        this.add('mobiletargeting:SendUsersMessages');
        return this;
    }
    /**
     * Adds tags to a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-post
     */
    tagResource() {
        this.add('mobiletargeting:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-tags.html#rest-api-tags-methods-delete
     */
    untagResource() {
        this.add('mobiletargeting:UntagResource');
        return this;
    }
    /**
     * Update the Amazon Device Messaging (ADM) channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-adm-channel.html#rest-api-adm-channel-methods
     */
    updateAdmChannel() {
        this.add('mobiletargeting:UpdateAdmChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-channel.html#rest-api-apns-channel-methods
     */
    updateApnsChannel() {
        this.add('mobiletargeting:UpdateApnsChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-sandbox-channel.html#rest-api-apns-sandbox-channel-methods
     */
    updateApnsSandboxChannel() {
        this.add('mobiletargeting:UpdateApnsSandboxChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) VoIP channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-channel.html#rest-api-apns-voip-channel-methods
     */
    updateApnsVoipChannel() {
        this.add('mobiletargeting:UpdateApnsVoipChannel');
        return this;
    }
    /**
     * Update the Apple Push Notification service (APNs) VoIP sandbox channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-apns-voip-sandbox-channel.html#rest-api-apns-voip-sandbox-channel-methods
     */
    updateApnsVoipSandboxChannel() {
        this.add('mobiletargeting:UpdateApnsVoipSandboxChannel');
        return this;
    }
    /**
     * Update the default settings for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-settings.html#rest-api-settings-methods
     */
    updateApplicationSettings() {
        this.add('mobiletargeting:UpdateApplicationSettings');
        return this;
    }
    /**
     * Update the Baidu channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-baidu-channel.html#rest-api-baidu-channel-methods
     */
    updateBaiduChannel() {
        this.add('mobiletargeting:UpdateBaiduChannel');
        return this;
    }
    /**
     * Update a specific campaign.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaign.html#rest-api-campaign-methods
     */
    updateCampaign() {
        this.add('mobiletargeting:UpdateCampaign');
        return this;
    }
    /**
     * Update the email channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-email-channel.html#rest-api-email-channel-methods
     */
    updateEmailChannel() {
        this.add('mobiletargeting:UpdateEmailChannel');
        return this;
    }
    /**
     * Update a specific email template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-email.html#templates-template-name-email-http-methods
     */
    updateEmailTemplate() {
        this.add('mobiletargeting:UpdateEmailTemplate');
        return this;
    }
    /**
     * Create an endpoint or update the information for an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoint.html#rest-api-endpoint-methods
     */
    updateEndpoint() {
        this.add('mobiletargeting:UpdateEndpoint');
        return this;
    }
    /**
     * Create or update endpoints as a batch operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-endpoints.html#rest-api-endpoints-methods
     */
    updateEndpointsBatch() {
        this.add('mobiletargeting:UpdateEndpointsBatch');
        return this;
    }
    /**
     * Update the Firebase Cloud Messaging (FCM) or Google Cloud Messaging (GCM) API key that allows to send push notifications to your Android app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-gcm-channel.html#rest-api-gcm-channel-methods
     */
    updateGcmChannel() {
        this.add('mobiletargeting:UpdateGcmChannel');
        return this;
    }
    /**
     * Update a specific journey.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id.html#apps-application-id-journeys-journey-id-http-methods
     */
    updateJourney() {
        this.add('mobiletargeting:UpdateJourney');
        return this;
    }
    /**
     * Update a specific journey state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys-journey-id-state.html#apps-application-id-journeys-journey-id-state-http-methods
     */
    updateJourneyState() {
        this.add('mobiletargeting:UpdateJourneyState');
        return this;
    }
    /**
     * Update a specific push notification template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-push.html#templates-template-name-push-http-methods
     */
    updatePushTemplate() {
        this.add('mobiletargeting:UpdatePushTemplate');
        return this;
    }
    /**
     * Update an Amazon Pinpoint configuration for a recommender model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders-recommender-id.html#UpdateRecommenderConfiguration
     */
    updateRecommenderConfiguration() {
        this.add('mobiletargeting:UpdateRecommenderConfiguration');
        return this;
    }
    /**
     * Update a specific segment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segment.html#rest-api-segment-methods
     */
    updateSegment() {
        this.add('mobiletargeting:UpdateSegment');
        return this;
    }
    /**
     * Update the SMS channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-sms-channel.html#rest-api-sms-channel-methods
     */
    updateSmsChannel() {
        this.add('mobiletargeting:UpdateSmsChannel');
        return this;
    }
    /**
     * Update a specific sms message template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-sms.html#templates-template-name-sms-http-methods
     */
    updateSmsTemplate() {
        this.add('mobiletargeting:UpdateSmsTemplate');
        return this;
    }
    /**
     * Upate the active version parameter of a specific template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-template-type-versions.html#templates-template-name-template-type-versions-http-methods
     */
    updateTemplateActiveVersion() {
        this.add('mobiletargeting:UpdateTemplateActiveVersion');
        return this;
    }
    /**
     * Update the Voice channel for an app.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-voice-channel.html#rest-api-voice-channel-methods
     */
    updateVoiceChannel() {
        this.add('mobiletargeting:UpdateVoiceChannel');
        return this;
    }
    /**
     * Update a specific voice message template under the same version or generate a new version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates-template-name-voice.html#templates-template-name-voice-http-methods
     */
    updateVoiceTemplate() {
        this.add('mobiletargeting:UpdateVoiceTemplate');
        return this;
    }
    /**
     * Adds a resource of type apps to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/developerguide/gettingstarted.html#gettingstarted-addapp
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onApps(appId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type campaigns to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-campaigns.html
     *
     * @param appId - Identifier for the appId.
     * @param campaignId - Identifier for the campaignId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCampaigns(appId, campaignId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/campaigns/${CampaignId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${CampaignId}', campaignId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type journeys to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/apps-application-id-journeys.html
     *
     * @param appId - Identifier for the appId.
     * @param journeyId - Identifier for the journeyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onJourneys(appId, journeyId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/journeys/${JourneyId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${JourneyId}', journeyId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type segments to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/rest-api-segments.html
     *
     * @param appId - Identifier for the appId.
     * @param segmentId - Identifier for the segmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSegments(appId, segmentId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:apps/${AppId}/segments/${SegmentId}';
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${SegmentId}', segmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type templates to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/templates.html
     *
     * @param templateName - Identifier for the templateName.
     * @param channelType - Identifier for the channelType.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onTemplates(templateName, channelType, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:templates/${TemplateName}/${ChannelType}';
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${ChannelType}', channelType);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type recommenders to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference/recommenders.html
     *
     * @param recommenderId - Identifier for the recommenderId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRecommenders(recommenderId, account, region, partition) {
        var arn = 'arn:${Partition}:mobiletargeting:${Region}:${Account}:recommenders/${RecommenderId}';
        arn = arn.replace('${RecommenderId}', recommenderId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mobiletargeting = Mobiletargeting;
//# sourceMappingURL=data:application/json;base64,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