"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service rekognition
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html
 */
class Rekognition extends shared_1.PolicyStatement {
    /**
     * Action provider for service rekognition
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'rekognition';
        this.actions = {
            "CompareFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html",
                "description": "Compares a face in source input image with each face detected in the target input image.",
                "accessLevel": "Read"
            },
            "CreateCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html",
                "description": "Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html",
                "description": "Creates a new Amazon Rekognition Custom Labels project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "CreateProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html",
                "description": "Creates a new version of a model and begins training.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    },
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "CreateStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html",
                "description": "Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    },
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DeleteCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html",
                "description": "Deletes the specified collection. Note that this operation removes all faces in the collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DeleteFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html",
                "description": "Deletes faces from a collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html",
                "description": "Deletes a project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html",
                "description": "Deletes a model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "DeleteStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html",
                "description": "Deletes the stream processor identified by Name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DescribeCollection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html",
                "description": "Describes the specified collection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "DescribeProjectVersions": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html",
                "description": "Lists and describes the model versions in an Amazon Rekognition Custom Labels project.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeProjects": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html",
                "description": "Lists and gets information about your Amazon Rekognition Custom Labels projects.",
                "accessLevel": "Read"
            },
            "DescribeStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html",
                "description": "Provides information about a stream processor created by CreateStreamProcessor.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "DetectCustomLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html",
                "description": "Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "DetectFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html",
                "description": "Detects human faces within an image (JPEG or PNG) provided as input.",
                "accessLevel": "Read"
            },
            "DetectLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html",
                "description": "Detects instances of real-world labels within an image (JPEG or PNG) provided as input.",
                "accessLevel": "Read"
            },
            "DetectModerationLabels": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html",
                "description": "Detects moderation labels within input image.",
                "accessLevel": "Read"
            },
            "DetectText": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html",
                "description": "Detects text in the input image and converts it into machine-readable text.",
                "accessLevel": "Read"
            },
            "GetCelebrityInfo": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html",
                "description": "Gets the name and additional information about a celebrity based on his or her Rekognition ID.",
                "accessLevel": "Read"
            },
            "GetCelebrityRecognition": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html",
                "description": "Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.",
                "accessLevel": "Read"
            },
            "GetContentModeration": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html",
                "description": "Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.",
                "accessLevel": "Read"
            },
            "GetFaceDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html",
                "description": "Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.",
                "accessLevel": "Read"
            },
            "GetFaceSearch": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html",
                "description": "Gets the face search results for Rekognition Video face search started by StartFaceSearch.",
                "accessLevel": "Read"
            },
            "GetLabelDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html",
                "description": "Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.",
                "accessLevel": "Read"
            },
            "GetPersonTracking": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html",
                "description": "Gets information about people detected within a video.",
                "accessLevel": "Read"
            },
            "GetSegmentDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html",
                "description": "Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.",
                "accessLevel": "Read"
            },
            "GetTextDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html",
                "description": "Gets text detection results for a Rekognition Video analysis started by StartTextDetection.",
                "accessLevel": "Read"
            },
            "IndexFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html",
                "description": "Detects faces in the input image and adds them to the specified collection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListCollections": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html",
                "description": "Returns a list of collection IDs in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html",
                "description": "Returns metadata for faces in the specified collection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "ListStreamProcessors": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html",
                "description": "Gets a list of stream processors that you have created with CreateStreamProcessor.",
                "accessLevel": "List",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "RecognizeCelebrities": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html",
                "description": "Returns an array of celebrities recognized in the input image.",
                "accessLevel": "Read"
            },
            "SearchFaces": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html",
                "description": "For a given input face ID, searches the specified collection for matching faces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "SearchFacesByImage": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html",
                "description": "For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "StartCelebrityRecognition": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html",
                "description": "Starts asynchronous recognition of celebrities in a video.",
                "accessLevel": "Write"
            },
            "StartContentModeration": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html",
                "description": "Starts asynchronous detection of explicit or suggestive adult content in a video.",
                "accessLevel": "Write"
            },
            "StartFaceDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html",
                "description": "Starts asynchronous detection of faces in a video.",
                "accessLevel": "Write"
            },
            "StartFaceSearch": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html",
                "description": "Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "collection": {
                        "required": true
                    }
                }
            },
            "StartLabelDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html",
                "description": "Starts asynchronous detection of labels in a video.",
                "accessLevel": "Write"
            },
            "StartPersonTracking": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html",
                "description": "Starts the asynchronous tracking of persons in a video.",
                "accessLevel": "Write"
            },
            "StartProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html",
                "description": "Starts the deployment of a model version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "StartSegmentDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html",
                "description": "Starts asynchronous detection of segments in a video.",
                "accessLevel": "Write"
            },
            "StartStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html",
                "description": "Starts processing a stream processor.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            },
            "StartTextDetection": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html",
                "description": "Starts asynchronous detection of text in a video.",
                "accessLevel": "Write"
            },
            "StopProjectVersion": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html",
                "description": "Stops a deployed model version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "projectversion": {
                        "required": true
                    }
                }
            },
            "StopStreamProcessor": {
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html",
                "description": "Stops a running stream processor that was created by CreateStreamProcessor.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streamprocessor": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "collection": {
                "name": "collection",
                "url": "https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}",
                "conditionKeys": []
            },
            "streamprocessor": {
                "name": "streamprocessor",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}",
                "conditionKeys": []
            },
            "project": {
                "name": "project",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}",
                "conditionKeys": []
            },
            "projectversion": {
                "name": "projectversion",
                "url": "",
                "arn": "arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Compares a face in source input image with each face detected in the target input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html
     */
    compareFaces() {
        this.add('rekognition:CompareFaces');
        return this;
    }
    /**
     * Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html
     */
    createCollection() {
        this.add('rekognition:CreateCollection');
        return this;
    }
    /**
     * Creates a new Amazon Rekognition Custom Labels project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html
     */
    createProject() {
        this.add('rekognition:CreateProject');
        return this;
    }
    /**
     * Creates a new version of a model and begins training.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html
     */
    createProjectVersion() {
        this.add('rekognition:CreateProjectVersion');
        return this;
    }
    /**
     * Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html
     */
    createStreamProcessor() {
        this.add('rekognition:CreateStreamProcessor');
        return this;
    }
    /**
     * Deletes the specified collection. Note that this operation removes all faces in the collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html
     */
    deleteCollection() {
        this.add('rekognition:DeleteCollection');
        return this;
    }
    /**
     * Deletes faces from a collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html
     */
    deleteFaces() {
        this.add('rekognition:DeleteFaces');
        return this;
    }
    /**
     * Deletes a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html
     */
    deleteProject() {
        this.add('rekognition:DeleteProject');
        return this;
    }
    /**
     * Deletes a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html
     */
    deleteProjectVersion() {
        this.add('rekognition:DeleteProjectVersion');
        return this;
    }
    /**
     * Deletes the stream processor identified by Name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html
     */
    deleteStreamProcessor() {
        this.add('rekognition:DeleteStreamProcessor');
        return this;
    }
    /**
     * Describes the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html
     */
    describeCollection() {
        this.add('rekognition:DescribeCollection');
        return this;
    }
    /**
     * Lists and describes the model versions in an Amazon Rekognition Custom Labels project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html
     */
    describeProjectVersions() {
        this.add('rekognition:DescribeProjectVersions');
        return this;
    }
    /**
     * Lists and gets information about your Amazon Rekognition Custom Labels projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html
     */
    describeProjects() {
        this.add('rekognition:DescribeProjects');
        return this;
    }
    /**
     * Provides information about a stream processor created by CreateStreamProcessor.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html
     */
    describeStreamProcessor() {
        this.add('rekognition:DescribeStreamProcessor');
        return this;
    }
    /**
     * Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html
     */
    detectCustomLabels() {
        this.add('rekognition:DetectCustomLabels');
        return this;
    }
    /**
     * Detects human faces within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html
     */
    detectFaces() {
        this.add('rekognition:DetectFaces');
        return this;
    }
    /**
     * Detects instances of real-world labels within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html
     */
    detectLabels() {
        this.add('rekognition:DetectLabels');
        return this;
    }
    /**
     * Detects moderation labels within input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html
     */
    detectModerationLabels() {
        this.add('rekognition:DetectModerationLabels');
        return this;
    }
    /**
     * Detects text in the input image and converts it into machine-readable text.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html
     */
    detectText() {
        this.add('rekognition:DetectText');
        return this;
    }
    /**
     * Gets the name and additional information about a celebrity based on his or her Rekognition ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html
     */
    getCelebrityInfo() {
        this.add('rekognition:GetCelebrityInfo');
        return this;
    }
    /**
     * Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html
     */
    getCelebrityRecognition() {
        this.add('rekognition:GetCelebrityRecognition');
        return this;
    }
    /**
     * Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html
     */
    getContentModeration() {
        this.add('rekognition:GetContentModeration');
        return this;
    }
    /**
     * Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html
     */
    getFaceDetection() {
        this.add('rekognition:GetFaceDetection');
        return this;
    }
    /**
     * Gets the face search results for Rekognition Video face search started by StartFaceSearch.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html
     */
    getFaceSearch() {
        this.add('rekognition:GetFaceSearch');
        return this;
    }
    /**
     * Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html
     */
    getLabelDetection() {
        this.add('rekognition:GetLabelDetection');
        return this;
    }
    /**
     * Gets information about people detected within a video.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html
     */
    getPersonTracking() {
        this.add('rekognition:GetPersonTracking');
        return this;
    }
    /**
     * Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html
     */
    getSegmentDetection() {
        this.add('rekognition:GetSegmentDetection');
        return this;
    }
    /**
     * Gets text detection results for a Rekognition Video analysis started by StartTextDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html
     */
    getTextDetection() {
        this.add('rekognition:GetTextDetection');
        return this;
    }
    /**
     * Detects faces in the input image and adds them to the specified collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html
     */
    indexFaces() {
        this.add('rekognition:IndexFaces');
        return this;
    }
    /**
     * Returns a list of collection IDs in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html
     */
    listCollections() {
        this.add('rekognition:ListCollections');
        return this;
    }
    /**
     * Returns metadata for faces in the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html
     */
    listFaces() {
        this.add('rekognition:ListFaces');
        return this;
    }
    /**
     * Gets a list of stream processors that you have created with CreateStreamProcessor.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html
     */
    listStreamProcessors() {
        this.add('rekognition:ListStreamProcessors');
        return this;
    }
    /**
     * Returns an array of celebrities recognized in the input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html
     */
    recognizeCelebrities() {
        this.add('rekognition:RecognizeCelebrities');
        return this;
    }
    /**
     * For a given input face ID, searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html
     */
    searchFaces() {
        this.add('rekognition:SearchFaces');
        return this;
    }
    /**
     * For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html
     */
    searchFacesByImage() {
        this.add('rekognition:SearchFacesByImage');
        return this;
    }
    /**
     * Starts asynchronous recognition of celebrities in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html
     */
    startCelebrityRecognition() {
        this.add('rekognition:StartCelebrityRecognition');
        return this;
    }
    /**
     * Starts asynchronous detection of explicit or suggestive adult content in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html
     */
    startContentModeration() {
        this.add('rekognition:StartContentModeration');
        return this;
    }
    /**
     * Starts asynchronous detection of faces in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html
     */
    startFaceDetection() {
        this.add('rekognition:StartFaceDetection');
        return this;
    }
    /**
     * Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html
     */
    startFaceSearch() {
        this.add('rekognition:StartFaceSearch');
        return this;
    }
    /**
     * Starts asynchronous detection of labels in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html
     */
    startLabelDetection() {
        this.add('rekognition:StartLabelDetection');
        return this;
    }
    /**
     * Starts the asynchronous tracking of persons in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html
     */
    startPersonTracking() {
        this.add('rekognition:StartPersonTracking');
        return this;
    }
    /**
     * Starts the deployment of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html
     */
    startProjectVersion() {
        this.add('rekognition:StartProjectVersion');
        return this;
    }
    /**
     * Starts asynchronous detection of segments in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html
     */
    startSegmentDetection() {
        this.add('rekognition:StartSegmentDetection');
        return this;
    }
    /**
     * Starts processing a stream processor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html
     */
    startStreamProcessor() {
        this.add('rekognition:StartStreamProcessor');
        return this;
    }
    /**
     * Starts asynchronous detection of text in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html
     */
    startTextDetection() {
        this.add('rekognition:StartTextDetection');
        return this;
    }
    /**
     * Stops a deployed model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html
     */
    stopProjectVersion() {
        this.add('rekognition:StopProjectVersion');
        return this;
    }
    /**
     * Stops a running stream processor that was created by CreateStreamProcessor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html
     */
    stopStreamProcessor() {
        this.add('rekognition:StopStreamProcessor');
        return this;
    }
    /**
     * Adds a resource of type collection to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html
     *
     * @param collectionId - Identifier for the collectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCollection(collectionId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}';
        arn = arn.replace('${CollectionId}', collectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streamprocessor to the statement
     *
     * @param streamprocessorId - Identifier for the streamprocessorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStreamprocessor(streamprocessorId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}';
        arn = arn.replace('${StreamprocessorId}', streamprocessorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type projectversion to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param versionName - Identifier for the versionName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProjectversion(projectName, versionName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${VersionName}', versionName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Rekognition = Rekognition;
//# sourceMappingURL=data:application/json;base64,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