"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service deeplens
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeeplens.html
 */
class Deeplens extends shared_1.PolicyStatement {
    /**
     * Action provider for service deeplens
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeeplens.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'deeplens';
        this.actions = {
            "AssociateServiceRoleToAccount": {
                "url": "",
                "description": "Associates the user's account with IAM roles controlling various permissions needed by AWS DeepLens for proper functionality.",
                "accessLevel": "Permissions management"
            },
            "BatchGetDevice": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens devices.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "BatchGetModel": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Models.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "BatchGetProject": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Projects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "CreateDeviceCertificates": {
                "url": "",
                "description": "Creates a certificate package that is used to successfully authenticate and Register an AWS DeepLens device.",
                "accessLevel": "Write"
            },
            "CreateModel": {
                "url": "",
                "description": "Creates a new AWS DeepLens Model.",
                "accessLevel": "Write"
            },
            "CreateProject": {
                "url": "",
                "description": "Creates a new AWS DeepLens Project.",
                "accessLevel": "Write"
            },
            "DeleteModel": {
                "url": "",
                "description": "Deletes an AWS DeepLens Model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "DeleteProject": {
                "url": "",
                "description": "Deletes an AWS DeepLens Project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeployProject": {
                "url": "",
                "description": "Deploys an AWS DeepLens project to a registered AWS DeepLens device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    },
                    "project": {
                        "required": true
                    }
                }
            },
            "DeregisterDevice": {
                "url": "",
                "description": "Begins a device de-registration workflow for a registered AWS DeepLens device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "GetAssociatedResources": {
                "url": "",
                "description": "Retrieves the account level resources associated with the user's account.",
                "accessLevel": "Read"
            },
            "GetDeploymentStatus": {
                "url": "",
                "description": "Retrieves the the deployment status of a particular AWS DeepLens device, along with any associated metadata.",
                "accessLevel": "Read"
            },
            "GetDevice": {
                "url": "",
                "description": "Retrieves information about an AWS DeepLens device.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "GetModel": {
                "url": "",
                "description": "Retrieves an AWS DeepLens Model.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "GetProject": {
                "url": "",
                "description": "Retrieves an AWS DeepLens Project.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ImportProjectFromTemplate": {
                "url": "",
                "description": "Creates a new AWS DeepLens project from a sample project template.",
                "accessLevel": "Write"
            },
            "ListDeployments": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Deployment identifiers.",
                "accessLevel": "List"
            },
            "ListDevices": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens device identifiers.",
                "accessLevel": "List"
            },
            "ListModels": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Model identifiers.",
                "accessLevel": "List"
            },
            "ListProjects": {
                "url": "",
                "description": "Retrieves a list of AWS DeepLens Project identifiers.",
                "accessLevel": "List"
            },
            "RegisterDevice": {
                "url": "",
                "description": "Begins a device registration workflow for an AWS DeepLens device.",
                "accessLevel": "Write"
            },
            "RemoveProject": {
                "url": "",
                "description": "Removes a deployed AWS DeepLens project from an AWS DeepLens device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "device": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "",
                "description": "Updates an existing AWS DeepLens Project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "device": {
                "name": "device",
                "url": "",
                "arn": "arn:${Partition}:deeplens:${Region}:${Account}:device/${DeviceName}",
                "conditionKeys": []
            },
            "project": {
                "name": "project",
                "url": "",
                "arn": "arn:${Partition}:deeplens:${Region}:${Account}:project/${ProjectName}",
                "conditionKeys": []
            },
            "model": {
                "name": "model",
                "url": "",
                "arn": "arn:${Partition}:deeplens:${Region}:${Account}:model/${ModelName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associates the user's account with IAM roles controlling various permissions needed by AWS DeepLens for proper functionality.
     *
     * Access Level: Permissions management
     */
    associateServiceRoleToAccount() {
        this.add('deeplens:AssociateServiceRoleToAccount');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens devices.
     *
     * Access Level: Read
     */
    batchGetDevice() {
        this.add('deeplens:BatchGetDevice');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Models.
     *
     * Access Level: Read
     */
    batchGetModel() {
        this.add('deeplens:BatchGetModel');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Projects.
     *
     * Access Level: Read
     */
    batchGetProject() {
        this.add('deeplens:BatchGetProject');
        return this;
    }
    /**
     * Creates a certificate package that is used to successfully authenticate and Register an AWS DeepLens device.
     *
     * Access Level: Write
     */
    createDeviceCertificates() {
        this.add('deeplens:CreateDeviceCertificates');
        return this;
    }
    /**
     * Creates a new AWS DeepLens Model.
     *
     * Access Level: Write
     */
    createModel() {
        this.add('deeplens:CreateModel');
        return this;
    }
    /**
     * Creates a new AWS DeepLens Project.
     *
     * Access Level: Write
     */
    createProject() {
        this.add('deeplens:CreateProject');
        return this;
    }
    /**
     * Deletes an AWS DeepLens Model.
     *
     * Access Level: Write
     */
    deleteModel() {
        this.add('deeplens:DeleteModel');
        return this;
    }
    /**
     * Deletes an AWS DeepLens Project.
     *
     * Access Level: Write
     */
    deleteProject() {
        this.add('deeplens:DeleteProject');
        return this;
    }
    /**
     * Deploys an AWS DeepLens project to a registered AWS DeepLens device.
     *
     * Access Level: Write
     */
    deployProject() {
        this.add('deeplens:DeployProject');
        return this;
    }
    /**
     * Begins a device de-registration workflow for a registered AWS DeepLens device.
     *
     * Access Level: Write
     */
    deregisterDevice() {
        this.add('deeplens:DeregisterDevice');
        return this;
    }
    /**
     * Retrieves the account level resources associated with the user's account.
     *
     * Access Level: Read
     */
    getAssociatedResources() {
        this.add('deeplens:GetAssociatedResources');
        return this;
    }
    /**
     * Retrieves the the deployment status of a particular AWS DeepLens device, along with any associated metadata.
     *
     * Access Level: Read
     */
    getDeploymentStatus() {
        this.add('deeplens:GetDeploymentStatus');
        return this;
    }
    /**
     * Retrieves information about an AWS DeepLens device.
     *
     * Access Level: Read
     */
    getDevice() {
        this.add('deeplens:GetDevice');
        return this;
    }
    /**
     * Retrieves an AWS DeepLens Model.
     *
     * Access Level: Read
     */
    getModel() {
        this.add('deeplens:GetModel');
        return this;
    }
    /**
     * Retrieves an AWS DeepLens Project.
     *
     * Access Level: Read
     */
    getProject() {
        this.add('deeplens:GetProject');
        return this;
    }
    /**
     * Creates a new AWS DeepLens project from a sample project template.
     *
     * Access Level: Write
     */
    importProjectFromTemplate() {
        this.add('deeplens:ImportProjectFromTemplate');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Deployment identifiers.
     *
     * Access Level: List
     */
    listDeployments() {
        this.add('deeplens:ListDeployments');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens device identifiers.
     *
     * Access Level: List
     */
    listDevices() {
        this.add('deeplens:ListDevices');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Model identifiers.
     *
     * Access Level: List
     */
    listModels() {
        this.add('deeplens:ListModels');
        return this;
    }
    /**
     * Retrieves a list of AWS DeepLens Project identifiers.
     *
     * Access Level: List
     */
    listProjects() {
        this.add('deeplens:ListProjects');
        return this;
    }
    /**
     * Begins a device registration workflow for an AWS DeepLens device.
     *
     * Access Level: Write
     */
    registerDevice() {
        this.add('deeplens:RegisterDevice');
        return this;
    }
    /**
     * Removes a deployed AWS DeepLens project from an AWS DeepLens device.
     *
     * Access Level: Write
     */
    removeProject() {
        this.add('deeplens:RemoveProject');
        return this;
    }
    /**
     * Updates an existing AWS DeepLens Project.
     *
     * Access Level: Write
     */
    updateProject() {
        this.add('deeplens:UpdateProject');
        return this;
    }
    /**
     * Adds a resource of type device to the statement
     *
     * @param deviceName - Identifier for the deviceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevice(deviceName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:device/${DeviceName}';
        arn = arn.replace('${DeviceName}', deviceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:project/${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onModel(modelName, account, region, partition) {
        var arn = 'arn:${Partition}:deeplens:${Region}:${Account}:model/${ModelName}';
        arn = arn.replace('${ModelName}', modelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deeplens = Deeplens;
//# sourceMappingURL=data:application/json;base64,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