"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service directconnect
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectconnect.html
 */
class Directconnect extends shared_1.PolicyStatement {
    /**
     * Action provider for service directconnect
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectconnect.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'directconnect';
        this.actions = {
            "AcceptDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html",
                "description": "Accepts a proposal request to attach a virtual private gateway to a Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "AllocateConnectionOnInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html",
                "description": "Creates a hosted connection on an interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "AllocateHostedConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html",
                "description": "Creates a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocatePrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html",
                "description": "Provisions a private virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocatePublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html",
                "description": "Provisions a public virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocateTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html",
                "description": "Provisions a transit virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AssociateConnectionWithLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html",
                "description": "Associates a connection with a LAG.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    },
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "AssociateHostedConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html",
                "description": "Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "AssociateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html",
                "description": "Associates a virtual interface with a specified link aggregation group (LAG) or connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    },
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "ConfirmConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html",
                "description": "Confirm the creation of a hosted connection on an interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "ConfirmPrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html",
                "description": "Accept ownership of a private virtual interface created by another customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "ConfirmPublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html",
                "description": "Accept ownership of a public virtual interface created by another customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "ConfirmTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html",
                "description": "Accept ownership of a transit virtual interface created by another customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "CreateBGPPeer": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html",
                "description": "Creates a BGP peer on the specified virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "CreateConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html",
                "description": "Creates a new connection between the customer network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDirectConnectGateway": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html",
                "description": "Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways.",
                "accessLevel": "Write"
            },
            "CreateDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html",
                "description": "Creates an association between a Direct Connect gateway and a virtual private gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "CreateDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html",
                "description": "Creates a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "CreateInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html",
                "description": "Creates a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html",
                "description": "Creates a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html",
                "description": "Creates a new private virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html",
                "description": "Creates a new public virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html",
                "description": "Creates a new transit virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteBGPPeer": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html",
                "description": "Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "DeleteConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html",
                "description": "Deletes the connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGateway": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html",
                "description": "Deletes the specified Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html",
                "description": "Deletes the association between the specified Direct Connect gateway and virtual private gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html",
                "description": "Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway.",
                "accessLevel": "Write"
            },
            "DeleteInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html",
                "description": "Deletes the specified interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DeleteLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html",
                "description": "Deletes the specified link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html",
                "description": "Deletes a virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "DescribeConnectionLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html",
                "description": "Returns the LOA-CFA for a Connection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeConnections": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html",
                "description": "Displays all connections in this region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                }
            },
            "DescribeConnectionsOnInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html",
                "description": "Return a list of connections that have been provisioned on the given interconnect.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeDirectConnectGatewayAssociationProposals": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html",
                "description": "Describes one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGatewayAssociations": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html",
                "description": "Lists the associations between your Direct Connect gateways and virtual private gateways.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGatewayAttachments": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html",
                "description": "Lists the attachments between your Direct Connect gateways and virtual interfaces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGateways": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html",
                "description": "Lists all your Direct Connect gateways or only the specified Direct Connect gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeHostedConnections": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html",
                "description": "Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeInterconnectLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html",
                "description": "Returns the LOA-CFA for an Interconnect.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeInterconnects": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html",
                "description": "Returns a list of interconnects owned by the AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                }
            },
            "DescribeLags": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html",
                "description": "Describes all your link aggregation groups (LAG) or the specified LAG.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html",
                "description": "Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeLocations": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html",
                "description": "Returns the list of AWS Direct Connect locations in the current AWS region.",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html",
                "description": "Describes the tags associated with the specified AWS Direct Connect resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                }
            },
            "DescribeVirtualGateways": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html",
                "description": "Returns a list of virtual private gateways owned by the AWS account.",
                "accessLevel": "Read"
            },
            "DescribeVirtualInterfaces": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html",
                "description": "Displays all virtual interfaces for an AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                }
            },
            "DisassociateConnectionFromLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html",
                "description": "Disassociates a connection from a link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    },
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "ListVirtualInterfaceTestHistory": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html",
                "description": "Lists the virtual interface failover test history.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "StartBgpFailoverTest": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html",
                "description": "Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "StopBgpFailoverTest": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html",
                "description": "Stops the virtual interface failover test.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html",
                "description": "Adds the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html",
                "description": "Removes one or more tags from the specified AWS Direct Connect resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html",
                "description": "Updates the specified attributes of the Direct Connect gateway association.",
                "accessLevel": "Write"
            },
            "UpdateLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html",
                "description": "Updates the attributes of the specified link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualInterfaceAttributes": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html",
                "description": "Updates the specified attributes of the specified virtual private interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "dxcon": {
                "name": "dxcon",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxcon/${ConnectionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dxlag": {
                "name": "dxlag",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxlag/${LagId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dxvif": {
                "name": "dxvif",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxvif/${VirtualInterfaceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dx-gateway": {
                "name": "dx-gateway",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html",
                "arn": "arn:${Partition}:directconnect::${Account}:dx-gateway/${DirectConnectGatewayId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Accepts a proposal request to attach a virtual private gateway to a Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html
     */
    acceptDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:AcceptDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html
     */
    allocateConnectionOnInterconnect() {
        this.add('directconnect:AllocateConnectionOnInterconnect');
        return this;
    }
    /**
     * Creates a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html
     */
    allocateHostedConnection() {
        this.add('directconnect:AllocateHostedConnection');
        return this;
    }
    /**
     * Provisions a private virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html
     */
    allocatePrivateVirtualInterface() {
        this.add('directconnect:AllocatePrivateVirtualInterface');
        return this;
    }
    /**
     * Provisions a public virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html
     */
    allocatePublicVirtualInterface() {
        this.add('directconnect:AllocatePublicVirtualInterface');
        return this;
    }
    /**
     * Provisions a transit virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html
     */
    allocateTransitVirtualInterface() {
        this.add('directconnect:AllocateTransitVirtualInterface');
        return this;
    }
    /**
     * Associates a connection with a LAG.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html
     */
    associateConnectionWithLag() {
        this.add('directconnect:AssociateConnectionWithLag');
        return this;
    }
    /**
     * Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html
     */
    associateHostedConnection() {
        this.add('directconnect:AssociateHostedConnection');
        return this;
    }
    /**
     * Associates a virtual interface with a specified link aggregation group (LAG) or connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html
     */
    associateVirtualInterface() {
        this.add('directconnect:AssociateVirtualInterface');
        return this;
    }
    /**
     * Confirm the creation of a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html
     */
    confirmConnection() {
        this.add('directconnect:ConfirmConnection');
        return this;
    }
    /**
     * Accept ownership of a private virtual interface created by another customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html
     */
    confirmPrivateVirtualInterface() {
        this.add('directconnect:ConfirmPrivateVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a public virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html
     */
    confirmPublicVirtualInterface() {
        this.add('directconnect:ConfirmPublicVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a transit virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html
     */
    confirmTransitVirtualInterface() {
        this.add('directconnect:ConfirmTransitVirtualInterface');
        return this;
    }
    /**
     * Creates a BGP peer on the specified virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html
     */
    createBGPPeer() {
        this.add('directconnect:CreateBGPPeer');
        return this;
    }
    /**
     * Creates a new connection between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html
     */
    createConnection() {
        this.add('directconnect:CreateConnection');
        return this;
    }
    /**
     * Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html
     */
    createDirectConnectGateway() {
        this.add('directconnect:CreateDirectConnectGateway');
        return this;
    }
    /**
     * Creates an association between a Direct Connect gateway and a virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html
     */
    createDirectConnectGatewayAssociation() {
        this.add('directconnect:CreateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Creates a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html
     */
    createDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:CreateDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html
     */
    createInterconnect() {
        this.add('directconnect:CreateInterconnect');
        return this;
    }
    /**
     * Creates a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html
     */
    createLag() {
        this.add('directconnect:CreateLag');
        return this;
    }
    /**
     * Creates a new private virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html
     */
    createPrivateVirtualInterface() {
        this.add('directconnect:CreatePrivateVirtualInterface');
        return this;
    }
    /**
     * Creates a new public virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html
     */
    createPublicVirtualInterface() {
        this.add('directconnect:CreatePublicVirtualInterface');
        return this;
    }
    /**
     * Creates a new transit virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html
     */
    createTransitVirtualInterface() {
        this.add('directconnect:CreateTransitVirtualInterface');
        return this;
    }
    /**
     * Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html
     */
    deleteBGPPeer() {
        this.add('directconnect:DeleteBGPPeer');
        return this;
    }
    /**
     * Deletes the connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html
     */
    deleteConnection() {
        this.add('directconnect:DeleteConnection');
        return this;
    }
    /**
     * Deletes the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html
     */
    deleteDirectConnectGateway() {
        this.add('directconnect:DeleteDirectConnectGateway');
        return this;
    }
    /**
     * Deletes the association between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html
     */
    deleteDirectConnectGatewayAssociation() {
        this.add('directconnect:DeleteDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html
     */
    deleteDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:DeleteDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Deletes the specified interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html
     */
    deleteInterconnect() {
        this.add('directconnect:DeleteInterconnect');
        return this;
    }
    /**
     * Deletes the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html
     */
    deleteLag() {
        this.add('directconnect:DeleteLag');
        return this;
    }
    /**
     * Deletes a virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html
     */
    deleteVirtualInterface() {
        this.add('directconnect:DeleteVirtualInterface');
        return this;
    }
    /**
     * Returns the LOA-CFA for a Connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html
     */
    describeConnectionLoa() {
        this.add('directconnect:DescribeConnectionLoa');
        return this;
    }
    /**
     * Displays all connections in this region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html
     */
    describeConnections() {
        this.add('directconnect:DescribeConnections');
        return this;
    }
    /**
     * Return a list of connections that have been provisioned on the given interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html
     */
    describeConnectionsOnInterconnect() {
        this.add('directconnect:DescribeConnectionsOnInterconnect');
        return this;
    }
    /**
     * Describes one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html
     */
    describeDirectConnectGatewayAssociationProposals() {
        this.add('directconnect:DescribeDirectConnectGatewayAssociationProposals');
        return this;
    }
    /**
     * Lists the associations between your Direct Connect gateways and virtual private gateways.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html
     */
    describeDirectConnectGatewayAssociations() {
        this.add('directconnect:DescribeDirectConnectGatewayAssociations');
        return this;
    }
    /**
     * Lists the attachments between your Direct Connect gateways and virtual interfaces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html
     */
    describeDirectConnectGatewayAttachments() {
        this.add('directconnect:DescribeDirectConnectGatewayAttachments');
        return this;
    }
    /**
     * Lists all your Direct Connect gateways or only the specified Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html
     */
    describeDirectConnectGateways() {
        this.add('directconnect:DescribeDirectConnectGateways');
        return this;
    }
    /**
     * Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html
     */
    describeHostedConnections() {
        this.add('directconnect:DescribeHostedConnections');
        return this;
    }
    /**
     * Returns the LOA-CFA for an Interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html
     */
    describeInterconnectLoa() {
        this.add('directconnect:DescribeInterconnectLoa');
        return this;
    }
    /**
     * Returns a list of interconnects owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html
     */
    describeInterconnects() {
        this.add('directconnect:DescribeInterconnects');
        return this;
    }
    /**
     * Describes all your link aggregation groups (LAG) or the specified LAG.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html
     */
    describeLags() {
        this.add('directconnect:DescribeLags');
        return this;
    }
    /**
     * Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html
     */
    describeLoa() {
        this.add('directconnect:DescribeLoa');
        return this;
    }
    /**
     * Returns the list of AWS Direct Connect locations in the current AWS region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html
     */
    describeLocations() {
        this.add('directconnect:DescribeLocations');
        return this;
    }
    /**
     * Describes the tags associated with the specified AWS Direct Connect resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html
     */
    describeTags() {
        this.add('directconnect:DescribeTags');
        return this;
    }
    /**
     * Returns a list of virtual private gateways owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html
     */
    describeVirtualGateways() {
        this.add('directconnect:DescribeVirtualGateways');
        return this;
    }
    /**
     * Displays all virtual interfaces for an AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html
     */
    describeVirtualInterfaces() {
        this.add('directconnect:DescribeVirtualInterfaces');
        return this;
    }
    /**
     * Disassociates a connection from a link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html
     */
    disassociateConnectionFromLag() {
        this.add('directconnect:DisassociateConnectionFromLag');
        return this;
    }
    /**
     * Lists the virtual interface failover test history.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html
     */
    listVirtualInterfaceTestHistory() {
        this.add('directconnect:ListVirtualInterfaceTestHistory');
        return this;
    }
    /**
     * Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html
     */
    startBgpFailoverTest() {
        this.add('directconnect:StartBgpFailoverTest');
        return this;
    }
    /**
     * Stops the virtual interface failover test.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html
     */
    stopBgpFailoverTest() {
        this.add('directconnect:StopBgpFailoverTest');
        return this;
    }
    /**
     * Adds the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('directconnect:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from the specified AWS Direct Connect resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('directconnect:UntagResource');
        return this;
    }
    /**
     * Updates the specified attributes of the Direct Connect gateway association.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html
     */
    updateDirectConnectGatewayAssociation() {
        this.add('directconnect:UpdateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Updates the attributes of the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html
     */
    updateLag() {
        this.add('directconnect:UpdateLag');
        return this;
    }
    /**
     * Updates the specified attributes of the specified virtual private interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html
     */
    updateVirtualInterfaceAttributes() {
        this.add('directconnect:UpdateVirtualInterfaceAttributes');
        return this;
    }
    /**
     * Adds a resource of type dxcon to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDxcon(connectionId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxcon/${ConnectionId}';
        arn = arn.replace('${ConnectionId}', connectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxlag to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html
     *
     * @param lagId - Identifier for the lagId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDxlag(lagId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxlag/${LagId}';
        arn = arn.replace('${LagId}', lagId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxvif to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html
     *
     * @param virtualInterfaceId - Identifier for the virtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDxvif(virtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxvif/${VirtualInterfaceId}';
        arn = arn.replace('${VirtualInterfaceId}', virtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dx-gateway to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html
     *
     * @param directConnectGatewayId - Identifier for the directConnectGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDxGateway(directConnectGatewayId, account, partition) {
        var arn = 'arn:${Partition}:directconnect::${Account}:dx-gateway/${DirectConnectGatewayId}';
        arn = arn.replace('${DirectConnectGatewayId}', directConnectGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Directconnect = Directconnect;
//# sourceMappingURL=data:application/json;base64,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