"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service iotevents
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotevents.html
 */
class Iotevents extends shared_1.PolicyStatement {
    /**
     * Action provider for service iotevents
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiotevents.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iotevents';
        this.actions = {
            "BatchPutMessage": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchPutMessage.html",
                "description": "Sends a set of messages to the AWS IoT Events system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "BatchUpdateDetector": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchUpdateDetector.html",
                "description": "Update an detector within the AWS IoT Events system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "CreateDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateDetectorModel.html",
                "description": "Creates a detector model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateInput.html",
                "description": "Creates an input.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteDetectorModel.html",
                "description": "Deletes a detector model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "DeleteInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteInput.html",
                "description": "Deletes an input.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "DescribeDetector": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_DescribeDetector.html",
                "description": "Returns information about the specified detector (instance).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "DescribeDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeDetectorModel.html",
                "description": "Describes a detector model.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "DescribeInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeInput.html",
                "description": "Describes an input.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "DescribeLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeLoggingOptions.html",
                "description": "Retrieves the current settings of the AWS IoT Events logging options.",
                "accessLevel": "Read"
            },
            "ListDetectorModelVersions": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModelVersions.html",
                "description": "Lists all the versions of a detector model. Only the metadata associated with each detector model version is returned.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "ListDetectorModels": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModels.html",
                "description": "Lists the detector models you have created. Only the metadata associated with each detector model is returned.",
                "accessLevel": "List"
            },
            "ListDetectors": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListDetectors.html",
                "description": "Lists detectors (the instances of a detector model).",
                "accessLevel": "List",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "ListInputs": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListInputs.html",
                "description": "Lists the inputs you have created.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListTagsForResource.html",
                "description": "Lists the tags (metadata) which you have assigned to the resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detectorModel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    }
                }
            },
            "PutLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_PutLoggingOptions.html",
                "description": "Sets or updates the AWS IoT Events logging options.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_TagResource.html",
                "description": "Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detectorModel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UntagResource.html",
                "description": "Removes the given tags (metadata) from the resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detectorModel": {
                        "required": false
                    },
                    "input": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDetectorModel": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateDetectorModel.html",
                "description": "Updates a detector model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detectorModel": {
                        "required": true
                    }
                }
            },
            "UpdateInput": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInput.html",
                "description": "Updates an input.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            },
            "UpdateInputRouting": {
                "url": "https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInputRouting.html",
                "description": "Updates input routing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "input": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "detectorModel": {
                "name": "detectorModel",
                "url": "https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html",
                "arn": "arn:${Partition}:iotevents:${Region}:${Account}:detectorModel/${DetectorModelName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "input": {
                "name": "input",
                "url": "https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html",
                "arn": "arn:${Partition}:iotevents:${Region}:${Account}:input/${InputName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Sends a set of messages to the AWS IoT Events system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchPutMessage.html
     */
    batchPutMessage() {
        this.add('iotevents:BatchPutMessage');
        return this;
    }
    /**
     * Update an detector within the AWS IoT Events system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_BatchUpdateDetector.html
     */
    batchUpdateDetector() {
        this.add('iotevents:BatchUpdateDetector');
        return this;
    }
    /**
     * Creates a detector model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateDetectorModel.html
     */
    createDetectorModel() {
        this.add('iotevents:CreateDetectorModel');
        return this;
    }
    /**
     * Creates an input.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_CreateInput.html
     */
    createInput() {
        this.add('iotevents:CreateInput');
        return this;
    }
    /**
     * Deletes a detector model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteDetectorModel.html
     */
    deleteDetectorModel() {
        this.add('iotevents:DeleteDetectorModel');
        return this;
    }
    /**
     * Deletes an input.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DeleteInput.html
     */
    deleteInput() {
        this.add('iotevents:DeleteInput');
        return this;
    }
    /**
     * Returns information about the specified detector (instance).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_DescribeDetector.html
     */
    describeDetector() {
        this.add('iotevents:DescribeDetector');
        return this;
    }
    /**
     * Describes a detector model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeDetectorModel.html
     */
    describeDetectorModel() {
        this.add('iotevents:DescribeDetectorModel');
        return this;
    }
    /**
     * Describes an input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeInput.html
     */
    describeInput() {
        this.add('iotevents:DescribeInput');
        return this;
    }
    /**
     * Retrieves the current settings of the AWS IoT Events logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_DescribeLoggingOptions.html
     */
    describeLoggingOptions() {
        this.add('iotevents:DescribeLoggingOptions');
        return this;
    }
    /**
     * Lists all the versions of a detector model. Only the metadata associated with each detector model version is returned.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModelVersions.html
     */
    listDetectorModelVersions() {
        this.add('iotevents:ListDetectorModelVersions');
        return this;
    }
    /**
     * Lists the detector models you have created. Only the metadata associated with each detector model is returned.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListDetectorModels.html
     */
    listDetectorModels() {
        this.add('iotevents:ListDetectorModels');
        return this;
    }
    /**
     * Lists detectors (the instances of a detector model).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_iotevents-data_ListDetectors.html
     */
    listDetectors() {
        this.add('iotevents:ListDetectors');
        return this;
    }
    /**
     * Lists the inputs you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListInputs.html
     */
    listInputs() {
        this.add('iotevents:ListInputs');
        return this;
    }
    /**
     * Lists the tags (metadata) which you have assigned to the resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('iotevents:ListTagsForResource');
        return this;
    }
    /**
     * Sets or updates the AWS IoT Events logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_PutLoggingOptions.html
     */
    putLoggingOptions() {
        this.add('iotevents:PutLoggingOptions');
        return this;
    }
    /**
     * Adds to or modifies the tags of the given resource. Tags are metadata which can be used to manage a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('iotevents:TagResource');
        return this;
    }
    /**
     * Removes the given tags (metadata) from the resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('iotevents:UntagResource');
        return this;
    }
    /**
     * Updates a detector model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateDetectorModel.html
     */
    updateDetectorModel() {
        this.add('iotevents:UpdateDetectorModel');
        return this;
    }
    /**
     * Updates an input.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInput.html
     */
    updateInput() {
        this.add('iotevents:UpdateInput');
        return this;
    }
    /**
     * Updates input routing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iotevents/latest/apireference/API_UpdateInputRouting.html
     */
    updateInputRouting() {
        this.add('iotevents:UpdateInputRouting');
        return this;
    }
    /**
     * Adds a resource of type detectorModel to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param detectorModelName - Identifier for the detectorModelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDetectorModel(detectorModelName, account, region, partition) {
        var arn = 'arn:${Partition}:iotevents:${Region}:${Account}:detectorModel/${DetectorModelName}';
        arn = arn.replace('${DetectorModelName}', detectorModelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type input to the statement
     *
     * https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-getting-started.html
     *
     * @param inputName - Identifier for the inputName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onInput(inputName, account, region, partition) {
        var arn = 'arn:${Partition}:iotevents:${Region}:${Account}:input/${InputName}';
        arn = arn.replace('${InputName}', inputName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Iotevents = Iotevents;
//# sourceMappingURL=data:application/json;base64,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