"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const shared_1 = require("../shared");
/**
 * Action provider for service wellarchitected
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswell-architectedtool.html
 */
class Wellarchitected extends shared_1.PolicyStatement {
    /**
     * Action provider for service wellarchitected
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswell-architectedtool.html
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'wellarchitected';
        this.actions = {
            "CreateWorkload": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/define-workload.html",
                "description": "Creates a new workload.",
                "accessLevel": "Write"
            },
            "CreateWorkloadShare": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-sharing.html",
                "description": "Shares a workload with another account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workload": {
                        "required": true
                    }
                }
            },
            "DeleteWorkload": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-delete.html",
                "description": "Deletes an existing workload.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workload": {
                        "required": true
                    }
                }
            },
            "GetWorkload": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workload-details.html",
                "description": "Retrieves the specified workload.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workload": {
                        "required": true
                    }
                }
            },
            "ListWorkloads": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-page.html",
                "description": "Lists the workloads in this account.",
                "accessLevel": "List"
            }
        };
        this.resourceTypes = {
            "workload": {
                "name": "workload",
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/iam-auth-access.html",
                "arn": "arn:${Partition}:wellarchitected:${Region}:${Account}:workload/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a new workload.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/define-workload.html
     */
    createWorkload() {
        this.add('wellarchitected:CreateWorkload');
        return this;
    }
    /**
     * Shares a workload with another account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-sharing.html
     */
    createWorkloadShare() {
        this.add('wellarchitected:CreateWorkloadShare');
        return this;
    }
    /**
     * Deletes an existing workload.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-delete.html
     */
    deleteWorkload() {
        this.add('wellarchitected:DeleteWorkload');
        return this;
    }
    /**
     * Retrieves the specified workload.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workload-details.html
     */
    getWorkload() {
        this.add('wellarchitected:GetWorkload');
        return this;
    }
    /**
     * Lists the workloads in this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-page.html
     */
    listWorkloads() {
        this.add('wellarchitected:ListWorkloads');
        return this;
    }
    /**
     * Adds a resource of type workload to the statement
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/iam-auth-access.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkload(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:wellarchitected:${Region}:${Account}:workload/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Wellarchitected = Wellarchitected;
//# sourceMappingURL=data:application/json;base64,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