"use strict";
/**
 * Modifiers can be used to extend the functionality of any `Operator`, except the `null` operator
 */
Object.defineProperty(exports, "__esModule", { value: true });
var OperatorModifier;
(function (OperatorModifier) {
    /**
     * You can add `IfExists` to the end of any condition operator name except the `Null` condition. For example, `StringLikeIfExists`. You do this to say "If the policy key is present in the context of the request, process the key as specified in the policy. If the key is not present, evaluate the condition element as true." Other condition elements in the statement can still result in a nonmatch, but not a missing key when checked with `...IfExists`.
     */
    OperatorModifier["IF_EXISTS"] = "IfExists";
    /**
     * Tests whether the value of every member of the request set is a subset of the condition key set. The condition returns true if every key value in the request matches at least one value in the policy. It also returns true if there are no keys in the request, or if the key values resolve to a null data set, such as an empty string.
     */
    OperatorModifier["FOR_ALL_VALUES"] = "ForAllValues:";
    /**
     * Tests whether at least one member of the set of request values matches at least one member of the set of condition key values. The condition returns true if any one of the key values in the request matches any one of the condition values in the policy. For no matching key or a null dataset, the condition returns false.
     */
    OperatorModifier["FOR_ANY_VALUES"] = "ForAnyValues:";
})(OperatorModifier = exports.OperatorModifier || (exports.OperatorModifier = {}));
/**
 * Use condition operators in the `Condition` element to match the condition key and value in the policy against values in the request context. For more information about the `Condition` element, see [IAM JSON Policy Elements: Condition](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
 *
 * The condition operator that you can use in a policy depends on the condition key you choose. You can choose a global condition key or a service-specific condition key. To learn which condition operator you can use for a global condition key, see [AWS Global Condition Context Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html). To learn which condition operator you can use for a service-specific condition key, see [Actions, Resources, and Condition Keys for AWS Services](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_actions-resources-contextkeys.html) and choose the service that you want to view.
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html
 */
class Operator {
    /**
     * Exact match, case sensitive.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringEquals(...modifiers) {
        return applyModifiers('StringEquals', ...modifiers);
    }
    /**
     * Negated exact match, case sensitive.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotEquals(...modifiers) {
        return applyModifiers('StringNotEquals', ...modifiers);
    }
    /**
     * Exact match, ignore case.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringEqualsIgnoreCase(...modifiers) {
        return applyModifiers('StringEqualsIgnoreCase', ...modifiers);
    }
    /**
     * Exact exact match, ignore case.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotEqualsIgnoreCase(...modifiers) {
        return applyModifiers('StringNotEqualsIgnoreCase', ...modifiers);
    }
    /**
     * Case-sensitive match.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringLike(...modifiers) {
        return applyModifiers('StringLike', ...modifiers);
    }
    /**
     * Negated case-sensitive matching.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotLike(...modifiers) {
        return applyModifiers('StringNotLike', ...modifiers);
    }
    /**
     * Exact match.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericEquals(...modifiers) {
        return applyModifiers('NumericEquals', ...modifiers);
    }
    /**
     * Negated exact match.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericNotEquals(...modifiers) {
        return applyModifiers('NumericNotEquals', ...modifiers);
    }
    /**
     * Match numbers lower than value: `<`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericLessThan(...modifiers) {
        return applyModifiers('NumericLessThan', ...modifiers);
    }
    /**
     * Match numbers lower or equal to value: `<=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericLessThanEquals(...modifiers) {
        return applyModifiers('NumericLessThanEquals', ...modifiers);
    }
    /**
     * Match numbers higher than value: `>`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericGreaterThan(...modifiers) {
        return applyModifiers('NumericGreaterThan', ...modifiers);
    }
    /**
     * Match numbers higher or equal value: `>=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericGreaterThanEquals(...modifiers) {
        return applyModifiers('NumericGreaterThanEquals', ...modifiers);
    }
    /**
     * Match a specific date: `=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateEquals(...modifiers) {
        return applyModifiers('DateEquals', ...modifiers);
    }
    /**
     * Negated match for a specific date: `!=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateNotEquals(...modifiers) {
        return applyModifiers('DateNotEquals', ...modifiers);
    }
    /**
     * Match before a specific date and time: `<`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateLessThan(...modifiers) {
        return applyModifiers('DateLessThan', ...modifiers);
    }
    /**
     * Match at or before a specific date and time: `<=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateLessThanEquals(...modifiers) {
        return applyModifiers('DateLessThanEquals', ...modifiers);
    }
    /**
     * Match after a specific a date and time: `>`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateGreaterThan(...modifiers) {
        return applyModifiers('DateGreaterThan', ...modifiers);
    }
    /**
     * Match at or after a specific date and time: `>=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateGreaterThanEquals(...modifiers) {
        return applyModifiers('DateGreaterThanEquals', ...modifiers);
    }
    /**
     * Boolean match
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static bool(...modifiers) {
        return applyModifiers('Bool', ...modifiers);
    }
    /**
     * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static binaryEquals(...modifiers) {
        return applyModifiers('BinaryEquals', ...modifiers);
    }
    /**
     * Match an IP address or range: `=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static ipAddress(...modifiers) {
        return applyModifiers('IpAddress', ...modifiers);
    }
    /**
     * All IP addresses except the specified IP address or range `!=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static notIpAddress(...modifiers) {
        return applyModifiers('NotIpAddress', ...modifiers);
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnEquals` and `ArnLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnEquals(...modifiers) {
        return applyModifiers('ArnEquals', ...modifiers);
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnNotEquals(...modifiers) {
        return applyModifiers('ArnNotEquals', ...modifiers);
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnLike(...modifiers) {
        return applyModifiers('ArnLike', ...modifiers);
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnNotLike(...modifiers) {
        return applyModifiers('ArnNotLike', ...modifiers);
    }
    /**
     * Check if a key is present at the time of authorization. In the policy statement, use either true (the key doesn't exist — it is null) or false (the key exists and its value is not null).
     */
    static null() {
        return 'Null';
    }
}
exports.Operator = Operator;
/**
 * Applies modifiers to an operator.
 *
 * @param operator - The original operator
 * @param modifiers - Any number of modifiers
 */
function applyModifiers(operator, ...modifiers) {
    if (modifiers.includes(OperatorModifier.IF_EXISTS)) {
        operator += OperatorModifier.IF_EXISTS;
    }
    if (modifiers.includes(OperatorModifier.FOR_ALL_VALUES)) {
        operator = `${OperatorModifier.FOR_ALL_VALUES}${operator}`;
    }
    if (modifiers.includes(OperatorModifier.FOR_ANY_VALUES)) {
        operator = `${OperatorModifier.FOR_ANY_VALUES}${operator}`;
    }
    return operator;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3BlcmF0b3JzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib3BlcmF0b3JzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7R0FFRzs7QUFFSCxJQUFZLGdCQWVYO0FBZkQsV0FBWSxnQkFBZ0I7SUFDMUI7O09BRUc7SUFDSCwwQ0FBc0IsQ0FBQTtJQUV0Qjs7T0FFRztJQUNILG9EQUFnQyxDQUFBO0lBRWhDOztPQUVHO0lBQ0gsb0RBQWdDLENBQUE7QUFDbEMsQ0FBQyxFQWZXLGdCQUFnQixHQUFoQix3QkFBZ0IsS0FBaEIsd0JBQWdCLFFBZTNCO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsTUFBYSxRQUFRO0lBQ25COzs7O09BSUc7SUFDSCxNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsU0FBNkI7UUFDbEQsT0FBTyxjQUFjLENBQUMsY0FBYyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsZUFBZSxDQUFDLEdBQUcsU0FBNkI7UUFDckQsT0FBTyxjQUFjLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxHQUFHLFNBQTZCO1FBQzVELE9BQU8sY0FBYyxDQUFDLHdCQUF3QixFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDaEUsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMseUJBQXlCLENBQUMsR0FBRyxTQUE2QjtRQUMvRCxPQUFPLGNBQWMsQ0FBQywyQkFBMkIsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ25FLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSCxNQUFNLENBQUMsVUFBVSxDQUFDLEdBQUcsU0FBNkI7UUFDaEQsT0FBTyxjQUFjLENBQUMsWUFBWSxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNILE1BQU0sQ0FBQyxhQUFhLENBQUMsR0FBRyxTQUE2QjtRQUNuRCxPQUFPLGNBQWMsQ0FBQyxlQUFlLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxhQUFhLENBQUMsR0FBRyxTQUE2QjtRQUNuRCxPQUFPLGNBQWMsQ0FBQyxlQUFlLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLFNBQTZCO1FBQ3RELE9BQU8sY0FBYyxDQUFDLGtCQUFrQixFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsZUFBZSxDQUFDLEdBQUcsU0FBNkI7UUFDckQsT0FBTyxjQUFjLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxHQUFHLFNBQTZCO1FBQzNELE9BQU8sY0FBYyxDQUFDLHVCQUF1QixFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsa0JBQWtCLENBQUMsR0FBRyxTQUE2QjtRQUN4RCxPQUFPLGNBQWMsQ0FBQyxvQkFBb0IsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsTUFBTSxDQUFDLHdCQUF3QixDQUFDLEdBQUcsU0FBNkI7UUFDOUQsT0FBTyxjQUFjLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBRyxTQUE2QjtRQUNoRCxPQUFPLGNBQWMsQ0FBQyxZQUFZLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxhQUFhLENBQUMsR0FBRyxTQUE2QjtRQUNuRCxPQUFPLGNBQWMsQ0FBQyxlQUFlLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBRyxTQUE2QjtRQUNsRCxPQUFPLGNBQWMsQ0FBQyxjQUFjLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLFNBQTZCO1FBQ3hELE9BQU8sY0FBYyxDQUFDLG9CQUFvQixFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsZUFBZSxDQUFDLEdBQUcsU0FBNkI7UUFDckQsT0FBTyxjQUFjLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxHQUFHLFNBQTZCO1FBQzNELE9BQU8sY0FBYyxDQUFDLHVCQUF1QixFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsU0FBNkI7UUFDMUMsT0FBTyxjQUFjLENBQUMsTUFBTSxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsU0FBNkI7UUFDbEQsT0FBTyxjQUFjLENBQUMsY0FBYyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBNkI7UUFDL0MsT0FBTyxjQUFjLENBQUMsV0FBVyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsU0FBNkI7UUFDbEQsT0FBTyxjQUFjLENBQUMsY0FBYyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0gsTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQTZCO1FBQy9DLE9BQU8sY0FBYyxDQUFDLFdBQVcsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNILE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBRyxTQUE2QjtRQUNsRCxPQUFPLGNBQWMsQ0FBQyxjQUFjLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLFNBQTZCO1FBQzdDLE9BQU8sY0FBYyxDQUFDLFNBQVMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNILE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBRyxTQUE2QjtRQUNoRCxPQUFPLGNBQWMsQ0FBQyxZQUFZLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxNQUFNLENBQUMsSUFBSTtRQUNULE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7Q0FDRjtBQW5RRCw0QkFtUUM7QUFFRDs7Ozs7R0FLRztBQUNILFNBQVMsY0FBYyxDQUNyQixRQUFnQixFQUNoQixHQUFHLFNBQTZCO0lBRWhDLElBQUksU0FBUyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUNsRCxRQUFRLElBQUksZ0JBQWdCLENBQUMsU0FBUyxDQUFDO0tBQ3hDO0lBQ0QsSUFBSSxTQUFTLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxFQUFFO1FBQ3ZELFFBQVEsR0FBRyxHQUFHLGdCQUFnQixDQUFDLGNBQWMsR0FBRyxRQUFRLEVBQUUsQ0FBQztLQUM1RDtJQUNELElBQUksU0FBUyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsRUFBRTtRQUN2RCxRQUFRLEdBQUcsR0FBRyxnQkFBZ0IsQ0FBQyxjQUFjLEdBQUcsUUFBUSxFQUFFLENBQUM7S0FDNUQ7SUFDRCxPQUFPLFFBQVEsQ0FBQztBQUNsQixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBNb2RpZmllcnMgY2FuIGJlIHVzZWQgdG8gZXh0ZW5kIHRoZSBmdW5jdGlvbmFsaXR5IG9mIGFueSBgT3BlcmF0b3JgLCBleGNlcHQgdGhlIGBudWxsYCBvcGVyYXRvclxuICovXG5cbmV4cG9ydCBlbnVtIE9wZXJhdG9yTW9kaWZpZXIge1xuICAvKipcbiAgICogWW91IGNhbiBhZGQgYElmRXhpc3RzYCB0byB0aGUgZW5kIG9mIGFueSBjb25kaXRpb24gb3BlcmF0b3IgbmFtZSBleGNlcHQgdGhlIGBOdWxsYCBjb25kaXRpb24uIEZvciBleGFtcGxlLCBgU3RyaW5nTGlrZUlmRXhpc3RzYC4gWW91IGRvIHRoaXMgdG8gc2F5IFwiSWYgdGhlIHBvbGljeSBrZXkgaXMgcHJlc2VudCBpbiB0aGUgY29udGV4dCBvZiB0aGUgcmVxdWVzdCwgcHJvY2VzcyB0aGUga2V5IGFzIHNwZWNpZmllZCBpbiB0aGUgcG9saWN5LiBJZiB0aGUga2V5IGlzIG5vdCBwcmVzZW50LCBldmFsdWF0ZSB0aGUgY29uZGl0aW9uIGVsZW1lbnQgYXMgdHJ1ZS5cIiBPdGhlciBjb25kaXRpb24gZWxlbWVudHMgaW4gdGhlIHN0YXRlbWVudCBjYW4gc3RpbGwgcmVzdWx0IGluIGEgbm9ubWF0Y2gsIGJ1dCBub3QgYSBtaXNzaW5nIGtleSB3aGVuIGNoZWNrZWQgd2l0aCBgLi4uSWZFeGlzdHNgLlxuICAgKi9cbiAgSUZfRVhJU1RTID0gJ0lmRXhpc3RzJyxcblxuICAvKipcbiAgICogVGVzdHMgd2hldGhlciB0aGUgdmFsdWUgb2YgZXZlcnkgbWVtYmVyIG9mIHRoZSByZXF1ZXN0IHNldCBpcyBhIHN1YnNldCBvZiB0aGUgY29uZGl0aW9uIGtleSBzZXQuIFRoZSBjb25kaXRpb24gcmV0dXJucyB0cnVlIGlmIGV2ZXJ5IGtleSB2YWx1ZSBpbiB0aGUgcmVxdWVzdCBtYXRjaGVzIGF0IGxlYXN0IG9uZSB2YWx1ZSBpbiB0aGUgcG9saWN5LiBJdCBhbHNvIHJldHVybnMgdHJ1ZSBpZiB0aGVyZSBhcmUgbm8ga2V5cyBpbiB0aGUgcmVxdWVzdCwgb3IgaWYgdGhlIGtleSB2YWx1ZXMgcmVzb2x2ZSB0byBhIG51bGwgZGF0YSBzZXQsIHN1Y2ggYXMgYW4gZW1wdHkgc3RyaW5nLlxuICAgKi9cbiAgRk9SX0FMTF9WQUxVRVMgPSAnRm9yQWxsVmFsdWVzOicsXG5cbiAgLyoqXG4gICAqIFRlc3RzIHdoZXRoZXIgYXQgbGVhc3Qgb25lIG1lbWJlciBvZiB0aGUgc2V0IG9mIHJlcXVlc3QgdmFsdWVzIG1hdGNoZXMgYXQgbGVhc3Qgb25lIG1lbWJlciBvZiB0aGUgc2V0IG9mIGNvbmRpdGlvbiBrZXkgdmFsdWVzLiBUaGUgY29uZGl0aW9uIHJldHVybnMgdHJ1ZSBpZiBhbnkgb25lIG9mIHRoZSBrZXkgdmFsdWVzIGluIHRoZSByZXF1ZXN0IG1hdGNoZXMgYW55IG9uZSBvZiB0aGUgY29uZGl0aW9uIHZhbHVlcyBpbiB0aGUgcG9saWN5LiBGb3Igbm8gbWF0Y2hpbmcga2V5IG9yIGEgbnVsbCBkYXRhc2V0LCB0aGUgY29uZGl0aW9uIHJldHVybnMgZmFsc2UuXG4gICAqL1xuICBGT1JfQU5ZX1ZBTFVFUyA9ICdGb3JBbnlWYWx1ZXM6Jyxcbn1cblxuLyoqXG4gKiBVc2UgY29uZGl0aW9uIG9wZXJhdG9ycyBpbiB0aGUgYENvbmRpdGlvbmAgZWxlbWVudCB0byBtYXRjaCB0aGUgY29uZGl0aW9uIGtleSBhbmQgdmFsdWUgaW4gdGhlIHBvbGljeSBhZ2FpbnN0IHZhbHVlcyBpbiB0aGUgcmVxdWVzdCBjb250ZXh0LiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgYENvbmRpdGlvbmAgZWxlbWVudCwgc2VlIFtJQU0gSlNPTiBQb2xpY3kgRWxlbWVudHM6IENvbmRpdGlvbl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb24uaHRtbCkuXG4gKlxuICogVGhlIGNvbmRpdGlvbiBvcGVyYXRvciB0aGF0IHlvdSBjYW4gdXNlIGluIGEgcG9saWN5IGRlcGVuZHMgb24gdGhlIGNvbmRpdGlvbiBrZXkgeW91IGNob29zZS4gWW91IGNhbiBjaG9vc2UgYSBnbG9iYWwgY29uZGl0aW9uIGtleSBvciBhIHNlcnZpY2Utc3BlY2lmaWMgY29uZGl0aW9uIGtleS4gVG8gbGVhcm4gd2hpY2ggY29uZGl0aW9uIG9wZXJhdG9yIHlvdSBjYW4gdXNlIGZvciBhIGdsb2JhbCBjb25kaXRpb24ga2V5LCBzZWUgW0FXUyBHbG9iYWwgQ29uZGl0aW9uIENvbnRleHQgS2V5c10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sKS4gVG8gbGVhcm4gd2hpY2ggY29uZGl0aW9uIG9wZXJhdG9yIHlvdSBjYW4gdXNlIGZvciBhIHNlcnZpY2Utc3BlY2lmaWMgY29uZGl0aW9uIGtleSwgc2VlIFtBY3Rpb25zLCBSZXNvdXJjZXMsIGFuZCBDb25kaXRpb24gS2V5cyBmb3IgQVdTIFNlcnZpY2VzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2FjdGlvbnMtcmVzb3VyY2VzLWNvbnRleHRrZXlzLmh0bWwpIGFuZCBjaG9vc2UgdGhlIHNlcnZpY2UgdGhhdCB5b3Ugd2FudCB0byB2aWV3LlxuICpcbiAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sXG4gKi9cbmV4cG9ydCBjbGFzcyBPcGVyYXRvciB7XG4gIC8qKlxuICAgKiBFeGFjdCBtYXRjaCwgY2FzZSBzZW5zaXRpdmUuXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBzdHJpbmdFcXVhbHMoLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ1N0cmluZ0VxdWFscycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTmVnYXRlZCBleGFjdCBtYXRjaCwgY2FzZSBzZW5zaXRpdmUuXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBzdHJpbmdOb3RFcXVhbHMoLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ1N0cmluZ05vdEVxdWFscycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogRXhhY3QgbWF0Y2gsIGlnbm9yZSBjYXNlLlxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgc3RyaW5nRXF1YWxzSWdub3JlQ2FzZSguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnU3RyaW5nRXF1YWxzSWdub3JlQ2FzZScsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogRXhhY3QgZXhhY3QgbWF0Y2gsIGlnbm9yZSBjYXNlLlxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgc3RyaW5nTm90RXF1YWxzSWdub3JlQ2FzZSguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnU3RyaW5nTm90RXF1YWxzSWdub3JlQ2FzZScsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogQ2FzZS1zZW5zaXRpdmUgbWF0Y2guXG4gICAqXG4gICAqIFRoZSB2YWx1ZXMgY2FuIGluY2x1ZGUgYSBtdWx0aS1jaGFyYWN0ZXIgbWF0Y2ggd2lsZGNhcmQgKGAqYCkgb3IgYSBzaW5nbGUtY2hhcmFjdGVyIG1hdGNoIHdpbGRjYXJkIChgP2ApIGFueXdoZXJlIGluIHRoZSBzdHJpbmcuXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBzdHJpbmdMaWtlKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdTdHJpbmdMaWtlJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBOZWdhdGVkIGNhc2Utc2Vuc2l0aXZlIG1hdGNoaW5nLlxuICAgKlxuICAgKiBUaGUgdmFsdWVzIGNhbiBpbmNsdWRlIGEgbXVsdGktY2hhcmFjdGVyIG1hdGNoIHdpbGRjYXJkIChgKmApIG9yIGEgc2luZ2xlLWNoYXJhY3RlciBtYXRjaCB3aWxkY2FyZCAoYD9gKSBhbnl3aGVyZSBpbiB0aGUgc3RyaW5nLlxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgc3RyaW5nTm90TGlrZSguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnU3RyaW5nTm90TGlrZScsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogRXhhY3QgbWF0Y2guXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBudW1lcmljRXF1YWxzKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdOdW1lcmljRXF1YWxzJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBOZWdhdGVkIGV4YWN0IG1hdGNoLlxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgbnVtZXJpY05vdEVxdWFscyguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnTnVtZXJpY05vdEVxdWFscycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2ggbnVtYmVycyBsb3dlciB0aGFuIHZhbHVlOiBgPGBcbiAgICpcbiAgICogQHBhcmFtIG1vZGlmaWVycyAtIFBhc3MgaW4gYW55IG51bWJlciBvZiBgT3BlcmF0b3JNb2RpZmllcmBzLlxuICAgKi9cbiAgc3RhdGljIG51bWVyaWNMZXNzVGhhbiguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnTnVtZXJpY0xlc3NUaGFuJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaCBudW1iZXJzIGxvd2VyIG9yIGVxdWFsIHRvIHZhbHVlOiBgPD1gXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBudW1lcmljTGVzc1RoYW5FcXVhbHMoLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ051bWVyaWNMZXNzVGhhbkVxdWFscycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2ggbnVtYmVycyBoaWdoZXIgdGhhbiB2YWx1ZTogYD5gXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBudW1lcmljR3JlYXRlclRoYW4oLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ051bWVyaWNHcmVhdGVyVGhhbicsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2ggbnVtYmVycyBoaWdoZXIgb3IgZXF1YWwgdmFsdWU6IGA+PWBcbiAgICpcbiAgICogQHBhcmFtIG1vZGlmaWVycyAtIFBhc3MgaW4gYW55IG51bWJlciBvZiBgT3BlcmF0b3JNb2RpZmllcmBzLlxuICAgKi9cbiAgc3RhdGljIG51bWVyaWNHcmVhdGVyVGhhbkVxdWFscyguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnTnVtZXJpY0dyZWF0ZXJUaGFuRXF1YWxzJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaCBhIHNwZWNpZmljIGRhdGU6IGA9YFxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgZGF0ZUVxdWFscyguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnRGF0ZUVxdWFscycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTmVnYXRlZCBtYXRjaCBmb3IgYSBzcGVjaWZpYyBkYXRlOiBgIT1gXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBkYXRlTm90RXF1YWxzKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdEYXRlTm90RXF1YWxzJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaCBiZWZvcmUgYSBzcGVjaWZpYyBkYXRlIGFuZCB0aW1lOiBgPGBcbiAgICpcbiAgICogQHBhcmFtIG1vZGlmaWVycyAtIFBhc3MgaW4gYW55IG51bWJlciBvZiBgT3BlcmF0b3JNb2RpZmllcmBzLlxuICAgKi9cbiAgc3RhdGljIGRhdGVMZXNzVGhhbiguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnRGF0ZUxlc3NUaGFuJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaCBhdCBvciBiZWZvcmUgYSBzcGVjaWZpYyBkYXRlIGFuZCB0aW1lOiBgPD1gXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBkYXRlTGVzc1RoYW5FcXVhbHMoLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ0RhdGVMZXNzVGhhbkVxdWFscycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2ggYWZ0ZXIgYSBzcGVjaWZpYyBhIGRhdGUgYW5kIHRpbWU6IGA+YFxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgZGF0ZUdyZWF0ZXJUaGFuKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdEYXRlR3JlYXRlclRoYW4nLCAuLi5tb2RpZmllcnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoIGF0IG9yIGFmdGVyIGEgc3BlY2lmaWMgZGF0ZSBhbmQgdGltZTogYD49YFxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgZGF0ZUdyZWF0ZXJUaGFuRXF1YWxzKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdEYXRlR3JlYXRlclRoYW5FcXVhbHMnLCAuLi5tb2RpZmllcnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEJvb2xlYW4gbWF0Y2hcbiAgICpcbiAgICogQHBhcmFtIG1vZGlmaWVycyAtIFBhc3MgaW4gYW55IG51bWJlciBvZiBgT3BlcmF0b3JNb2RpZmllcmBzLlxuICAgKi9cbiAgc3RhdGljIGJvb2woLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ0Jvb2wnLCAuLi5tb2RpZmllcnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBCaW5hcnlFcXVhbHMgY29uZGl0aW9uIG9wZXJhdG9yIGxldHMgeW91IGNvbnN0cnVjdCBDb25kaXRpb24gZWxlbWVudHMgdGhhdCB0ZXN0IGtleSB2YWx1ZXMgdGhhdCBhcmUgaW4gYmluYXJ5IGZvcm1hdC4gSXQgY29tcGFyZXMgdGhlIHZhbHVlIG9mIHRoZSBzcGVjaWZpZWQga2V5IGJ5dGUgZm9yIGJ5dGUgYWdhaW5zdCBhIFtiYXNlLTY0XShodHRwczovL2VuLndpa2lwZWRpYS5vcmcvd2lraS9CYXNlNjQpIGVuY29kZWQgcmVwcmVzZW50YXRpb24gb2YgdGhlIGJpbmFyeSB2YWx1ZSBpbiB0aGUgcG9saWN5LlxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgYmluYXJ5RXF1YWxzKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdCaW5hcnlFcXVhbHMnLCAuLi5tb2RpZmllcnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoIGFuIElQIGFkZHJlc3Mgb3IgcmFuZ2U6IGA9YFxuICAgKlxuICAgKiBAcGFyYW0gbW9kaWZpZXJzIC0gUGFzcyBpbiBhbnkgbnVtYmVyIG9mIGBPcGVyYXRvck1vZGlmaWVyYHMuXG4gICAqL1xuICBzdGF0aWMgaXBBZGRyZXNzKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdJcEFkZHJlc3MnLCAuLi5tb2RpZmllcnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFsbCBJUCBhZGRyZXNzZXMgZXhjZXB0IHRoZSBzcGVjaWZpZWQgSVAgYWRkcmVzcyBvciByYW5nZSBgIT1gXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBub3RJcEFkZHJlc3MoLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ05vdElwQWRkcmVzcycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2ggb2YgdGhlIEFSTiwgY2FzZSBzZW5zaXRpdmUuXG4gICAqXG4gICAqIEVhY2ggb2YgdGhlIHNpeCBjb2xvbi1kZWxpbWl0ZWQgY29tcG9uZW50cyBvZiB0aGUgQVJOIGlzIGNoZWNrZWQgc2VwYXJhdGVseSBhbmQgZWFjaCBjYW4gaW5jbHVkZSBhIG11bHRpLWNoYXJhY3RlciBtYXRjaCB3aWxkY2FyZCAoYCpgKSBvciBhIHNpbmdsZS1jaGFyYWN0ZXIgbWF0Y2ggd2lsZGNhcmQgKGA/YCkuXG4gICAqXG4gICAqIGBBcm5FcXVhbHNgIGFuZCBgQXJuTGlrZWAgYmVoYXZlIGlkZW50aWNhbC5cbiAgICpcbiAgICogQHBhcmFtIG1vZGlmaWVycyAtIFBhc3MgaW4gYW55IG51bWJlciBvZiBgT3BlcmF0b3JNb2RpZmllcmBzLlxuICAgKi9cbiAgc3RhdGljIGFybkVxdWFscyguLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXSkge1xuICAgIHJldHVybiBhcHBseU1vZGlmaWVycygnQXJuRXF1YWxzJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBOZWdhdGVkIG1hdGNoIG9mIHRoZSBBUk4sIGNhc2Ugc2Vuc2l0aXZlLlxuICAgKlxuICAgKiBFYWNoIG9mIHRoZSBzaXggY29sb24tZGVsaW1pdGVkIGNvbXBvbmVudHMgb2YgdGhlIEFSTiBpcyBjaGVja2VkIHNlcGFyYXRlbHkgYW5kIGVhY2ggY2FuIGluY2x1ZGUgYSBtdWx0aS1jaGFyYWN0ZXIgbWF0Y2ggd2lsZGNhcmQgKGAqYCkgb3IgYSBzaW5nbGUtY2hhcmFjdGVyIG1hdGNoIHdpbGRjYXJkIChgP2ApLlxuICAgKlxuICAgKiBgQXJuTm90RXF1YWxzYCBhbmQgYEFybk5vdExpa2VgIGJlaGF2ZSBpZGVudGljYWwuXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBhcm5Ob3RFcXVhbHMoLi4ubW9kaWZpZXJzOiBPcGVyYXRvck1vZGlmaWVyW10pIHtcbiAgICByZXR1cm4gYXBwbHlNb2RpZmllcnMoJ0Fybk5vdEVxdWFscycsIC4uLm1vZGlmaWVycyk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2ggb2YgdGhlIEFSTiwgY2FzZSBzZW5zaXRpdmUuXG4gICAqXG4gICAqIEVhY2ggb2YgdGhlIHNpeCBjb2xvbi1kZWxpbWl0ZWQgY29tcG9uZW50cyBvZiB0aGUgQVJOIGlzIGNoZWNrZWQgc2VwYXJhdGVseSBhbmQgZWFjaCBjYW4gaW5jbHVkZSBhIG11bHRpLWNoYXJhY3RlciBtYXRjaCB3aWxkY2FyZCAoYCpgKSBvciBhIHNpbmdsZS1jaGFyYWN0ZXIgbWF0Y2ggd2lsZGNhcmQgKGA/YCkuXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBhcm5MaWtlKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdBcm5MaWtlJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBOZWdhdGVkIG1hdGNoIG9mIHRoZSBBUk4sIGNhc2Ugc2Vuc2l0aXZlLlxuICAgKlxuICAgKiBFYWNoIG9mIHRoZSBzaXggY29sb24tZGVsaW1pdGVkIGNvbXBvbmVudHMgb2YgdGhlIEFSTiBpcyBjaGVja2VkIHNlcGFyYXRlbHkgYW5kIGVhY2ggY2FuIGluY2x1ZGUgYSBtdWx0aS1jaGFyYWN0ZXIgbWF0Y2ggd2lsZGNhcmQgKGAqYCkgb3IgYSBzaW5nbGUtY2hhcmFjdGVyIG1hdGNoIHdpbGRjYXJkIChgP2ApLlxuICAgKlxuICAgKiBgQXJuTm90RXF1YWxzYCBhbmQgYEFybk5vdExpa2VgIGJlaGF2ZSBpZGVudGljYWwuXG4gICAqXG4gICAqIEBwYXJhbSBtb2RpZmllcnMgLSBQYXNzIGluIGFueSBudW1iZXIgb2YgYE9wZXJhdG9yTW9kaWZpZXJgcy5cbiAgICovXG4gIHN0YXRpYyBhcm5Ob3RMaWtlKC4uLm1vZGlmaWVyczogT3BlcmF0b3JNb2RpZmllcltdKSB7XG4gICAgcmV0dXJuIGFwcGx5TW9kaWZpZXJzKCdBcm5Ob3RMaWtlJywgLi4ubW9kaWZpZXJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBpZiBhIGtleSBpcyBwcmVzZW50IGF0IHRoZSB0aW1lIG9mIGF1dGhvcml6YXRpb24uIEluIHRoZSBwb2xpY3kgc3RhdGVtZW50LCB1c2UgZWl0aGVyIHRydWUgKHRoZSBrZXkgZG9lc24ndCBleGlzdCDigJQgaXQgaXMgbnVsbCkgb3IgZmFsc2UgKHRoZSBrZXkgZXhpc3RzIGFuZCBpdHMgdmFsdWUgaXMgbm90IG51bGwpLlxuICAgKi9cbiAgc3RhdGljIG51bGwoKSB7XG4gICAgcmV0dXJuICdOdWxsJztcbiAgfVxufVxuXG4vKipcbiAqIEFwcGxpZXMgbW9kaWZpZXJzIHRvIGFuIG9wZXJhdG9yLlxuICpcbiAqIEBwYXJhbSBvcGVyYXRvciAtIFRoZSBvcmlnaW5hbCBvcGVyYXRvclxuICogQHBhcmFtIG1vZGlmaWVycyAtIEFueSBudW1iZXIgb2YgbW9kaWZpZXJzXG4gKi9cbmZ1bmN0aW9uIGFwcGx5TW9kaWZpZXJzKFxuICBvcGVyYXRvcjogc3RyaW5nLFxuICAuLi5tb2RpZmllcnM6IE9wZXJhdG9yTW9kaWZpZXJbXVxuKTogc3RyaW5nIHtcbiAgaWYgKG1vZGlmaWVycy5pbmNsdWRlcyhPcGVyYXRvck1vZGlmaWVyLklGX0VYSVNUUykpIHtcbiAgICBvcGVyYXRvciArPSBPcGVyYXRvck1vZGlmaWVyLklGX0VYSVNUUztcbiAgfVxuICBpZiAobW9kaWZpZXJzLmluY2x1ZGVzKE9wZXJhdG9yTW9kaWZpZXIuRk9SX0FMTF9WQUxVRVMpKSB7XG4gICAgb3BlcmF0b3IgPSBgJHtPcGVyYXRvck1vZGlmaWVyLkZPUl9BTExfVkFMVUVTfSR7b3BlcmF0b3J9YDtcbiAgfVxuICBpZiAobW9kaWZpZXJzLmluY2x1ZGVzKE9wZXJhdG9yTW9kaWZpZXIuRk9SX0FOWV9WQUxVRVMpKSB7XG4gICAgb3BlcmF0b3IgPSBgJHtPcGVyYXRvck1vZGlmaWVyLkZPUl9BTllfVkFMVUVTfSR7b3BlcmF0b3J9YDtcbiAgfVxuICByZXR1cm4gb3BlcmF0b3I7XG59XG4iXX0=