/// @file
/// @author David Pilger <dpilger26@gmail.com>
/// [GitHub Repository](https://github.com/dpilger26/NumCpp)
///
/// License
/// Copyright 2018-2022 David Pilger
///
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this
/// software and associated documentation files(the "Software"), to deal in the Software
/// without restriction, including without limitation the rights to use, copy, modify,
/// merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
/// permit persons to whom the Software is furnished to do so, subject to the following
/// conditions :
///
/// The above copyright notice and this permission notice shall be included in all copies
/// or substantial portions of the Software.
///
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
/// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
/// PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
/// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
/// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
/// DEALINGS IN THE SOFTWARE.
///
/// Description
/// Special Functions
///
#pragma once

#ifndef NUMCPP_NO_USE_BOOST

#include <string>

#include "boost/math/special_functions/prime.hpp"

#include "NumCpp/Core/Internal/Error.hpp"
#include "NumCpp/Core/Internal/StlAlgorithms.hpp"
#include "NumCpp/Core/Types.hpp"
#include "NumCpp/NdArray.hpp"

namespace nc
{
    namespace special
    {
        //============================================================================
        // Method Description:
        /// The function prime provides fast table lookup to the first 10000 prime numbers
        /// (starting from 2 as the zeroth prime: as 1 isn't terribly useful in practice).
        /// NOTE: Use of this function requires using the Boost includes.
        ///
        /// @param n: the nth prime number to return
        /// @return uint32
        ///
        inline uint32 prime(uint32 n)
        {
            if (n > boost::math::max_prime)
            {
                THROW_INVALID_ARGUMENT_ERROR("input n must be less than or equal to " +
                                             std::to_string(boost::math::max_prime));
            }

            return boost::math::prime(n);
        }

        //============================================================================
        // Method Description:
        /// The function prime provides fast table lookup to the first 10000 prime numbers
        /// (starting from 2 as the zeroth prime: as 1 isn't terribly useful in practice).
        /// NOTE: Use of this function requires using the Boost includes.
        ///
        /// @param inArray
        /// @return NdArray<uint32>
        ///
        inline NdArray<uint32> prime(const NdArray<uint32>& inArray)
        {
            NdArray<uint32> returnArray(inArray.shape());

            stl_algorithms::transform(inArray.cbegin(),
                                      inArray.cend(),
                                      returnArray.begin(),
                                      [](uint32 inValue) -> uint32 { return prime(inValue); });

            return returnArray;
        }
    } // namespace special
} // namespace nc

#endif // #ifndef NUMCPP_NO_USE_BOOST
