"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const metadata_schema_1 = require("./metadata-schema");
const assembly = require("./schema");
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
/**
 * Protocol utility class.
 */
class Manifest {
    constructor() { }
    /**
     * Save manifest to file.
     *
     * @param manifest - manifest.
     */
    static save(manifest, filePath) {
        fs.writeFileSync(filePath, JSON.stringify(manifest, undefined, 2));
    }
    /**
     * Load manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static load(filePath) {
        const raw = JSON.parse(fs.readFileSync(filePath, 'UTF-8'));
        Manifest.patchStackTags(raw);
        Manifest.validate(raw);
        return raw;
    }
    /**
     * Fetch the current schema version number.
     */
    static version() {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require('../schema/cloud-assembly.version.json').version;
    }
    static validate(manifest) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, Manifest.schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTags(manifest) {
        for (const artifact of Object.values(manifest.artifacts || [])) {
            if (artifact.type === assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                for (const metadataEntries of Object.values(artifact.metadata || [])) {
                    for (const metadataEntry of metadataEntries) {
                        if (metadataEntry.type === metadata_schema_1.ArtifactMetadataEntryType.STACK_TAGS && metadataEntry.data) {
                            const metadataAny = metadataEntry;
                            metadataAny.data = metadataAny.data.map((t) => {
                                return { key: t.Key, value: t.Value };
                            });
                        }
                    }
                }
            }
        }
    }
}
exports.Manifest = Manifest;
// eslint-disable-next-line @typescript-eslint/no-require-imports
Manifest.schema = require('../schema/cloud-assembly.schema.json');
//# sourceMappingURL=data:application/json;base64,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