"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const os = require("os");
const path = require("path");
const semver = require("semver");
const lib_1 = require("../lib");
const FIXTURES = path.join(__dirname, 'fixtures');
function fixture(name) {
    return path.join(FIXTURES, name, 'manifest.json');
}
function clone(obj) {
    return JSON.parse(JSON.stringify(obj));
}
function removeStringKeys(obj, keys) {
    function _recurse(o) {
        for (const prop in o) {
            if (keys.includes(prop) && typeof o[prop] === 'string') {
                delete o[prop];
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    const cloned = clone(obj);
    _recurse(cloned);
    return cloned;
}
test('manifest save', () => {
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'schema-tests'));
    const manifestFile = path.join(outdir, 'manifest.json');
    const assemblyManifest = {
        version: 'version',
    };
    lib_1.Manifest.save(assemblyManifest, manifestFile);
    const saved = JSON.parse(fs.readFileSync(manifestFile, 'UTF-8'));
    expect(saved).toEqual(assemblyManifest);
});
test('if this test fails, run "yarn update-schema"', () => {
    // when we compare schemas we ignore changes the
    // description that is generated from the ts docstrings.
    const docStringFields = [
        'description',
    ];
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const schema = require('../scripts/update-schema.js');
    const expected = removeStringKeys(schema.generate(), docStringFields);
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const actual = removeStringKeys(require('../schema/cloud-assembly.schema.json'), docStringFields);
    try {
        expect(actual).toEqual(expected);
    }
    catch (err) {
        // I couldn't for the life of me figure out how to provide additional error message
        // to jest...any ideas?
        err.message = `Whoops, Looks like the schema has changed. Did you forget to run 'yarn update-schema'?\n\n${err.message}`;
        throw err;
    }
});
test('manifest load', () => {
    const loaded = lib_1.Manifest.load(fixture('only-version'));
    expect(loaded).toMatchSnapshot();
});
test('manifest load fails for invalid nested property', () => {
    expect(() => lib_1.Manifest.load(fixture('invalid-nested-property'))).toThrow(/Invalid assembly manifest/);
});
test('manifest load fails for invalid artifact type', () => {
    expect(() => lib_1.Manifest.load(fixture('invalid-artifact-type'))).toThrow(/Invalid assembly manifest/);
});
test('manifest load fails on higher major version', () => {
    expect(() => lib_1.Manifest.load(fixture('high-version'))).toThrow(/Cloud assembly schema version mismatch/);
});
// once we start introducing minor version bumps that are considered
// non breaking, this test can be removed.
test('manifest load fails on higher minor version', () => {
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'schema-tests'));
    const manifestFile = path.join(outdir, 'manifest.json');
    const newVersion = semver.inc(lib_1.Manifest.version(), 'minor');
    expect(newVersion).toBeTruthy();
    if (newVersion) {
        const assemblyManifest = {
            version: newVersion,
        };
        lib_1.Manifest.save(assemblyManifest, manifestFile);
        expect(() => lib_1.Manifest.load(manifestFile)).toThrow(/Cloud assembly schema version mismatch/);
    }
});
// once we start introducing patch version bumps that are considered
// non breaking, this test can be removed.
test('manifest load fails on higher patch version', () => {
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'schema-tests'));
    const manifestFile = path.join(outdir, 'manifest.json');
    const newVersion = semver.inc(lib_1.Manifest.version(), 'patch');
    expect(newVersion).toBeTruthy();
    if (newVersion) {
        const assemblyManifest = {
            version: newVersion,
        };
        lib_1.Manifest.save(assemblyManifest, manifestFile);
        expect(() => lib_1.Manifest.load(manifestFile)).toThrow(/Cloud assembly schema version mismatch/);
    }
});
test('manifest load fails on invalid version', () => {
    expect(() => lib_1.Manifest.load(fixture('invalid-version'))).toThrow(/Invalid semver string/);
});
test('manifest load succeeds on unknown properties', () => {
    const manifest = lib_1.Manifest.load(fixture('unknown-property'));
    expect(manifest.version).toEqual('0.0.0');
});
test('stack-tags are deserialized properly', () => {
    var _a, _b, _c;
    const m = lib_1.Manifest.load(fixture('with-stack-tags'));
    if ((_c = (_b = (_a = m.artifacts) === null || _a === void 0 ? void 0 : _a.stack) === null || _b === void 0 ? void 0 : _b.metadata) === null || _c === void 0 ? void 0 : _c.AwsCdkPlaygroundBatch[0].data) {
        const entry = m.artifacts.stack.metadata.AwsCdkPlaygroundBatch[0].data;
        expect(entry[0].key).toEqual('hello');
        expect(entry[0].value).toEqual('world');
    }
    expect(m.version).toEqual('0.0.0');
});
test('can access random metadata', () => {
    var _a, _b, _c, _d, _e, _f;
    const loaded = lib_1.Manifest.load(fixture('random-metadata'));
    const randomArray = (_b = (_a = loaded.artifacts) === null || _a === void 0 ? void 0 : _a.stack.metadata) === null || _b === void 0 ? void 0 : _b.AwsCdkPlaygroundBatch[0].data;
    const randomNumber = (_d = (_c = loaded.artifacts) === null || _c === void 0 ? void 0 : _c.stack.metadata) === null || _d === void 0 ? void 0 : _d.AwsCdkPlaygroundBatch[1].data;
    const randomMap = (_f = (_e = loaded.artifacts) === null || _e === void 0 ? void 0 : _e.stack.metadata) === null || _f === void 0 ? void 0 : _f.AwsCdkPlaygroundBatch[2].data;
    expect(randomArray).toEqual(['42']);
    expect(randomNumber).toEqual(42);
    expect(randomMap).toEqual({
        key: 'value',
    });
    expect(randomMap).toBeTruthy();
    if (randomMap) {
        expect(randomMap.key).toEqual('value');
    }
});
//# sourceMappingURL=data:application/json;base64,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