"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * The record type.
 */
var RecordType;
(function (RecordType) {
    RecordType["A"] = "A";
    RecordType["AAAA"] = "AAAA";
    RecordType["CAA"] = "CAA";
    RecordType["CNAME"] = "CNAME";
    RecordType["MX"] = "MX";
    RecordType["NAPTR"] = "NAPTR";
    RecordType["NS"] = "NS";
    RecordType["PTR"] = "PTR";
    RecordType["SOA"] = "SOA";
    RecordType["SPF"] = "SPF";
    RecordType["SRV"] = "SRV";
    RecordType["TXT"] = "TXT";
})(RecordType = exports.RecordType || (exports.RecordType = {}));
/**
 * Type union for a record that accepts multiple types of target.
 */
class RecordTarget {
    constructor(values, aliasTarget) {
        this.values = values;
        this.aliasTarget = aliasTarget;
    }
    /**
     * Use string values as target.
     */
    static fromValues(...values) {
        return new RecordTarget(values);
    }
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget) {
        return new RecordTarget(undefined, aliasTarget);
    }
    /**
     * Use ip adresses as target.
     */
    static fromIpAddresses(...ipAddresses) {
        return RecordTarget.fromValues(...ipAddresses);
    }
}
exports.RecordTarget = RecordTarget;
/**
 * A record set.
 */
class RecordSet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const ttl = props.target.aliasTarget ? undefined : ((props.ttl && props.ttl.toSeconds()) || 1800).toString();
        const recordSet = new route53_generated_1.CfnRecordSet(this, 'Resource', {
            hostedZoneId: props.zone.hostedZoneId,
            name: util_1.determineFullyQualifiedDomainName(props.recordName || props.zone.zoneName, props.zone),
            type: props.recordType,
            resourceRecords: props.target.values,
            aliasTarget: props.target.aliasTarget && props.target.aliasTarget.bind(this),
            ttl,
            comment: props.comment
        });
        this.domainName = recordSet.ref;
    }
}
exports.RecordSet = RecordSet;
/**
 * @deprecated Use RecordTarget
 */
class AddressRecordTarget extends RecordTarget {
}
exports.AddressRecordTarget = AddressRecordTarget;
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
class ARecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.A,
            target: props.target,
        });
    }
}
exports.ARecord = ARecord;
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class AaaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.AAAA,
            target: props.target,
        });
    }
}
exports.AaaaRecord = AaaaRecord;
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
class CnameRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CNAME,
            target: RecordTarget.fromValues(props.domainName)
        });
    }
}
exports.CnameRecord = CnameRecord;
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
class TxtRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.TXT,
            target: RecordTarget.fromValues(...props.values.map(v => JSON.stringify(v))),
        });
    }
}
exports.TxtRecord = TxtRecord;
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
class SrvRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.SRV,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.weight} ${v.port} ${v.hostName}`)),
        });
    }
}
exports.SrvRecord = SrvRecord;
/**
 * The CAA tag.
 */
var CaaTag;
(function (CaaTag) {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    CaaTag["ISSUE"] = "issue";
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    CaaTag["ISSUEWILD"] = "issuewild";
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    CaaTag["IODEF"] = "iodef";
})(CaaTag = exports.CaaTag || (exports.CaaTag = {}));
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CAA,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.flag} ${v.tag} "${v.value}"`)),
        });
    }
}
exports.CaaRecord = CaaRecord;
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaAmazonRecord extends CaaRecord {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            values: [
                {
                    flag: 0,
                    tag: CaaTag.ISSUE,
                    value: 'amazon.com'
                }
            ],
            recordName: props.zone.zoneName
        });
    }
}
exports.CaaAmazonRecord = CaaAmazonRecord;
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
class MxRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.MX,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.hostName}`))
        });
    }
}
exports.MxRecord = MxRecord;
/**
 * A record to delegate further lookups to a different set of name servers.
 */
class ZoneDelegationRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.NS,
            target: RecordTarget.fromValues(...core_1.Token.isUnresolved(props.nameServers)
                ? props.nameServers // Can't map a string-array token!
                : props.nameServers.map(ns => (core_1.Token.isUnresolved(ns) || ns.endsWith('.')) ? ns : `${ns}.`)),
            ttl: props.ttl || core_1.Duration.days(2)
        });
    }
}
exports.ZoneDelegationRecord = ZoneDelegationRecord;
//# sourceMappingURL=data:application/json;base64,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