"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
/**
 * Validates a zone name is valid by Route53 specifc naming rules,
 * and that there is no trailing dot in the name.
 *
 * @param zoneName the zone name to be validated.
 * @returns +zoneName+
 * @throws ValidationError if the name is not valid.
 */
function validateZoneName(zoneName) {
    if (zoneName.endsWith('.')) {
        throw new ValidationError('zone name must not end with a trailing dot');
    }
    if (zoneName.length > 255) {
        throw new ValidationError('zone name cannot be more than 255 bytes long');
    }
    if (zoneName.split('.').find(label => label.length > 63)) {
        throw new ValidationError('zone name labels cannot be more than 63 bytes long');
    }
    if (!zoneName.match(/^[a-z0-9!"#$%&'()*+,/:;<=>?@[\\\]^_`{|}~.-]+$/i)) {
        throw new ValidationError('zone names can only contain a-z, 0-9, -, ! " # $ % & \' ( ) * + , - / : ; < = > ? @ [ \ ] ^ _ ` { | } ~ .');
    }
}
exports.validateZoneName = validateZoneName;
class ValidationError extends Error {
    constructor(message) {
        super(message);
    }
}
/**
 * Route53 requires the record names are specified as fully qualified names, but this
 * forces lots of redundant work on the user (repeating the zone name over and over).
 * This function allows the user to be lazier and offers a nicer experience, by
 * qualifying relative names appropriately:
 *
 * @param providedName the user-specified name of the record.
 * @param zoneName   the fully-qualified name of the zone the record will be created in.
 *
 * @returns <ul>
 *        <li>If +providedName+ ends with a +.+, use it as-is</li>
 *        <li>If +providedName+ ends with or equals +zoneName+, append a trailing +.+</li>
 *        <li>Otherwise, append +.+, +zoneName+ and a trailing +.+</li>
 *      </ul>
 */
function determineFullyQualifiedDomainName(providedName, hostedZone) {
    if (providedName.endsWith('.')) {
        return providedName;
    }
    const suffix = `.${hostedZone.zoneName}`;
    if (providedName.endsWith(suffix) || providedName === hostedZone.zoneName) {
        return `${providedName}.`;
    }
    return `${providedName}${suffix}.`;
}
exports.determineFullyQualifiedDomainName = determineFullyQualifiedDomainName;
function makeHostedZoneArn(construct, hostedZoneId) {
    return core_1.Stack.of(construct).formatArn({
        account: '',
        region: '',
        service: 'route53',
        resource: 'hostedzone',
        resourceName: hostedZoneId
    });
}
exports.makeHostedZoneArn = makeHostedZoneArn;
//# sourceMappingURL=data:application/json;base64,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