import random
import unittest
from tqsdk import TqApi, TqSim, TargetPosTask, TqBacktest, BacktestFinished, utils
from tqsdk.test.api.helper import MockServer, MockInsServer
from tqsdk.tafunc import ma
from contextlib import closing
from datetime import date, datetime
import json
import os


class Unit_test(unittest.TestCase):
    '''
     回测测试example

     注：
    1. 在本地运行测试用例前需设置运行环境变量(Environment variables), 保证api中dict及set等类型的数据序列在每次运行时元素顺序一致: PYTHONHASHSEED=32
    2. 若测试用例中调用了会使用uuid的功能函数时（如insert_order()会使用uuid生成order_id）,
        则：在生成script文件时及测试用例中都需设置 utils.RD = random.Random(x), 以保证两次生成的uuid一致, x取值范围为0-2^32
    3. 對盤中的測試用例（即非回測）：因为TqSim模拟交易 Order 的 insert_date_time 和 Trade 的 trade_date_time 不是固定值，所以改为判断范围。
        盘中时：self.assertAlmostEqual(1575292560005832000 / 1e9, order1.insert_date_time / 1e9, places=1)
        回测时：self.assertEqual(1575291600000000000, order1.insert_date_time)
    '''

    def setUp(self):
        self.ins = MockInsServer(5000)
        self.mock = MockServer()
        # self.tq = WebsocketServer(5300)
        self.ins_url_2019_07_03 = "http://127.0.0.1:5000/t/md/symbols/2019-07-03.json"
        self.ins_url_2019_06_05 = "http://127.0.0.1:5000/t/md/symbols/2019-06-05.json"
        self.ins_url_2020_06_10 = "http://127.0.0.1:5000/t/md/symbols/2019-06-10.json"
        self.md_url = "ws://127.0.0.1:5100/"
        self.td_url = "ws://127.0.0.1:5200/"

    def tearDown(self):
        self.ins.close()
        self.mock.close()

    # @unittest.skip("无条件跳过: test_doublema")
    def test_doublema(self):
        '''
        设定：
            回测周期: 2019,6,10-2019,6,15
            合约代码: SHFE.bu1912
            合约文件: "http://127.0.0.1:5000/t/md/symbols/2019-07-03.json"
        '''
        # 预设服务器端响应
        dir_path = os.path.dirname(os.path.realpath(__file__))
        self.mock.run(os.path.join(dir_path, "examples_log_file", "test_example_doublema.script.lzma"))
        # 测试
        SHORT = 30  # 短周期
        LONG = 60  # 长周期
        SYMBOL = "SHFE.bu1912"  # 合约代码

        utils.RD = random.Random(4)
        sim = TqSim()
        api = TqApi(sim, backtest=TqBacktest(start_dt=datetime(2019, 6, 10), end_dt=datetime(2019, 6, 15)),
                    _ins_url=self.ins_url_2019_07_03, _td_url=self.td_url, _md_url=self.md_url)
        data_length = LONG + 2  # k线数据长度
        klines = api.get_kline_serial(SYMBOL, duration_seconds=240, data_length=data_length)
        target_pos = TargetPosTask(api, SYMBOL)

        try:
            with closing(api):
                while True:
                    api.wait_update()
                    if api.is_changing(klines.iloc[-1], "datetime"):  # 产生新k线:重新计算SMA
                        short_avg = ma(klines["close"], SHORT)  # 短周期
                        long_avg = ma(klines["close"], LONG)  # 长周期

                        # 均线下穿，做空
                        if long_avg.iloc[-2] < short_avg.iloc[-2] and long_avg.iloc[-1] > short_avg.iloc[-1]:
                            target_pos.set_target_volume(-3)

                        # 均线上穿，做多
                        if short_avg.iloc[-2] < long_avg.iloc[-2] and short_avg.iloc[-1] > long_avg.iloc[-1]:
                            target_pos.set_target_volume(3)

        except BacktestFinished:
            # 测试
            trade_log = json.dumps(sim.trade_log)
            # print(trade_log)  # 用于生成测试用例except数据
            excepted_trade_log = '{"2019-06-10": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10000000.0, "static_balance": 10000000.0, "balance": 10000000.0, "available": 10000000.0, "float_profit": 0.0, "position_profit": 0.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {}}, "2019-06-11": {"trades": [{"user_id": "TQSIM", "order_id": "8534f45738d048ec0f1099c6c3e1b258", "trade_id": "8534f45738d048ec0f1099c6c3e1b258|3", "exchange_trade_id": "8534f45738d048ec0f1099c6c3e1b258|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "BUY", "offset": "OPEN", "price": 2972.0, "volume": 3, "trade_date_time": 1560217680000000000, "commission": 9.702}], "account": {"currency": "CNY", "pre_balance": 10000000.0, "static_balance": 10000000.0, "balance": 10002570.298, "available": 9993838.498, "float_profit": 2580.0, "position_profit": 2580.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8731.8, "frozen_commission": 0.0, "commission": 9.702, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.000872955624390454, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.bu1912": {"exchange_id": "SHFE", "instrument_id": "bu1912", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 2972.0, "open_price_short": NaN, "open_cost_long": 89160.0, "open_cost_short": 0.0, "position_price_long": 2972.0, "position_price_short": NaN, "position_cost_long": 89160.0, "position_cost_short": 0.0, "float_profit_long": 2580.0, "float_profit_short": 0.0, "float_profit": 2580.0, "position_profit_long": 2580.0, "position_profit_short": 0.0, "position_profit": 2580.0, "margin_long": 8731.8, "margin_short": 0.0, "margin": 8731.8, "last_price": 3058.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-12": {"trades": [{"user_id": "TQSIM", "order_id": "c79d679346d4ac7a5c3902b38963dc6e", "trade_id": "c79d679346d4ac7a5c3902b38963dc6e|3", "exchange_trade_id": "c79d679346d4ac7a5c3902b38963dc6e|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "SELL", "offset": "CLOSE", "price": 3030.0, "volume": 3, "trade_date_time": 1560303600000000000, "commission": 9.702}, {"user_id": "TQSIM", "order_id": "43000de01b2ed40ed3addccb2c33be0a", "trade_id": "43000de01b2ed40ed3addccb2c33be0a|3", "exchange_trade_id": "43000de01b2ed40ed3addccb2c33be0a|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "SELL", "offset": "OPEN", "price": 3030.0, "volume": 3, "trade_date_time": 1560303600000000000, "commission": 9.702}], "account": {"currency": "CNY", "pre_balance": 10002570.298, "static_balance": 10002570.298, "balance": 10002130.894000001, "available": 9993399.094, "float_profit": 420.0, "position_profit": 420.0, "close_profit": -840.0, "frozen_margin": 0.0, "margin": 8731.8, "frozen_commission": 0.0, "commission": 19.404, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008729939742378259, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.bu1912": {"exchange_id": "SHFE", "instrument_id": "bu1912", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 3, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 3, "volume_short_his": 0, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 3030.0, "open_cost_long": 0.0, "open_cost_short": 90900.0, "position_price_long": NaN, "position_price_short": 3030.0, "position_cost_long": 0.0, "position_cost_short": 90900.0, "float_profit_long": 0.0, "float_profit_short": 420.0, "float_profit": 420.0, "position_profit_long": 0.0, "position_profit_short": 420.0, "position_profit": 420.0, "margin_long": 0.0, "margin_short": 8731.8, "margin": 8731.8, "last_price": 3016.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-13": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10002130.894000001, "static_balance": 10002130.894000001, "balance": 10002430.894000001, "available": 9993699.094, "float_profit": 720.0, "position_profit": 300.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8731.8, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.000872967790783519, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.bu1912": {"exchange_id": "SHFE", "instrument_id": "bu1912", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 3, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 3, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 3030.0, "open_cost_long": 0.0, "open_cost_short": 90900.0, "position_price_long": 3016.0, "position_price_short": 3016.0, "position_cost_long": 0.0, "position_cost_short": 91320.0, "float_profit_long": 0.0, "float_profit_short": 720.0, "float_profit": 720.0, "position_profit_long": 0.0, "position_profit_short": 300.0, "position_profit": 300.0, "margin_long": 0.0, "margin_short": 8731.8, "margin": 8731.8, "last_price": 3006.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-14": {"trades": [{"user_id": "TQSIM", "order_id": "06905269ed6f0b09f165c8ce36e2f24b", "trade_id": "06905269ed6f0b09f165c8ce36e2f24b|3", "exchange_trade_id": "06905269ed6f0b09f165c8ce36e2f24b|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "BUY", "offset": "CLOSE", "price": 3012.0, "volume": 3, "trade_date_time": 1560431760000000000, "commission": 9.702}, {"user_id": "TQSIM", "order_id": "42a00403ce80c4b0a4042bb3d4341aad", "trade_id": "42a00403ce80c4b0a4042bb3d4341aad|3", "exchange_trade_id": "42a00403ce80c4b0a4042bb3d4341aad|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "BUY", "offset": "OPEN", "price": 3012.0, "volume": 3, "trade_date_time": 1560431760000000000, "commission": 9.702}, {"user_id": "TQSIM", "order_id": "2a3187853184ff27459142deccea2645", "trade_id": "2a3187853184ff27459142deccea2645|3", "exchange_trade_id": "2a3187853184ff27459142deccea2645|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "SELL", "offset": "CLOSETODAY", "price": 2944.0, "volume": 3, "trade_date_time": 1560475200000000000, "commission": 9.702}, {"user_id": "TQSIM", "order_id": "de08caa1a08179104a25e4664f5253a0", "trade_id": "de08caa1a08179104a25e4664f5253a0|3", "exchange_trade_id": "de08caa1a08179104a25e4664f5253a0|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "SELL", "offset": "OPEN", "price": 2944.0, "volume": 3, "trade_date_time": 1560475200000000000, "commission": 9.702}, {"user_id": "TQSIM", "order_id": "d93936e1daca3c06f5ff0c03bb5d7385", "trade_id": "d93936e1daca3c06f5ff0c03bb5d7385|3", "exchange_trade_id": "d93936e1daca3c06f5ff0c03bb5d7385|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "BUY", "offset": "CLOSETODAY", "price": 2968.0, "volume": 3, "trade_date_time": 1560492000000000000, "commission": 9.702}, {"user_id": "TQSIM", "order_id": "d8441b5616332aca5f552773e14b0190", "trade_id": "d8441b5616332aca5f552773e14b0190|3", "exchange_trade_id": "d8441b5616332aca5f552773e14b0190|3", "exchange_id": "SHFE", "instrument_id": "bu1912", "direction": "BUY", "offset": "OPEN", "price": 2968.0, "volume": 3, "trade_date_time": 1560492000000000000, "commission": 9.702}], "account": {"currency": "CNY", "pre_balance": 10002430.894000001, "static_balance": 10002430.894000001, "balance": 9998772.682000004, "available": 9990040.882000003, "float_profit": -660.0, "position_profit": -660.0, "close_profit": -2940.0, "frozen_margin": 0.0, "margin": 8731.8, "frozen_commission": 0.0, "commission": 58.211999999999996, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008732871801075311, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.bu1912": {"exchange_id": "SHFE", "instrument_id": "bu1912", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 2968.0, "open_price_short": NaN, "open_cost_long": 89040.0, "open_cost_short": 0.0, "position_price_long": 2968.0, "position_price_short": NaN, "position_cost_long": 89040.0, "position_cost_short": 0.0, "float_profit_long": -660.0, "float_profit_short": 0.0, "float_profit": -660.0, "position_profit_long": -660.0, "position_profit_short": 0.0, "position_profit": -660.0, "margin_long": 8731.8, "margin_short": 0.0, "margin": 8731.8, "last_price": 2946.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-17": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9998772.682000004, "static_balance": 9998772.682000004, "balance": 9999372.682000004, "available": 9990640.882000003, "float_profit": -60.0, "position_profit": 600.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8731.8, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008732347795895459, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.bu1912": {"exchange_id": "SHFE", "instrument_id": "bu1912", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 2968.0, "open_price_short": NaN, "open_cost_long": 89040.0, "open_cost_short": 0.0, "position_price_long": 2946.0, "position_price_short": 2946.0, "position_cost_long": 88380.0, "position_cost_short": 0.0, "float_profit_long": -60.0, "float_profit_short": 0.0, "float_profit": -60.0, "position_profit_long": 600.0, "position_profit_short": 0.0, "position_profit": 600.0, "margin_long": 8731.8, "margin_short": 0.0, "margin": 8731.8, "last_price": 2966.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}}'
            self.assertEqual(trade_log, excepted_trade_log, msg="交易记录及收盘时的权益及持仓: 信息错误")

    # @unittest.skip("无条件跳过: test_dualthrust")
    def test_dualthrust(self):
        '''
        设定：
            回测周期: 2019,5,1-2019,6,10
            合约代码: DCE.jd1909
            合约文件: "http://127.0.0.1:5000/t/md/symbols/2019-07-03.json"
        '''
        # 预设服务器端响应
        dir_path = os.path.dirname(os.path.realpath(__file__))
        self.mock.run(os.path.join(dir_path, "examples_log_file", "test_example_dualthrust.script.lzma"))
        # 测试
        SYMBOL = "DCE.jd1909"  # 合约代码
        NDAY = 5  # 天数
        K1 = 0.2  # 上轨K值
        K2 = 0.2  # 下轨K值
        utils.RD = random.Random(4)
        sim = TqSim()
        api = TqApi(sim, backtest=TqBacktest(start_dt=date(2019, 5, 1), end_dt=date(2019, 6, 10)),
                    _ins_url=self.ins_url_2019_07_03, _td_url=self.td_url, _md_url=self.md_url)

        quote = api.get_quote(SYMBOL)
        klines = api.get_kline_serial(SYMBOL, 24 * 60 * 60)  # 86400使用日线
        target_pos = TargetPosTask(api, SYMBOL)

        def dual_thrust(quote, klines):
            current_open = klines.iloc[-1]["open"]
            HH = max(klines.high.iloc[-NDAY - 1:-1])  # N日最高价的最高价
            HC = max(klines.close.iloc[-NDAY - 1:-1])  # N日收盘价的最高价
            LC = min(klines.close.iloc[-NDAY - 1:-1])  # N日收盘价的最低价
            LL = min(klines.low.iloc[-NDAY - 1:-1])  # N日最低价的最低价
            range = max(HH - LC, HC - LL)
            buy_line = current_open + range * K1  # 上轨
            sell_line = current_open - range * K2  # 下轨
            return buy_line, sell_line

        buy_line, sell_line = dual_thrust(quote, klines)  # 获取上下轨

        try:
            with closing(api):
                while True:
                    api.wait_update()
                    if api.is_changing(klines.iloc[-1], ["datetime", "open"]):  # 新产生一根日线或开盘价发生变化: 重新计算上下轨
                        buy_line, sell_line = dual_thrust(quote, klines)

                    if api.is_changing(quote, "last_price"):
                        if quote.last_price > buy_line:  # 高于上轨
                            target_pos.set_target_volume(3)  # 交易
                        elif quote.last_price < sell_line:  # 低于下轨
                            target_pos.set_target_volume(-3)  # 交易
                        else:
                            pass

        except BacktestFinished:
            # 测试
            trade_log = json.dumps(sim.trade_log)
            # print(trade_log)  # 用于生成测试用例except数据
            excepted_trade_log = '{"2019-05-01": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10000000.0, "static_balance": 10000000.0, "balance": 10000000.0, "available": 10000000.0, "float_profit": 0.0, "position_profit": 0.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {}}, "2019-05-06": {"trades": [{"user_id": "TQSIM", "order_id": "8534f45738d048ec0f1099c6c3e1b258", "trade_id": "8534f45738d048ec0f1099c6c3e1b258|3", "exchange_trade_id": "8534f45738d048ec0f1099c6c3e1b258|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4286.0, "volume": 3, "trade_date_time": 1557104579999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "c79d679346d4ac7a5c3902b38963dc6e", "trade_id": "c79d679346d4ac7a5c3902b38963dc6e|3", "exchange_trade_id": "c79d679346d4ac7a5c3902b38963dc6e|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "CLOSE", "price": 4341.0, "volume": 3, "trade_date_time": 1557105299999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "43000de01b2ed40ed3addccb2c33be0a", "trade_id": "43000de01b2ed40ed3addccb2c33be0a|3", "exchange_trade_id": "43000de01b2ed40ed3addccb2c33be0a|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "OPEN", "price": 4341.0, "volume": 3, "trade_date_time": 1557105299999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 10000000.0, "static_balance": 10000000.0, "balance": 9997904.892999997, "available": 9989332.692999998, "float_profit": -390.0, "position_profit": -390.0, "close_profit": -1650.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 55.107, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008573996343975827, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4341.0, "position_price_short": NaN, "position_cost_long": 130230.0, "position_cost_short": 0.0, "float_profit_long": -390.0, "float_profit_short": 0.0, "float_profit": -390.0, "position_profit_long": -390.0, "position_profit_short": 0.0, "position_profit": -390.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4328.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-07": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9997904.892999997, "static_balance": 9997904.892999997, "balance": 9998804.892999997, "available": 9990232.692999998, "float_profit": 510.0, "position_profit": 900.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008573224592072258, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4328.0, "position_price_short": 4328.0, "position_cost_long": 129840.0, "position_cost_short": 0.0, "float_profit_long": 510.0, "float_profit_short": 0.0, "float_profit": 510.0, "position_profit_long": 900.0, "position_profit_short": 0.0, "position_profit": 900.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4358.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-08": {"trades": [{"user_id": "TQSIM", "order_id": "06905269ed6f0b09f165c8ce36e2f24b", "trade_id": "06905269ed6f0b09f165c8ce36e2f24b|3", "exchange_trade_id": "06905269ed6f0b09f165c8ce36e2f24b|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "CLOSE", "price": 4332.0, "volume": 3, "trade_date_time": 1557285419999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "42a00403ce80c4b0a4042bb3d4341aad", "trade_id": "42a00403ce80c4b0a4042bb3d4341aad|3", "exchange_trade_id": "42a00403ce80c4b0a4042bb3d4341aad|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4332.0, "volume": 3, "trade_date_time": 1557285419999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9998804.892999997, "static_balance": 9998804.892999997, "balance": 9997448.154999996, "available": 9988875.954999996, "float_profit": -540.0, "position_profit": -540.0, "close_profit": -780.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574388050927586, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 3, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 3, "volume_short_his": 0, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4332.0, "open_cost_long": 0.0, "open_cost_short": 129960.0, "position_price_long": NaN, "position_price_short": 4332.0, "position_cost_long": 0.0, "position_cost_short": 129960.0, "float_profit_long": 0.0, "float_profit_short": -540.0, "float_profit": -540.0, "position_profit_long": 0.0, "position_profit_short": -540.0, "position_profit": -540.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4350.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-09": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9997448.154999996, "static_balance": 9997448.154999996, "balance": 9997328.154999996, "available": 9988755.954999996, "float_profit": -660.0, "position_profit": -120.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574490971082867, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 3, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 3, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4332.0, "open_cost_long": 0.0, "open_cost_short": 129960.0, "position_price_long": 4350.0, "position_price_short": 4350.0, "position_cost_long": 0.0, "position_cost_short": 129420.0, "float_profit_long": 0.0, "float_profit_short": -660.0, "float_profit": -660.0, "position_profit_long": 0.0, "position_profit_short": -120.0, "position_profit": -120.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4354.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-10": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9997328.154999996, "static_balance": 9997328.154999996, "balance": 9997838.154999996, "available": 9989265.954999996, "float_profit": -150.0, "position_profit": 510.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574053577485626, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 3, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 3, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4332.0, "open_cost_long": 0.0, "open_cost_short": 129960.0, "position_price_long": 4354.0, "position_price_short": 4354.0, "position_cost_long": 0.0, "position_cost_short": 129300.0, "float_profit_long": 0.0, "float_profit_short": -150.0, "float_profit": -150.0, "position_profit_long": 0.0, "position_profit_short": 510.0, "position_profit": 510.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4337.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-13": {"trades": [{"user_id": "TQSIM", "order_id": "2a3187853184ff27459142deccea2645", "trade_id": "2a3187853184ff27459142deccea2645|3", "exchange_trade_id": "2a3187853184ff27459142deccea2645|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "CLOSE", "price": 4349.0, "volume": 3, "trade_date_time": 1557709259999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "de08caa1a08179104a25e4664f5253a0", "trade_id": "de08caa1a08179104a25e4664f5253a0|3", "exchange_trade_id": "de08caa1a08179104a25e4664f5253a0|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "OPEN", "price": 4349.0, "volume": 3, "trade_date_time": 1557709259999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9997838.154999996, "static_balance": 9997838.154999996, "balance": 9997531.416999994, "available": 9988959.216999995, "float_profit": 90.0, "position_profit": 90.0, "close_profit": -360.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574316641229722, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4349.0, "open_price_short": NaN, "open_cost_long": 130470.0, "open_cost_short": 0.0, "position_price_long": 4349.0, "position_price_short": NaN, "position_cost_long": 130470.0, "position_cost_short": 0.0, "float_profit_long": 90.0, "float_profit_short": 0.0, "float_profit": 90.0, "position_profit_long": 90.0, "position_profit_short": 0.0, "position_profit": 90.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4352.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-14": {"trades": [{"user_id": "TQSIM", "order_id": "d93936e1daca3c06f5ff0c03bb5d7385", "trade_id": "d93936e1daca3c06f5ff0c03bb5d7385|3", "exchange_trade_id": "d93936e1daca3c06f5ff0c03bb5d7385|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "CLOSE", "price": 4329.0, "volume": 3, "trade_date_time": 1557795959999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "d8441b5616332aca5f552773e14b0190", "trade_id": "d8441b5616332aca5f552773e14b0190|3", "exchange_trade_id": "d8441b5616332aca5f552773e14b0190|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4329.0, "volume": 3, "trade_date_time": 1557795959999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "634f806fabf4a07c566002249b191bf4", "trade_id": "634f806fabf4a07c566002249b191bf4|3", "exchange_trade_id": "634f806fabf4a07c566002249b191bf4|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "CLOSE", "price": 4358.0, "volume": 3, "trade_date_time": 1557798239999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "3f5082492d83a8233fb62d2c81862fc9", "trade_id": "3f5082492d83a8233fb62d2c81862fc9|3", "exchange_trade_id": "3f5082492d83a8233fb62d2c81862fc9|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "OPEN", "price": 4358.0, "volume": 3, "trade_date_time": 1557798239999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9997531.416999994, "static_balance": 9997531.416999994, "balance": 9996047.94099999, "available": 9987475.740999991, "float_profit": 150.0, "position_profit": 150.0, "close_profit": -1560.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 73.476, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.000857558912341756, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4358.0, "open_price_short": NaN, "open_cost_long": 130740.0, "open_cost_short": 0.0, "position_price_long": 4358.0, "position_price_short": NaN, "position_cost_long": 130740.0, "position_cost_short": 0.0, "float_profit_long": 150.0, "float_profit_short": 0.0, "float_profit": 150.0, "position_profit_long": 150.0, "position_profit_short": 0.0, "position_profit": 150.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4363.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-15": {"trades": [{"user_id": "TQSIM", "order_id": "f1cfd99216df648647adec26793d0e45", "trade_id": "f1cfd99216df648647adec26793d0e45|3", "exchange_trade_id": "f1cfd99216df648647adec26793d0e45|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "CLOSE", "price": 4363.0, "volume": 3, "trade_date_time": 1557882239999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "f1347e0cdd905ecfd160c5d0ef412ed6", "trade_id": "f1347e0cdd905ecfd160c5d0ef412ed6|3", "exchange_trade_id": "f1347e0cdd905ecfd160c5d0ef412ed6|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4363.0, "volume": 3, "trade_date_time": 1557882239999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9996047.94099999, "static_balance": 9996047.94099999, "balance": 9995681.202999989, "available": 9987109.00299999, "float_profit": -330.0, "position_profit": -330.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008575903758742565, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 3, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 3, "volume_short_his": 0, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4363.0, "open_cost_long": 0.0, "open_cost_short": 130890.0, "position_price_long": NaN, "position_price_short": 4363.0, "position_cost_long": 0.0, "position_cost_short": 130890.0, "float_profit_long": 0.0, "float_profit_short": -330.0, "float_profit": -330.0, "position_profit_long": 0.0, "position_profit_short": -330.0, "position_profit": -330.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4374.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-16": {"trades": [{"user_id": "TQSIM", "order_id": "01d89a024cdce7a6d7288ff68c320f89", "trade_id": "01d89a024cdce7a6d7288ff68c320f89|3", "exchange_trade_id": "01d89a024cdce7a6d7288ff68c320f89|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "CLOSE", "price": 4398.0, "volume": 3, "trade_date_time": 1557970319999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "b474c7e89286a1754abcb06ae8abb93f", "trade_id": "b474c7e89286a1754abcb06ae8abb93f|3", "exchange_trade_id": "b474c7e89286a1754abcb06ae8abb93f|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "OPEN", "price": 4398.0, "volume": 3, "trade_date_time": 1557970319999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9995681.202999989, "static_balance": 9995681.202999989, "balance": 9994984.464999987, "available": 9986412.264999988, "float_profit": 60.0, "position_profit": 60.0, "close_profit": -720.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008576501574382399, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4398.0, "open_price_short": NaN, "open_cost_long": 131940.0, "open_cost_short": 0.0, "position_price_long": 4398.0, "position_price_short": NaN, "position_cost_long": 131940.0, "position_cost_short": 0.0, "float_profit_long": 60.0, "float_profit_short": 0.0, "float_profit": 60.0, "position_profit_long": 60.0, "position_profit_short": 0.0, "position_profit": 60.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4400.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-17": {"trades": [{"user_id": "TQSIM", "order_id": "c3e4a892d9196ada4fcfa583e1df8af9", "trade_id": "c3e4a892d9196ada4fcfa583e1df8af9|3", "exchange_trade_id": "c3e4a892d9196ada4fcfa583e1df8af9|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "CLOSE", "price": 4381.0, "volume": 3, "trade_date_time": 1558075079999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "6c79a3de69f85e3131f3b9238224b122", "trade_id": "6c79a3de69f85e3131f3b9238224b122|3", "exchange_trade_id": "6c79a3de69f85e3131f3b9238224b122|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4381.0, "volume": 3, "trade_date_time": 1558075079999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9994984.464999987, "static_balance": 9994984.464999987, "balance": 9995307.726999985, "available": 9986735.526999986, "float_profit": 930.0, "position_profit": 930.0, "close_profit": -570.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008576224198524882, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 3, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 3, "volume_short_his": 0, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4381.0, "open_cost_long": 0.0, "open_cost_short": 131430.0, "position_price_long": NaN, "position_price_short": 4381.0, "position_cost_long": 0.0, "position_cost_short": 131430.0, "float_profit_long": 0.0, "float_profit_short": 930.0, "float_profit": 930.0, "position_profit_long": 0.0, "position_profit_short": 930.0, "position_profit": 930.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4350.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-20": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9995307.726999985, "static_balance": 9995307.726999985, "balance": 9996717.726999985, "available": 9988145.526999986, "float_profit": 2340.0, "position_profit": 1410.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008575014553874492, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 3, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 3, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4381.0, "open_cost_long": 0.0, "open_cost_short": 131430.0, "position_price_long": 4350.0, "position_price_short": 4350.0, "position_cost_long": 0.0, "position_cost_short": 132360.0, "float_profit_long": 0.0, "float_profit_short": 2340.0, "float_profit": 2340.0, "position_profit_long": 0.0, "position_profit_short": 1410.0, "position_profit": 1410.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4303.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-21": {"trades": [{"user_id": "TQSIM", "order_id": "738d243a6e58d5ca49c7b59b995253fd", "trade_id": "738d243a6e58d5ca49c7b59b995253fd|3", "exchange_trade_id": "738d243a6e58d5ca49c7b59b995253fd|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "CLOSE", "price": 4342.0, "volume": 3, "trade_date_time": 1558419059999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "4278c2614e1bcb383bb4a570294c4ea3", "trade_id": "4278c2614e1bcb383bb4a570294c4ea3|3", "exchange_trade_id": "4278c2614e1bcb383bb4a570294c4ea3|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "OPEN", "price": 4342.0, "volume": 3, "trade_date_time": 1558419059999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9996717.726999985, "static_balance": 9996717.726999985, "balance": 9995690.988999983, "available": 9987118.788999984, "float_profit": 180.0, "position_profit": 180.0, "close_profit": -1170.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008575895362745307, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4342.0, "open_price_short": NaN, "open_cost_long": 130260.0, "open_cost_short": 0.0, "position_price_long": 4342.0, "position_price_short": NaN, "position_cost_long": 130260.0, "position_cost_short": 0.0, "float_profit_long": 180.0, "float_profit_short": 0.0, "float_profit": 180.0, "position_profit_long": 180.0, "position_profit_short": 0.0, "position_profit": 180.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4348.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-22": {"trades": [{"user_id": "TQSIM", "order_id": "14c15c910b11ad28cc21ce88d0060cc5", "trade_id": "14c15c910b11ad28cc21ce88d0060cc5|3", "exchange_trade_id": "14c15c910b11ad28cc21ce88d0060cc5|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "CLOSE", "price": 4302.0, "volume": 3, "trade_date_time": 1558504439999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "ff5a52f1a05885ac7671863c0bdbc23a", "trade_id": "ff5a52f1a05885ac7671863c0bdbc23a|3", "exchange_trade_id": "ff5a52f1a05885ac7671863c0bdbc23a|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4302.0, "volume": 3, "trade_date_time": 1558504439999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9995690.988999983, "static_balance": 9995690.988999983, "balance": 9993734.250999982, "available": 9985162.050999982, "float_profit": -540.0, "position_profit": -540.0, "close_profit": -1380.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008577574492880136, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 3, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 3, "volume_short_his": 0, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4302.0, "open_cost_long": 0.0, "open_cost_short": 129060.0, "position_price_long": NaN, "position_price_short": 4302.0, "position_cost_long": 0.0, "position_cost_short": 129060.0, "float_profit_long": 0.0, "float_profit_short": -540.0, "float_profit": -540.0, "position_profit_long": 0.0, "position_profit_short": -540.0, "position_profit": -540.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4320.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-23": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9993734.250999982, "static_balance": 9993734.250999982, "balance": 9994004.250999982, "available": 9985432.050999982, "float_profit": -270.0, "position_profit": 270.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008577342759427265, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 3, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 3, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4302.0, "open_cost_long": 0.0, "open_cost_short": 129060.0, "position_price_long": 4320.0, "position_price_short": 4320.0, "position_cost_long": 0.0, "position_cost_short": 128520.0, "float_profit_long": 0.0, "float_profit_short": -270.0, "float_profit": -270.0, "position_profit_long": 0.0, "position_profit_short": 270.0, "position_profit": 270.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4311.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-24": {"trades": [{"user_id": "TQSIM", "order_id": "a5e333cb88dcf94384d4cd1f47ca7883", "trade_id": "a5e333cb88dcf94384d4cd1f47ca7883|3", "exchange_trade_id": "a5e333cb88dcf94384d4cd1f47ca7883|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "CLOSE", "price": 4341.0, "volume": 3, "trade_date_time": 1558663319999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "2522d53857c49391b36cc9aa78a330a1", "trade_id": "2522d53857c49391b36cc9aa78a330a1|3", "exchange_trade_id": "2522d53857c49391b36cc9aa78a330a1|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "OPEN", "price": 4341.0, "volume": 3, "trade_date_time": 1558663319999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9994004.250999982, "static_balance": 9994004.250999982, "balance": 9994507.51299998, "available": 9985935.31299998, "float_profit": 1440.0, "position_profit": 1440.0, "close_profit": -900.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008576910857138317, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4341.0, "position_price_short": NaN, "position_cost_long": 130230.0, "position_cost_short": 0.0, "float_profit_long": 1440.0, "float_profit_short": 0.0, "float_profit": 1440.0, "position_profit_long": 1440.0, "position_profit_short": 0.0, "position_profit": 1440.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4389.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-27": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9994507.51299998, "static_balance": 9994507.51299998, "balance": 9993997.51299998, "available": 9985425.31299998, "float_profit": 930.0, "position_profit": -510.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008577348542311988, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4389.0, "position_price_short": 4389.0, "position_cost_long": 131670.0, "position_cost_short": 0.0, "float_profit_long": 930.0, "float_profit_short": 0.0, "float_profit": 930.0, "position_profit_long": -510.0, "position_profit_short": 0.0, "position_profit": -510.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4372.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-28": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9993997.51299998, "static_balance": 9993997.51299998, "balance": 9994567.51299998, "available": 9985995.31299998, "float_profit": 1500.0, "position_profit": 570.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008576859367701605, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4372.0, "position_price_short": 4372.0, "position_cost_long": 131160.0, "position_cost_short": 0.0, "float_profit_long": 1500.0, "float_profit_short": 0.0, "float_profit": 1500.0, "position_profit_long": 570.0, "position_profit_short": 0.0, "position_profit": 570.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4391.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-29": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9994567.51299998, "static_balance": 9994567.51299998, "balance": 9996577.51299998, "available": 9988005.31299998, "float_profit": 3510.0, "position_profit": 2010.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008575134828747482, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4391.0, "position_price_short": 4391.0, "position_cost_long": 131730.0, "position_cost_short": 0.0, "float_profit_long": 3510.0, "float_profit_short": 0.0, "float_profit": 3510.0, "position_profit_long": 2010.0, "position_profit_short": 0.0, "position_profit": 2010.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4458.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-30": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9996577.51299998, "static_balance": 9996577.51299998, "balance": 9995887.51299998, "available": 9987315.31299998, "float_profit": 2820.0, "position_profit": -690.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008575726756480176, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4458.0, "position_price_short": 4458.0, "position_cost_long": 133740.0, "position_cost_short": 0.0, "float_profit_long": 2820.0, "float_profit_short": 0.0, "float_profit": 2820.0, "position_profit_long": -690.0, "position_profit_short": 0.0, "position_profit": -690.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4435.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-31": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9995887.51299998, "static_balance": 9995887.51299998, "balance": 9997777.51299998, "available": 9989205.31299998, "float_profit": 4710.0, "position_profit": 1890.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574105583819684, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4435.0, "position_price_short": 4435.0, "position_cost_long": 133050.0, "position_cost_short": 0.0, "float_profit_long": 4710.0, "float_profit_short": 0.0, "float_profit": 4710.0, "position_profit_long": 1890.0, "position_profit_short": 0.0, "position_profit": 1890.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4498.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-03": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9997777.51299998, "static_balance": 9997777.51299998, "balance": 9998167.51299998, "available": 9989595.31299998, "float_profit": 5100.0, "position_profit": 390.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008573771132414131, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 3, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 3, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 4341.0, "open_price_short": NaN, "open_cost_long": 130230.0, "open_cost_short": 0.0, "position_price_long": 4498.0, "position_price_short": 4498.0, "position_cost_long": 134940.0, "position_cost_short": 0.0, "float_profit_long": 5100.0, "float_profit_short": 0.0, "float_profit": 5100.0, "position_profit_long": 390.0, "position_profit_short": 0.0, "position_profit": 390.0, "margin_long": 8572.2, "margin_short": 0.0, "margin": 8572.2, "last_price": 4511.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-04": {"trades": [{"user_id": "TQSIM", "order_id": "11021c9e32111ac1ac7cc4a4ff4dab10", "trade_id": "11021c9e32111ac1ac7cc4a4ff4dab10|3", "exchange_trade_id": "11021c9e32111ac1ac7cc4a4ff4dab10|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "CLOSE", "price": 4471.0, "volume": 3, "trade_date_time": 1559614139999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "a2909cb633e238b4e9dd38b869ace913", "trade_id": "a2909cb633e238b4e9dd38b869ace913|3", "exchange_trade_id": "a2909cb633e238b4e9dd38b869ace913|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4471.0, "volume": 3, "trade_date_time": 1559614139999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9998167.51299998, "static_balance": 9998167.51299998, "balance": 9997170.774999978, "available": 9988598.574999979, "float_profit": 240.0, "position_profit": 240.0, "close_profit": -1200.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 36.738, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574625954611663, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 3, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 3, "volume_short_his": 0, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4471.0, "open_cost_long": 0.0, "open_cost_short": 134130.0, "position_price_long": NaN, "position_price_short": 4471.0, "position_cost_long": 0.0, "position_cost_short": 134130.0, "float_profit_long": 0.0, "float_profit_short": 240.0, "float_profit": 240.0, "position_profit_long": 0.0, "position_profit_short": 240.0, "position_profit": 240.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4463.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-05": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9997170.774999978, "static_balance": 9997170.774999978, "balance": 9997230.774999978, "available": 9988658.574999979, "float_profit": 300.0, "position_profit": 60.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574574492604947, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 3, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 3, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4471.0, "open_cost_long": 0.0, "open_cost_short": 134130.0, "position_price_long": 4463.0, "position_price_short": 4463.0, "position_cost_long": 0.0, "position_cost_short": 134370.0, "float_profit_long": 0.0, "float_profit_short": 300.0, "float_profit": 300.0, "position_profit_long": 0.0, "position_profit_short": 60.0, "position_profit": 60.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4461.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-06": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9997230.774999978, "static_balance": 9997230.774999978, "balance": 9997890.774999978, "available": 9989318.574999979, "float_profit": 960.0, "position_profit": 660.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008574008451297588, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 3, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 3, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4471.0, "open_cost_long": 0.0, "open_cost_short": 134130.0, "position_price_long": 4461.0, "position_price_short": 4461.0, "position_cost_long": 0.0, "position_cost_short": 134430.0, "float_profit_long": 0.0, "float_profit_short": 960.0, "float_profit": 960.0, "position_profit_long": 0.0, "position_profit_short": 660.0, "position_profit": 660.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4439.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-10": {"trades": [{"user_id": "TQSIM", "order_id": "2f0733c846bbe9e870ef55b1a1f65507", "trade_id": "2f0733c846bbe9e870ef55b1a1f65507|3", "exchange_trade_id": "2f0733c846bbe9e870ef55b1a1f65507|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "CLOSE", "price": 4490.0, "volume": 3, "trade_date_time": 1560128459999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "96b98b5fbf37a2be6f98bca35b17b966", "trade_id": "96b98b5fbf37a2be6f98bca35b17b966|3", "exchange_trade_id": "96b98b5fbf37a2be6f98bca35b17b966|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "BUY", "offset": "OPEN", "price": 4490.0, "volume": 3, "trade_date_time": 1560128459999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "32decd6b8efbc170a26a25c852175b7a", "trade_id": "32decd6b8efbc170a26a25c852175b7a|3", "exchange_trade_id": "32decd6b8efbc170a26a25c852175b7a|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "CLOSE", "price": 4417.0, "volume": 3, "trade_date_time": 1560132179999999000, "commission": 18.369}, {"user_id": "TQSIM", "order_id": "d6e4a51519d9c9cc52d32377e78131c1", "trade_id": "d6e4a51519d9c9cc52d32377e78131c1|3", "exchange_trade_id": "d6e4a51519d9c9cc52d32377e78131c1|3", "exchange_id": "DCE", "instrument_id": "jd1909", "direction": "SELL", "offset": "OPEN", "price": 4417.0, "volume": 3, "trade_date_time": 1560132179999999000, "commission": 18.369}], "account": {"currency": "CNY", "pre_balance": 9997890.774999978, "static_balance": 9997890.774999978, "balance": 9994157.298999975, "available": 9985585.098999975, "float_profit": 60.0, "position_profit": 60.0, "close_profit": -3720.0, "frozen_margin": 0.0, "margin": 8572.2, "frozen_commission": 0.0, "commission": 73.476, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0008577211408167194, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"DCE.jd1909": {"exchange_id": "DCE", "instrument_id": "jd1909", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 3, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 3, "volume_short_his": 0, "volume_short": 3, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": 4417.0, "open_cost_long": 0.0, "open_cost_short": 132510.0, "position_price_long": NaN, "position_price_short": 4417.0, "position_cost_long": 0.0, "position_cost_short": 132510.0, "float_profit_long": 0.0, "float_profit_short": 60.0, "float_profit": 60.0, "position_profit_long": 0.0, "position_profit_short": 60.0, "position_profit": 60.0, "margin_long": 0.0, "margin_short": 8572.2, "margin": 8572.2, "last_price": 4415.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}}'
            self.assertEqual(excepted_trade_log, trade_log)

    # @unittest.skip("无条件跳过: test_rbreaker")
    def test_rbreaker(self):
        '''
        R-Breaker策略(隔夜留仓)
        设定：
            回测周期: 2019,5,1-2019,5,20
            合约代码: SHFE.au1912
            合约文件: "http://127.0.0.1:5000/t/md/symbols/2019-06-05.json"
        '''
        # 预设服务器端响应
        dir_path = os.path.dirname(os.path.realpath(__file__))
        self.mock.run(os.path.join(dir_path, "examples_log_file", "test_example_rbreaker.script.lzma"))
        # 测试
        SYMBOL = "SHFE.au1912"  # 合约代码
        STOP_LOSS_PRICE = 10  # 止损点(价格)

        def get_index_line(klines):
            '''计算指标线'''
            high = klines.high.iloc[-2]  # 前一日的最高价
            low = klines.low.iloc[-2]  # 前一日的最低价
            close = klines.close.iloc[-2]  # 前一日的收盘价

            pivot = (high + low + close) / 3  # 枢轴点
            b_break = high + 2 * (pivot - low)  # 突破买入价
            s_setup = pivot + (high - low)  # 观察卖出价
            s_enter = 2 * pivot - low  # 反转卖出价
            b_enter = 2 * pivot - high  # 反转买入价
            b_setup = pivot - (high - low)  # 观察买入价
            s_break = low - 2 * (high - pivot)  # 突破卖出价

            return pivot, b_break, s_setup, s_enter, b_enter, b_setup, s_break

        utils.RD = random.Random(4)
        sim = TqSim()
        api = TqApi(sim, backtest=TqBacktest(start_dt=date(2019, 5, 1), end_dt=date(2019, 6, 20)),
                    _ins_url=self.ins_url_2019_06_05, _td_url=self.td_url, _md_url=self.md_url)
        quote = api.get_quote(SYMBOL)
        klines = api.get_kline_serial(SYMBOL, 24 * 60 * 60)  # 86400: 使用日线
        position = api.get_position(SYMBOL)
        target_pos = TargetPosTask(api, SYMBOL)
        target_pos_value = position.volume_long - position.volume_short  # 目标净持仓数
        open_position_price = position.open_price_long if target_pos_value > 0 else position.open_price_short  # 开仓价
        pivot, b_break, s_setup, s_enter, b_enter, b_setup, s_break = get_index_line(klines)  # 七条标准线
        try:
            with closing(api):
                while True:
                    target_pos.set_target_volume(target_pos_value)

                    api.wait_update()
                    if api.is_changing(klines.iloc[-1], "datetime"):  # 产生新k线,则重新计算7条指标线
                        pivot, b_break, s_setup, s_enter, b_enter, b_setup, s_break = get_index_line(klines)
                    '''交易规则'''
                    if api.is_changing(quote, "last_price"):
                        # 开仓价与当前行情价之差大于止损点则止损
                        if (target_pos_value > 0 and open_position_price - quote.last_price >= STOP_LOSS_PRICE) or \
                                (target_pos_value < 0 and quote.last_price - open_position_price >= STOP_LOSS_PRICE):
                            target_pos_value = 0  # 平仓
                        # 反转:
                        if target_pos_value > 0:  # 多头持仓
                            if quote.highest > s_setup and quote.last_price < s_enter:
                                target_pos_value = -1  # 做空
                                open_position_price = quote.last_price
                        elif target_pos_value < 0:  # 空头持仓
                            if quote.lowest < b_setup and quote.last_price > b_enter:
                                target_pos_value = 1  # 做多
                                open_position_price = quote.last_price

                        # 突破:
                        elif target_pos_value == 0:  # 空仓条件
                            if quote.last_price > b_break:
                                target_pos_value = 1  # 做多
                                open_position_price = quote.last_price
                            elif quote.last_price < s_break:
                                target_pos_value = -1  # 做空
                                open_position_price = quote.last_price
        except BacktestFinished:
            # 测试
            trade_log = json.dumps(sim.trade_log)
            # print(trade_log)  # 用于生成测试用例except数据
            excepted_trade_log = '{"2019-05-01": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10000000.0, "static_balance": 10000000.0, "balance": 10000000.0, "available": 10000000.0, "float_profit": 0.0, "position_profit": 0.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {}}, "2019-05-06": {"trades": [{"user_id": "TQSIM", "order_id": "8534f45738d048ec0f1099c6c3e1b258", "trade_id": "8534f45738d048ec0f1099c6c3e1b258|1", "exchange_trade_id": "8534f45738d048ec0f1099c6c3e1b258|1", "exchange_id": "SHFE", "instrument_id": "au1912", "direction": "BUY", "offset": "OPEN", "price": 287.15000000000003, "volume": 1, "trade_date_time": 1557104459999999000, "commission": 10.0}], "account": {"currency": "CNY", "pre_balance": 10000000.0, "static_balance": 10000000.0, "balance": 9999090.0, "available": 9984090.0, "float_profit": -900.0000000000341, "position_profit": -900.0000000000341, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 10.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0015001365124226304, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 0, "pos_long_today": 1, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 1, "volume_long_his": 0, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 287.15000000000003, "position_price_short": NaN, "position_cost_long": 287150.00000000006, "position_cost_short": 0.0, "float_profit_long": -900.0000000000341, "float_profit_short": 0.0, "float_profit": -900.0000000000341, "position_profit_long": -900.0000000000341, "position_profit_short": 0.0, "position_profit": -900.0000000000341, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 286.25, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-07": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9999090.0, "static_balance": 9999090.0, "balance": 9998540.0, "available": 9983540.0, "float_profit": -1450.0000000000455, "position_profit": -550.0000000000114, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001500219031978669, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 286.25, "position_price_short": 286.25, "position_cost_long": 286250.0, "position_cost_short": 0.0, "float_profit_long": -1450.0000000000455, "float_profit_short": 0.0, "float_profit": -1450.0000000000455, "position_profit_long": -550.0000000000114, "position_profit_short": 0.0, "position_profit": -550.0000000000114, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 285.7, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-08": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9998540.0, "static_balance": 9998540.0, "balance": 9999990.0, "available": 9984990.0, "float_profit": -5.6843418860808015e-11, "position_profit": 1449.9999999999886, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0015000015000015, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 285.7, "position_price_short": 285.7, "position_cost_long": 285700.0, "position_cost_short": 0.0, "float_profit_long": -5.6843418860808015e-11, "float_profit_short": 0.0, "float_profit": -5.6843418860808015e-11, "position_profit_long": 1449.9999999999886, "position_profit_short": 0.0, "position_profit": 1449.9999999999886, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 287.15, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-09": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9999990.0, "static_balance": 9999990.0, "balance": 10000040.0, "available": 9985040.0, "float_profit": 49.999999999954525, "position_profit": 50.00000000001137, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014999940000239998, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 287.15, "position_price_short": 287.15, "position_cost_long": 287150.0, "position_cost_short": 0.0, "float_profit_long": 49.999999999954525, "float_profit_short": 0.0, "float_profit": 49.999999999954525, "position_profit_long": 50.00000000001137, "position_profit_short": 0.0, "position_profit": 50.00000000001137, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 287.2, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-10": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10000040.0, "static_balance": 10000040.0, "balance": 10000090.0, "available": 9985090.0, "float_profit": 99.9999999999659, "position_profit": 50.00000000001137, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001499986500121499, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 287.2, "position_price_short": 287.2, "position_cost_long": 287200.0, "position_cost_short": 0.0, "float_profit_long": 99.9999999999659, "float_profit_short": 0.0, "float_profit": 99.9999999999659, "position_profit_long": 50.00000000001137, "position_profit_short": 0.0, "position_profit": 50.00000000001137, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 287.25, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-13": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10000090.0, "static_balance": 10000090.0, "balance": 10002040.0, "available": 9987040.0, "float_profit": 2049.9999999999545, "position_profit": 1949.9999999999886, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014996940624112682, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 287.25, "position_price_short": 287.25, "position_cost_long": 287250.0, "position_cost_short": 0.0, "float_profit_long": 2049.9999999999545, "float_profit_short": 0.0, "float_profit": 2049.9999999999545, "position_profit_long": 1949.9999999999886, "position_profit_short": 0.0, "position_profit": 1949.9999999999886, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 289.2, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-14": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10002040.0, "static_balance": 10002040.0, "balance": 10005590.0, "available": 9990590.0, "float_profit": 5599.999999999965, "position_profit": 3550.000000000008, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001499161968459631, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 289.2, "position_price_short": 289.2, "position_cost_long": 289200.0, "position_cost_short": 0.0, "float_profit_long": 5599.999999999965, "float_profit_short": 0.0, "float_profit": 5599.999999999965, "position_profit_long": 3550.000000000008, "position_profit_short": 0.0, "position_profit": 3550.000000000008, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 292.75, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-15": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10005590.0, "static_balance": 10005590.0, "balance": 10004090.0, "available": 9989090.0, "float_profit": 4099.999999999967, "position_profit": -1500.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001499386750818915, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 292.75, "position_price_short": 292.75, "position_cost_long": 292750.0, "position_cost_short": 0.0, "float_profit_long": 4099.999999999967, "float_profit_short": 0.0, "float_profit": 4099.999999999967, "position_profit_long": -1500.0, "position_profit_short": 0.0, "position_profit": -1500.0, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 291.25, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-16": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10004090.0, "static_balance": 10004090.0, "balance": 10005190.0, "available": 9990190.0, "float_profit": 5199.999999999989, "position_profit": 1100.0000000000227, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014992219038319112, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 291.25, "position_price_short": 291.25, "position_cost_long": 291250.0, "position_cost_short": 0.0, "float_profit_long": 5199.999999999989, "float_profit_short": 0.0, "float_profit": 5199.999999999989, "position_profit_long": 1100.0000000000227, "position_profit_short": 0.0, "position_profit": 1100.0000000000227, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 292.35, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-17": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10005190.0, "static_balance": 10005190.0, "balance": 10004540.0, "available": 9989540.0, "float_profit": 4549.999999999955, "position_profit": -650.0000000000341, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014993193090336986, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 292.35, "position_price_short": 292.35, "position_cost_long": 292350.00000000006, "position_cost_short": 0.0, "float_profit_long": 4549.999999999955, "float_profit_short": 0.0, "float_profit": 4549.999999999955, "position_profit_long": -650.0000000000341, "position_profit_short": 0.0, "position_profit": -650.0000000000341, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 291.7, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-20": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10004540.0, "static_balance": 10004540.0, "balance": 10002240.0, "available": 9987240.0, "float_profit": 2249.9999999999445, "position_profit": -2300.0000000000114, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014996640752471446, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 291.7, "position_price_short": 291.7, "position_cost_long": 291700.0, "position_cost_short": 0.0, "float_profit_long": 2249.9999999999445, "float_profit_short": 0.0, "float_profit": 2249.9999999999445, "position_profit_long": -2300.0000000000114, "position_profit_short": 0.0, "position_profit": -2300.0000000000114, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 289.4, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-21": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10002240.0, "static_balance": 10002240.0, "balance": 10001990.0, "available": 9986990.0, "float_profit": 1999.9999999999445, "position_profit": -250.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014997015593896815, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 289.4, "position_price_short": 289.4, "position_cost_long": 289400.0, "position_cost_short": 0.0, "float_profit_long": 1999.9999999999445, "float_profit_short": 0.0, "float_profit": 1999.9999999999445, "position_profit_long": -250.0, "position_profit_short": 0.0, "position_profit": -250.0, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 289.15, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-22": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10001990.0, "static_balance": 10001990.0, "balance": 10001740.0, "available": 9986740.0, "float_profit": 1749.9999999999445, "position_profit": -250.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014997390454060994, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 289.15, "position_price_short": 289.15, "position_cost_long": 289150.0, "position_cost_short": 0.0, "float_profit_long": 1749.9999999999445, "float_profit_short": 0.0, "float_profit": 1749.9999999999445, "position_profit_long": -250.0, "position_profit_short": 0.0, "position_profit": -250.0, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 288.9, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-23": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10001740.0, "static_balance": 10001740.0, "balance": 10002190.0, "available": 9987190.0, "float_profit": 2199.99999999999, "position_profit": 450.0000000000455, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014996715719257482, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 288.9, "position_price_short": 288.9, "position_cost_long": 288900.0, "position_cost_short": 0.0, "float_profit_long": 2199.99999999999, "float_profit_short": 0.0, "float_profit": 2199.99999999999, "position_profit_long": 450.0000000000455, "position_profit_short": 0.0, "position_profit": 450.0000000000455, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 289.35, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-24": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10002190.0, "static_balance": 10002190.0, "balance": 10003940.0, "available": 9988940.0, "float_profit": 3949.999999999988, "position_profit": 1750.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014994092327622916, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 289.35, "position_price_short": 289.35, "position_cost_long": 289350.00000000006, "position_cost_short": 0.0, "float_profit_long": 3949.999999999988, "float_profit_short": 0.0, "float_profit": 3949.999999999988, "position_profit_long": 1750.0, "position_profit_short": 0.0, "position_profit": 1750.0, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 291.1, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-27": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10003940.0, "static_balance": 10003940.0, "balance": 10004090.0, "available": 9989090.0, "float_profit": 4099.999999999965, "position_profit": 149.99999999997726, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001499386750818915, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 291.1, "position_price_short": 291.1, "position_cost_long": 291100.00000000006, "position_cost_short": 0.0, "float_profit_long": 4099.999999999965, "float_profit_short": 0.0, "float_profit": 4099.999999999965, "position_profit_long": 149.99999999997726, "position_profit_short": 0.0, "position_profit": 149.99999999997726, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 291.25, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-28": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10004090.0, "static_balance": 10004090.0, "balance": 10003840.0, "available": 9988840.0, "float_profit": 3849.9999999999654, "position_profit": -250.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014994242210990979, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 291.25, "position_price_short": 291.25, "position_cost_long": 291250.0, "position_cost_short": 0.0, "float_profit_long": 3849.9999999999654, "float_profit_short": 0.0, "float_profit": 3849.9999999999654, "position_profit_long": -250.0, "position_profit_short": 0.0, "position_profit": -250.0, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 291.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-29": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10003840.0, "static_balance": 10003840.0, "balance": 10004290.0, "available": 9989290.0, "float_profit": 4299.999999999953, "position_profit": 449.99999999998863, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014993567759431204, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 291.0, "position_price_short": 291.0, "position_cost_long": 291000.0, "position_cost_short": 0.0, "float_profit_long": 4299.999999999953, "float_profit_short": 0.0, "float_profit": 4299.999999999953, "position_profit_long": 449.99999999998863, "position_profit_short": 0.0, "position_profit": 449.99999999998863, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 291.45, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-30": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10004290.0, "static_balance": 10004290.0, "balance": 10002940.0, "available": 9987940.0, "float_profit": 2949.9999999999864, "position_profit": -1349.999999999966, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001499559129615893, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 291.45, "position_price_short": 291.45, "position_cost_long": 291450.0, "position_cost_short": 0.0, "float_profit_long": 2949.9999999999864, "float_profit_short": 0.0, "float_profit": 2949.9999999999864, "position_profit_long": -1349.999999999966, "position_profit_short": 0.0, "position_profit": -1349.999999999966, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 290.1, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-05-31": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10002940.0, "static_balance": 10002940.0, "balance": 10006290.0, "available": 9991290.0, "float_profit": 6299.999999999949, "position_profit": 3349.999999999966, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014990570930884474, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 290.1, "position_price_short": 290.1, "position_cost_long": 290100.00000000006, "position_cost_short": 0.0, "float_profit_long": 6299.999999999949, "float_profit_short": 0.0, "float_profit": 6299.999999999949, "position_profit_long": 3349.999999999966, "position_profit_short": 0.0, "position_profit": 3349.999999999966, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 293.45, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-03": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10006290.0, "static_balance": 10006290.0, "balance": 10009540.0, "available": 9994540.0, "float_profit": 9549.999999999942, "position_profit": 3250.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014985703638728653, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 293.45, "position_price_short": 293.45, "position_cost_long": 293450.0, "position_cost_short": 0.0, "float_profit_long": 9549.999999999942, "float_profit_short": 0.0, "float_profit": 9549.999999999942, "position_profit_long": 3250.0, "position_profit_short": 0.0, "position_profit": 3250.0, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 296.7, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-04": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10009540.0, "static_balance": 10009540.0, "balance": 10012590.0, "available": 9997590.0, "float_profit": 12599.999999999949, "position_profit": 3050.0000000000114, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014981138746318384, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 296.7, "position_price_short": 296.7, "position_cost_long": 296700.0, "position_cost_short": 0.0, "float_profit_long": 12599.999999999949, "float_profit_short": 0.0, "float_profit": 12599.999999999949, "position_profit_long": 3050.0000000000114, "position_profit_short": 0.0, "position_profit": 3050.0000000000114, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 299.75, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-05": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10012590.0, "static_balance": 10012590.0, "balance": 10014540.0, "available": 9999540.0, "float_profit": 14549.999999999935, "position_profit": 1949.9999999999886, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014978221665698074, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 299.75, "position_price_short": 299.75, "position_cost_long": 299750.0, "position_cost_short": 0.0, "float_profit_long": 14549.999999999935, "float_profit_short": 0.0, "float_profit": 14549.999999999935, "position_profit_long": 1949.9999999999886, "position_profit_short": 0.0, "position_profit": 1949.9999999999886, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 301.7, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-06": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10014540.0, "static_balance": 10014540.0, "balance": 10014540.0, "available": 9999540.0, "float_profit": 14549.999999999933, "position_profit": 0.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014978221665698074, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 301.7, "position_price_short": 301.7, "position_cost_long": 301700.0, "position_cost_short": 0.0, "float_profit_long": 14549.999999999933, "float_profit_short": 0.0, "float_profit": 14549.999999999933, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 301.7, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-10": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10014540.0, "static_balance": 10014540.0, "balance": 10013690.0, "available": 9998690.0, "float_profit": 13699.999999999956, "position_profit": -849.9999999999659, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001497949307398172, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 301.7, "position_price_short": 301.7, "position_cost_long": 301700.0, "position_cost_short": 0.0, "float_profit_long": 13699.999999999956, "float_profit_short": 0.0, "float_profit": 13699.999999999956, "position_profit_long": -849.9999999999659, "position_profit_short": 0.0, "position_profit": -849.9999999999659, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 300.85, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-11": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10013690.0, "static_balance": 10013690.0, "balance": 10012390.0, "available": 9997390.0, "float_profit": 12399.999999999949, "position_profit": -1300.0000000000114, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001498143799832008, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 300.85, "position_price_short": 300.85, "position_cost_long": 300850.0, "position_cost_short": 0.0, "float_profit_long": 12399.999999999949, "float_profit_short": 0.0, "float_profit": 12399.999999999949, "position_profit_long": -1300.0000000000114, "position_profit_short": 0.0, "position_profit": -1300.0000000000114, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 299.55, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-12": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10012390.0, "static_balance": 10012390.0, "balance": 10015440.0, "available": 10000440.0, "float_profit": 15449.99999999995, "position_profit": 3050.0000000000114, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014976875703913157, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 299.55, "position_price_short": 299.55, "position_cost_long": 299550.0, "position_cost_short": 0.0, "float_profit_long": 15449.99999999995, "float_profit_short": 0.0, "float_profit": 15449.99999999995, "position_profit_long": 3050.0000000000114, "position_profit_short": 0.0, "position_profit": 3050.0000000000114, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 302.6, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-13": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10015440.0, "static_balance": 10015440.0, "balance": 10016040.0, "available": 10001040.0, "float_profit": 16049.999999999924, "position_profit": 599.9999999999659, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.001497597853043718, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 302.6, "position_price_short": 302.6, "position_cost_long": 302600.0, "position_cost_short": 0.0, "float_profit_long": 16049.999999999924, "float_profit_short": 0.0, "float_profit": 16049.999999999924, "position_profit_long": 599.9999999999659, "position_profit_short": 0.0, "position_profit": 599.9999999999659, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 303.2, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-14": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10016040.0, "static_balance": 10016040.0, "balance": 10020590.0, "available": 10005590.0, "float_profit": 20599.999999999935, "position_profit": 4550.000000000014, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014969178461547673, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 303.2, "position_price_short": 303.2, "position_cost_long": 303200.0, "position_cost_short": 0.0, "float_profit_long": 20599.999999999935, "float_profit_short": 0.0, "float_profit": 20599.999999999935, "position_profit_long": 4550.000000000014, "position_profit_short": 0.0, "position_profit": 4550.000000000014, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 307.75, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-17": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10020590.0, "static_balance": 10020590.0, "balance": 10016940.0, "available": 10001940.0, "float_profit": 16949.999999999956, "position_profit": -3649.9999999999777, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014974632971745862, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 307.75, "position_price_short": 307.75, "position_cost_long": 307750.0, "position_cost_short": 0.0, "float_profit_long": 16949.999999999956, "float_profit_short": 0.0, "float_profit": 16949.999999999956, "position_profit_long": -3649.9999999999777, "position_profit_short": 0.0, "position_profit": -3649.9999999999777, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 304.1, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-18": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10016940.0, "static_balance": 10016940.0, "balance": 10018990.0, "available": 10003990.0, "float_profit": 18999.999999999913, "position_profit": 2049.9999999999545, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014971568990487065, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 304.1, "position_price_short": 304.1, "position_cost_long": 304100.0, "position_cost_short": 0.0, "float_profit_long": 18999.999999999913, "float_profit_short": 0.0, "float_profit": 18999.999999999913, "position_profit_long": 2049.9999999999545, "position_profit_short": 0.0, "position_profit": 2049.9999999999545, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 306.15, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-19": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10018990.0, "static_balance": 10018990.0, "balance": 10017140.0, "available": 10002140.0, "float_profit": 17149.999999999938, "position_profit": -1849.999999999966, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014974333991538502, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 306.15, "position_price_short": 306.15, "position_cost_long": 306150.0, "position_cost_short": 0.0, "float_profit_long": 17149.999999999938, "float_profit_short": 0.0, "float_profit": 17149.999999999938, "position_profit_long": -1849.999999999966, "position_profit_short": 0.0, "position_profit": -1849.999999999966, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 304.3, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-06-20": {"trades": [], "account": {"currency": "CNY", "pre_balance": 10017140.0, "static_balance": 10017140.0, "balance": 10024090.0, "available": 10009090.0, "float_profit": 24099.999999999927, "position_profit": 6949.999999999992, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 15000.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0014963951840017398, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.au1912": {"exchange_id": "SHFE", "instrument_id": "au1912", "pos_long_his": 1, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 1, "volume_long": 1, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 287.15000000000003, "open_price_short": NaN, "open_cost_long": 287150.00000000006, "open_cost_short": 0.0, "position_price_long": 304.3, "position_price_short": 304.3, "position_cost_long": 304300.0, "position_cost_short": 0.0, "float_profit_long": 24099.999999999927, "float_profit_short": 0.0, "float_profit": 24099.999999999927, "position_profit_long": 6949.999999999992, "position_profit_short": 0.0, "position_profit": 6949.999999999992, "margin_long": 15000.0, "margin_short": 0.0, "margin": 15000.0, "last_price": 311.25, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}}'
            self.assertEqual(excepted_trade_log, trade_log)

    # @unittest.skip("无条件跳过: test_fairy_four_price")
    def test_fairy_four_price(self):
        '''
        设定：
            回测周期: 2019,4,1-2019,4,20
            合约代码: SHFE.cu1905
            合约文件: "http://127.0.0.1:5000/t/md/symbols/2019-06-10.json"
        '''
        # 预设服务器端响应
        dir_path = os.path.dirname(os.path.realpath(__file__))
        self.mock.run(os.path.join(dir_path, "examples_log_file", "test_example_fairy_four_price.script.lzma"))
        # # 测试
        symbol = "SHFE.cu1905"  # 合约代码
        close_hour, close_minute = 14, 50  # 平仓时间
        utils.RD = random.Random(4)
        sim = TqSim()
        api = TqApi(sim, backtest=TqBacktest(start_dt=datetime(2019, 4, 1), end_dt=datetime(2019, 4, 20)),
                    _ins_url=self.ins_url_2020_06_10, _td_url=self.td_url, _md_url=self.md_url)
        quote = api.get_quote(symbol)  # 获取指定合约的盘口行情
        klines = api.get_kline_serial(symbol, 24 * 60 * 60)  # 获取日线
        position = api.get_position(symbol)  # 持仓信息
        target_pos = TargetPosTask(api, symbol)  # 目标持仓

        top_rail = klines.high.iloc[-2]  # 上轨: 昨日高点
        bottom_rail = klines.low.iloc[-2]  # 下轨: 昨日低点

        try:
            with closing(api):
                while True:
                    api.wait_update()
                    if api.is_changing(klines.iloc[-1], "datetime"):  # 如果产生一根新日线 (即到达下一个交易日): 重新获取上下轨
                        top_rail = klines.high.iloc[-2]
                        bottom_rail = klines.low.iloc[-2]

                    if api.is_changing(quote, "last_price"):  # 如果行情最新价发生变化
                        # 开仓突破
                        if quote.last_price > top_rail and position.volume_long == 0:  # 如果价格突破上轨: 买入开仓
                            target_pos.set_target_volume(3)  # 设置目标持仓手数，将指定合约调整到目标头寸
                        elif quote.last_price < bottom_rail and position.volume_short == 0:  # 如果价格跌破下轨: 卖出开仓
                            target_pos.set_target_volume(-3)

                        # 平仓止损: 当价格 向上突破上轨 或 向下突破下轨 后, 再次回破当日开盘价
                        if (quote.highest > top_rail and quote.last_price <= quote.open) or (
                                quote.lowest < bottom_rail and quote.last_price >= quote.open):
                            target_pos.set_target_volume(0)

                    if api.is_changing(quote, "datetime"):
                        now_time = datetime.strptime(quote.datetime, "%Y-%m-%d %H:%M:%S.%f")  # 获取当前的行情时间
                        if now_time.hour == close_hour and now_time.minute >= close_minute:  # 到达平仓时间: 平仓
                            target_pos.set_target_volume(0)
                            top_rail = bottom_rail = float("nan")
        except BacktestFinished:
            # 测试
            trade_log = json.dumps(sim.trade_log)
            # print(trade_log)  # 用于生成测试用例except数据
            excepted_trade_log = '{"2019-04-01": {"trades": [{"user_id": "TQSIM", "order_id": "8534f45738d048ec0f1099c6c3e1b258", "trade_id": "8534f45738d048ec0f1099c6c3e1b258|3", "exchange_trade_id": "8534f45738d048ec0f1099c6c3e1b258|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49350.0, "volume": 3, "trade_date_time": 1554080459999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "c79d679346d4ac7a5c3902b38963dc6e", "trade_id": "c79d679346d4ac7a5c3902b38963dc6e|3", "exchange_trade_id": "c79d679346d4ac7a5c3902b38963dc6e|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "CLOSETODAY", "price": 49420.0, "volume": 3, "trade_date_time": 1554101459999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 10000000.0, "static_balance": 10000000.0, "balance": 10000980.489999998, "available": 10000980.489999998, "float_profit": 0.0, "position_profit": 0.0, "close_profit": 1050.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": NaN, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49460.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-02": {"trades": [{"user_id": "TQSIM", "order_id": "43000de01b2ed40ed3addccb2c33be0a", "trade_id": "43000de01b2ed40ed3addccb2c33be0a|3", "exchange_trade_id": "43000de01b2ed40ed3addccb2c33be0a|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "OPEN", "price": 49080.0, "volume": 3, "trade_date_time": 1554136139999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "06905269ed6f0b09f165c8ce36e2f24b", "trade_id": "06905269ed6f0b09f165c8ce36e2f24b|3", "exchange_trade_id": "06905269ed6f0b09f165c8ce36e2f24b|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "CLOSETODAY", "price": 49260.0, "volume": 3, "trade_date_time": 1554187859999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 10000980.489999998, "static_balance": 10000980.489999998, "balance": 9998210.979999997, "available": 9998210.979999997, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -2700.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": 49460.0, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49270.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-03": {"trades": [{"user_id": "TQSIM", "order_id": "42a00403ce80c4b0a4042bb3d4341aad", "trade_id": "42a00403ce80c4b0a4042bb3d4341aad|3", "exchange_trade_id": "42a00403ce80c4b0a4042bb3d4341aad|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "OPEN", "price": 49000.0, "volume": 3, "trade_date_time": 1554217499999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "2a3187853184ff27459142deccea2645", "trade_id": "2a3187853184ff27459142deccea2645|3", "exchange_trade_id": "2a3187853184ff27459142deccea2645|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "CLOSETODAY", "price": 49480.0, "volume": 3, "trade_date_time": 1554273059999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "de08caa1a08179104a25e4664f5253a0", "trade_id": "de08caa1a08179104a25e4664f5253a0|3", "exchange_trade_id": "de08caa1a08179104a25e4664f5253a0|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49480.0, "volume": 3, "trade_date_time": 1554273059999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "d93936e1daca3c06f5ff0c03bb5d7385", "trade_id": "d93936e1daca3c06f5ff0c03bb5d7385|3", "exchange_trade_id": "d93936e1daca3c06f5ff0c03bb5d7385|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "CLOSETODAY", "price": 49450.0, "volume": 3, "trade_date_time": 1554274259999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9998210.979999997, "static_balance": 9998210.979999997, "balance": 9990421.959999993, "available": 9990421.959999993, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -7650.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 139.02, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": NaN, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49510.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-04": {"trades": [{"user_id": "TQSIM", "order_id": "d8441b5616332aca5f552773e14b0190", "trade_id": "d8441b5616332aca5f552773e14b0190|3", "exchange_trade_id": "d8441b5616332aca5f552773e14b0190|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49570.0, "volume": 3, "trade_date_time": 1554303959999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "634f806fabf4a07c566002249b191bf4", "trade_id": "634f806fabf4a07c566002249b191bf4|3", "exchange_trade_id": "634f806fabf4a07c566002249b191bf4|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "CLOSETODAY", "price": 49480.0, "volume": 3, "trade_date_time": 1554360659999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9990421.959999993, "static_balance": 9990421.959999993, "balance": 9989002.449999992, "available": 9989002.449999992, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -1350.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": NaN, "position_price_short": 49510.0, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49520.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-08": {"trades": [{"user_id": "TQSIM", "order_id": "3f5082492d83a8233fb62d2c81862fc9", "trade_id": "3f5082492d83a8233fb62d2c81862fc9|3", "exchange_trade_id": "3f5082492d83a8233fb62d2c81862fc9|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "OPEN", "price": 49130.0, "volume": 3, "trade_date_time": 1554685259999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "f1cfd99216df648647adec26793d0e45", "trade_id": "f1cfd99216df648647adec26793d0e45|3", "exchange_trade_id": "f1cfd99216df648647adec26793d0e45|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "CLOSETODAY", "price": 49290.0, "volume": 3, "trade_date_time": 1554706259999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9989002.449999992, "static_balance": 9989002.449999992, "balance": 9986532.93999999, "available": 9986532.93999999, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -2400.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": 49520.0, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49260.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-09": {"trades": [{"user_id": "TQSIM", "order_id": "f1347e0cdd905ecfd160c5d0ef412ed6", "trade_id": "f1347e0cdd905ecfd160c5d0ef412ed6|3", "exchange_trade_id": "f1347e0cdd905ecfd160c5d0ef412ed6|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49400.0, "volume": 3, "trade_date_time": 1554729299999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "01d89a024cdce7a6d7288ff68c320f89", "trade_id": "01d89a024cdce7a6d7288ff68c320f89|3", "exchange_trade_id": "01d89a024cdce7a6d7288ff68c320f89|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "CLOSETODAY", "price": 49720.0, "volume": 3, "trade_date_time": 1554792659999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9986532.93999999, "static_balance": 9986532.93999999, "balance": 9991263.429999989, "available": 9991263.429999989, "float_profit": 0.0, "position_profit": 0.0, "close_profit": 4800.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": NaN, "position_price_short": 49260.0, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49790.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-10": {"trades": [], "account": {"currency": "CNY", "pre_balance": 9991263.429999989, "static_balance": 9991263.429999989, "balance": 9991263.429999989, "available": 9991263.429999989, "float_profit": 0.0, "position_profit": 0.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 0.0, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": 49790.0, "position_price_short": 49790.0, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49490.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-11": {"trades": [{"user_id": "TQSIM", "order_id": "b474c7e89286a1754abcb06ae8abb93f", "trade_id": "b474c7e89286a1754abcb06ae8abb93f|3", "exchange_trade_id": "b474c7e89286a1754abcb06ae8abb93f|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "OPEN", "price": 49300.0, "volume": 3, "trade_date_time": 1554901259999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "c3e4a892d9196ada4fcfa583e1df8af9", "trade_id": "c3e4a892d9196ada4fcfa583e1df8af9|3", "exchange_trade_id": "c3e4a892d9196ada4fcfa583e1df8af9|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "CLOSETODAY", "price": 49220.0, "volume": 3, "trade_date_time": 1554965459999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9991263.429999989, "static_balance": 9991263.429999989, "balance": 9992393.919999987, "available": 9992393.919999987, "float_profit": 0.0, "position_profit": 0.0, "close_profit": 1200.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": 49490.0, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49170.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-12": {"trades": [{"user_id": "TQSIM", "order_id": "6c79a3de69f85e3131f3b9238224b122", "trade_id": "6c79a3de69f85e3131f3b9238224b122|3", "exchange_trade_id": "6c79a3de69f85e3131f3b9238224b122|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "OPEN", "price": 49110.0, "volume": 3, "trade_date_time": 1554992459999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "738d243a6e58d5ca49c7b59b995253fd", "trade_id": "738d243a6e58d5ca49c7b59b995253fd|3", "exchange_trade_id": "738d243a6e58d5ca49c7b59b995253fd|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "CLOSETODAY", "price": 49210.0, "volume": 3, "trade_date_time": 1555051859999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9992393.919999987, "static_balance": 9992393.919999987, "balance": 9990824.409999985, "available": 9990824.409999985, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -1500.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": 49170.0, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49200.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-15": {"trades": [{"user_id": "TQSIM", "order_id": "4278c2614e1bcb383bb4a570294c4ea3", "trade_id": "4278c2614e1bcb383bb4a570294c4ea3|3", "exchange_trade_id": "4278c2614e1bcb383bb4a570294c4ea3|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49580.0, "volume": 3, "trade_date_time": 1555074059999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "14c15c910b11ad28cc21ce88d0060cc5", "trade_id": "14c15c910b11ad28cc21ce88d0060cc5|3", "exchange_trade_id": "14c15c910b11ad28cc21ce88d0060cc5|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "CLOSETODAY", "price": 49370.0, "volume": 3, "trade_date_time": 1555311059999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9990824.409999985, "static_balance": 9990824.409999985, "balance": 9987604.899999984, "available": 9987604.899999984, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -3150.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": NaN, "position_price_short": 49200.0, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49320.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-16": {"trades": [{"user_id": "TQSIM", "order_id": "ff5a52f1a05885ac7671863c0bdbc23a", "trade_id": "ff5a52f1a05885ac7671863c0bdbc23a|3", "exchange_trade_id": "ff5a52f1a05885ac7671863c0bdbc23a|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "OPEN", "price": 49300.0, "volume": 3, "trade_date_time": 1555333919999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "a5e333cb88dcf94384d4cd1f47ca7883", "trade_id": "a5e333cb88dcf94384d4cd1f47ca7883|3", "exchange_trade_id": "a5e333cb88dcf94384d4cd1f47ca7883|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "CLOSETODAY", "price": 49480.0, "volume": 3, "trade_date_time": 1555397459999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9987604.899999984, "static_balance": 9987604.899999984, "balance": 9984835.389999982, "available": 9984835.389999982, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -2700.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": 49320.0, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49450.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-17": {"trades": [{"user_id": "TQSIM", "order_id": "2522d53857c49391b36cc9aa78a330a1", "trade_id": "2522d53857c49391b36cc9aa78a330a1|3", "exchange_trade_id": "2522d53857c49391b36cc9aa78a330a1|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49610.0, "volume": 3, "trade_date_time": 1555482899999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "11021c9e32111ac1ac7cc4a4ff4dab10", "trade_id": "11021c9e32111ac1ac7cc4a4ff4dab10|3", "exchange_trade_id": "11021c9e32111ac1ac7cc4a4ff4dab10|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "CLOSETODAY", "price": 49530.0, "volume": 3, "trade_date_time": 1555483859999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9984835.389999982, "static_balance": 9984835.389999982, "balance": 9983565.87999998, "available": 9983565.87999998, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -1200.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": NaN, "position_price_short": 49450.0, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49470.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-18": {"trades": [{"user_id": "TQSIM", "order_id": "a2909cb633e238b4e9dd38b869ace913", "trade_id": "a2909cb633e238b4e9dd38b869ace913|3", "exchange_trade_id": "a2909cb633e238b4e9dd38b869ace913|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49780.0, "volume": 3, "trade_date_time": 1555506059999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "2f0733c846bbe9e870ef55b1a1f65507", "trade_id": "2f0733c846bbe9e870ef55b1a1f65507|3", "exchange_trade_id": "2f0733c846bbe9e870ef55b1a1f65507|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "CLOSETODAY", "price": 49460.0, "volume": 3, "trade_date_time": 1555570259999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9983565.87999998, "static_balance": 9983565.87999998, "balance": 9978696.369999979, "available": 9978696.369999979, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -4800.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": NaN, "position_price_short": 49470.0, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49460.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-19": {"trades": [{"user_id": "TQSIM", "order_id": "96b98b5fbf37a2be6f98bca35b17b966", "trade_id": "96b98b5fbf37a2be6f98bca35b17b966|3", "exchange_trade_id": "96b98b5fbf37a2be6f98bca35b17b966|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "SELL", "offset": "OPEN", "price": 49280.0, "volume": 3, "trade_date_time": 1555592459999999000, "commission": 34.755}, {"user_id": "TQSIM", "order_id": "32decd6b8efbc170a26a25c852175b7a", "trade_id": "32decd6b8efbc170a26a25c852175b7a|3", "exchange_trade_id": "32decd6b8efbc170a26a25c852175b7a|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "CLOSETODAY", "price": 49490.0, "volume": 3, "trade_date_time": 1555656659999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9978696.369999979, "static_balance": 9978696.369999979, "balance": 9975476.859999977, "available": 9975476.859999977, "float_profit": 0.0, "position_profit": 0.0, "close_profit": -3150.0, "frozen_margin": 0.0, "margin": 0.0, "frozen_commission": 0.0, "commission": 69.51, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 0, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 0, "volume_long_his": 0, "volume_long": 0, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": NaN, "open_price_short": NaN, "open_cost_long": 0.0, "open_cost_short": 0.0, "position_price_long": 49460.0, "position_price_short": NaN, "position_cost_long": 0.0, "position_cost_short": 0.0, "float_profit_long": 0.0, "float_profit_short": 0.0, "float_profit": 0.0, "position_profit_long": 0.0, "position_profit_short": 0.0, "position_profit": 0.0, "margin_long": 0.0, "margin_short": 0.0, "margin": 0.0, "last_price": 49500.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}, "2019-04-22": {"trades": [{"user_id": "TQSIM", "order_id": "d6e4a51519d9c9cc52d32377e78131c1", "trade_id": "d6e4a51519d9c9cc52d32377e78131c1|3", "exchange_trade_id": "d6e4a51519d9c9cc52d32377e78131c1|3", "exchange_id": "SHFE", "instrument_id": "cu1905", "direction": "BUY", "offset": "OPEN", "price": 49540.0, "volume": 3, "trade_date_time": 1555679579999999000, "commission": 34.755}], "account": {"currency": "CNY", "pre_balance": 9975476.859999977, "static_balance": 9975476.859999977, "balance": 9974842.104999976, "available": 9926185.104999976, "float_profit": -600.0, "position_profit": -600.0, "close_profit": 0.0, "frozen_margin": 0.0, "margin": 48657.0, "frozen_commission": 0.0, "commission": 34.755, "frozen_premium": 0.0, "premium": 0.0, "deposit": 0.0, "withdraw": 0.0, "risk_ratio": 0.0048779719506146626, "market_value": 0.0, "ctp_balance": NaN, "ctp_available": NaN}, "positions": {"SHFE.cu1905": {"exchange_id": "SHFE", "instrument_id": "cu1905", "pos_long_his": 0, "pos_long_today": 3, "pos_short_his": 0, "pos_short_today": 0, "volume_long_today": 3, "volume_long_his": 0, "volume_long": 3, "volume_long_frozen_today": 0, "volume_long_frozen_his": 0, "volume_long_frozen": 0, "volume_short_today": 0, "volume_short_his": 0, "volume_short": 0, "volume_short_frozen_today": 0, "volume_short_frozen_his": 0, "volume_short_frozen": 0, "open_price_long": 49540.0, "open_price_short": NaN, "open_cost_long": 743100.0, "open_cost_short": 0.0, "position_price_long": 49540.0, "position_price_short": 49500.0, "position_cost_long": 743100.0, "position_cost_short": 0.0, "float_profit_long": -600.0, "float_profit_short": 0.0, "float_profit": -600.0, "position_profit_long": -600.0, "position_profit_short": 0.0, "position_profit": -600.0, "margin_long": 48657.0, "margin_short": 0.0, "margin": 48657.0, "last_price": 49500.0, "market_value_long": 0.0, "market_value_short": 0.0, "market_value": 0.0}}}}'
            self.assertEqual(excepted_trade_log, trade_log)
