import { Chart, ChartProps, Size } from 'cdk8s';
import { CpuResources, MemoryResources, PersistentVolumeAccessMode, ServiceType } from 'cdk8s-plus-26';
import { Construct } from 'constructs';
/**
 * Props for configuring the valheim server backups
 */
export interface BackupProps {
    /**
     * Should backups be enabled
     * @default true
     */
    readonly enabled: boolean;
    /**
     * The cron schedule for the backup job
     * @default 0 * * * *
     */
    readonly scheduleCron?: string;
    /**
     * The directory to store backups
     * @default /config/backups
     */
    readonly directory?: string;
    /**
     * The retention age for backups
     * @default 3
     */
    readonly retentionAge?: number;
    /**
     * The retention count for backups
     * @default unlimited
     */
    readonly maxBackups?: number;
    /**
     * Only backup if server idle
     * @default true
     */
    readonly performIfIdle?: boolean;
    /**
     * The grace period for the server to be idle
     * @default 3600s
     */
    readonly idleGracePeriod?: number;
    /**
     * Should the backups be zipped
     * @default true
     */
    readonly zip?: boolean;
    /**
     * Permission mask for the backup directory
     */
    readonly permissionUmask?: string;
}
/**
 * Props for configuring valheim plus
 */
export interface ValheimPlusProps {
    /**
     * Should valheim plus be enabled
     * @default false
     */
    readonly enabled: boolean;
    /**
     * The version of valheim plus to use
     * @default latest
     */
    readonly release?: string;
}
/**
 * Props for configuring the supervisor
 */
export interface SupervisorHttpProps {
    /**
     * Should the supervisor http server be enabled
     * @default false
     */
    readonly enabled: boolean;
    /**
     * The port the supervisor http server runs on
     * @default 9001
     */
    readonly port?: number;
    /**
     * The supervisor username
     * @default admin
     */
    readonly username?: string;
    /**
     * The supervisor password
     */
    readonly password: PasswordProps;
    /**
     * The service type for the supervisor http server
     * @default ServiceType.CLUSTER_IP
     */
    readonly serviceType?: ServiceType;
}
/**
 * Props for configuring the status http server
 */
export interface StatusHttpProps {
    /**
     * Should the status http server be enabled
     * @default false
     */
    readonly enabled: boolean;
    /**
     * The port the status http server runs on
     * @default 80
     */
    readonly port?: number;
    /**
     * Path to the busybox httpd config
     * @deafult /config/httpd.conf
     */
    readonly configPath?: string;
    /**
     * Path to the status httpd htdocs where status.json is written
     * @deafult /opt/valheim/htdocs
     */
    readonly htdocLocation?: string;
    /**
     * The service type for the status http server
     * @default ServiceType.CLUSTER_IP
     */
    readonly serviceType?: ServiceType;
}
/**
 * Props for configuring syslog
 */
export interface SysLogProps {
    /**
     * The remote syslog host
     */
    readonly remoteHost?: string;
    /**
     * The remote syslog port
     * @default 514
     */
    readonly remotePort?: number;
    /**
     * Should logging be done local
     */
    readonly logLocal?: boolean;
}
/**
 * Password properties. Used to determine if the password should be a raw string in manifest or retrieved from an existing secret
 */
export interface PasswordProps {
    /**
     * The raw password string. Will be visible in manifest. Should not use.
     */
    readonly raw?: string;
    /**
     * The name of the secret to retrieve the password from. The secret should be stored in a key named "password"
     */
    readonly secret?: string;
}
/**
 * Props for configuring a Valheim server
 */
export interface ServerProps {
    /**
     * The port the server runs on. This and the port + 1 must be open on the host
     * The specified port is used for game conncections, and the increment port is
     * used for the server query
     * @default 2456
     */
    readonly port?: number;
    /**
     * The name of the server
     * @default "My Server"
     */
    readonly name?: string;
    /**
     * The world name
     * @default "Dedicated"
     */
    readonly worldName?: string;
    /**
     * The server password
     */
    readonly password?: PasswordProps;
    /**
     * If the server is public
     * @default true
     */
    readonly public?: boolean;
    /**
     * Arguments to pass to the server on start
     */
    readonly launchArgs?: string;
    /**
     * Space separated list of admin SteamIDs in SteamID64 format. Overrides any existing adminlist.txt entries!
     */
    readonly adminList?: string[];
    /**
     * Space separated list of banned SteamIDs in SteamID64 format. Overrides any existing banlist.txt entries!
     */
    readonly blockList?: string[];
    /**
     * Space separated list of allowed SteamIDs in SteamID64 format. Overrides any existing permittedlist.txt entries!
     */
    readonly allowList?: string[];
    /**
     * Should enable crossplay
     */
    readonly crossplay?: boolean;
    /**
     * The server update schedule
     * @default "*\/15 * * * *"
     */
    readonly updateCron?: string;
    /**
     * The time window, in seconds, to wait for incoming UDP datagrams on non-public servers before determining if the server is idle
     */
    readonly idleDatagramWindow?: number;
    /**
     * The number of incoming UDP datagrams the container should tolerate (including useless datagrams such as mDNS, as well as useful datagrams like queries against the UDP query port and active connections by players) on non-public servers before deciding that the server is not idle
     */
    readonly idleDatagramMaxCount?: number;
    /**
     * Only run update check if no players are connected to the server (true or false)
     * @default true
     */
    readonly updateWhenIdle?: boolean;
    /**
     * The server restart schedule
     * @default "0 5 * * *"
     */
    readonly restartCron?: string;
    /**
     * Only restart the server if no players are connected to the server (true or false)
     * @default true
     */
    readonly restartIfIdle?: boolean;
    /**
     * The container timezone
     * @default "Etc/UTC
     */
    readonly timezone?: string;
    /**
     * If the beta server branch should be used
     */
    readonly publicBeta?: boolean;
    /**
     * The service type in the cluster to expose the server on
     * @default ServiceType.LOAD_BALANCER
     */
    readonly serviceType?: ServiceType;
    /**
     * The arguments to pass to the steamcmd command
     */
    readonly steamCmdArgs?: string;
    /**
     * Properties for ValheimPlus
     */
    readonly valheimPlus?: ValheimPlusProps;
}
export interface PersistanceProps {
    /**
     * PVC configuration for server specific files
     */
    readonly server?: PersistentVolumeClaimConfigProps;
    /**
     * PVC configuration for data specific files
     */
    readonly config?: PersistentVolumeClaimConfigProps;
}
/**
 * Props for configuring a persistent volume claim
 * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/
 *
 */
export interface PersistentVolumeClaimConfigProps {
    /**
     * The name of the storage class
     */
    readonly storageClass: string;
    /**
     * The access mode from the volume
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#access-modes
     * @default = [READ_WRITE_ONCE]
     */
    readonly accessModes?: PersistentVolumeAccessMode[];
    /**
     * The size of the volume
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#capacity
     */
    readonly storage?: Size;
}
/**
 * Props for configuring resource limits
 * @see https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
*/
export interface ResourceLimitsProps {
    /**
     * The CPU resources to allocate to the container
     * @see https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/#meaning-of-cpu
     * @default = 2000m
     */
    readonly cpu?: CpuResources;
    /**
     * The memory resources to allocate to the container
     * @see https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/#meaning-of-memory
     * @default = 4Gi
     */
    readonly memory?: MemoryResources;
}
/**
 * Props for configuring security aspects of the container
 */
export interface SecurityProps {
    readonly user?: number;
    readonly group?: number;
    readonly privileged?: boolean;
    readonly readOnlyRootFilesystem?: boolean;
    readonly allowPrivilegeEscalation?: boolean;
}
/**
 * The props for the chart
 */
export interface ValheimChartProps extends ChartProps {
    readonly server?: ServerProps;
    readonly persistence?: PersistanceProps;
    readonly imageTag?: string;
    readonly resourceLimits?: ResourceLimitsProps;
    readonly backup?: BackupProps;
    readonly supervisorHttp?: SupervisorHttpProps;
    readonly statusHttp?: StatusHttpProps;
    readonly sysLog?: SysLogProps;
    readonly security?: SecurityProps;
}
/**
 * A chart to deploy a Valheim server
 * Uses the container by @lloesche
 * @see https://github.com/lloesche/valheim-server-docker
 */
export declare class ValheimChart extends Chart {
    constructor(scope: Construct, name: string, props?: ValheimChartProps);
    private formImage;
    private formPortRecords;
    private formService;
    private formHostVolume;
    private formPersistanceVolume;
    private formServerPersistance;
    private formConfigPersistance;
    private formPasswordEnvValue;
    private formEnvironment;
}
