"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateHostedZone = exports.PublicHostedZone = exports.HostedZone = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * Container for records, and records contain information about how to route traffic for a
 * specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com)
 */
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.anyValue({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs }),
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * Use when hosted zone ID is known. Hosted zone name becomes unavailable through this query.
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param hostedZoneId the ID of the hosted zone to import
     */
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error('HostedZone.fromHostedZoneId doesn\'t support "zoneName"');
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     *
     * Use when both hosted zone ID and hosted zone name are known.
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param attrs the HostedZoneAttributes (hosted zone ID and hosted zone name)
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     * Requires environment, you must specify env for the stack.
     *
     * Use to easily query hosted zones.
     *
     * @see https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query,
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param publicHostedZoneId the ID of the public hosted zone to import
     */
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this,
            });
        }
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param privateHostedZoneId the ID of the private hosted zone to import
     */
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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