"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.anyValue({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs })
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error(`HostedZone.fromHostedZoneId doesn't support "zoneName"`);
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error(`cannot retrieve "zoneName" from an an imported hosted zone`); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this
            });
        }
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error(`cannot retrieve "zoneName" from an an imported hosted zone`); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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