"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * The record type.
 */
var RecordType;
(function (RecordType) {
    RecordType["A"] = "A";
    RecordType["AAAA"] = "AAAA";
    RecordType["CAA"] = "CAA";
    RecordType["CNAME"] = "CNAME";
    RecordType["MX"] = "MX";
    RecordType["NAPTR"] = "NAPTR";
    RecordType["NS"] = "NS";
    RecordType["PTR"] = "PTR";
    RecordType["SOA"] = "SOA";
    RecordType["SPF"] = "SPF";
    RecordType["SRV"] = "SRV";
    RecordType["TXT"] = "TXT";
})(RecordType = exports.RecordType || (exports.RecordType = {}));
/**
 * Type union for a record that accepts multiple types of target.
 */
class RecordTarget {
    constructor(values, aliasTarget) {
        this.values = values;
        this.aliasTarget = aliasTarget;
    }
    /**
     * Use string values as target.
     */
    static fromValues(...values) {
        return new RecordTarget(values);
    }
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget) {
        return new RecordTarget(undefined, aliasTarget);
    }
    /**
     * Use ip adresses as target.
     */
    static fromIpAddresses(...ipAddresses) {
        return RecordTarget.fromValues(...ipAddresses);
    }
}
exports.RecordTarget = RecordTarget;
/**
 * A record set.
 */
class RecordSet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const ttl = props.target.aliasTarget ? undefined : ((props.ttl && props.ttl.toSeconds()) || 1800).toString();
        const recordSet = new route53_generated_1.CfnRecordSet(this, 'Resource', {
            hostedZoneId: props.zone.hostedZoneId,
            name: util_1.determineFullyQualifiedDomainName(props.recordName || props.zone.zoneName, props.zone),
            type: props.recordType,
            resourceRecords: props.target.values,
            aliasTarget: props.target.aliasTarget && props.target.aliasTarget.bind(this),
            ttl,
            comment: props.comment
        });
        this.domainName = recordSet.ref;
    }
}
exports.RecordSet = RecordSet;
/**
 * @deprecated Use RecordTarget
 */
class AddressRecordTarget extends RecordTarget {
}
exports.AddressRecordTarget = AddressRecordTarget;
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
class ARecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.A,
            target: props.target,
        });
    }
}
exports.ARecord = ARecord;
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class AaaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.AAAA,
            target: props.target,
        });
    }
}
exports.AaaaRecord = AaaaRecord;
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
class CnameRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CNAME,
            target: RecordTarget.fromValues(props.domainName)
        });
    }
}
exports.CnameRecord = CnameRecord;
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
class TxtRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.TXT,
            target: RecordTarget.fromValues(...props.values.map(v => JSON.stringify(v))),
        });
    }
}
exports.TxtRecord = TxtRecord;
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
class SrvRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.SRV,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.weight} ${v.port} ${v.hostName}`)),
        });
    }
}
exports.SrvRecord = SrvRecord;
/**
 * The CAA tag.
 */
var CaaTag;
(function (CaaTag) {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    CaaTag["ISSUE"] = "issue";
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    CaaTag["ISSUEWILD"] = "issuewild";
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    CaaTag["IODEF"] = "iodef";
})(CaaTag = exports.CaaTag || (exports.CaaTag = {}));
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CAA,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.flag} ${v.tag} "${v.value}"`)),
        });
    }
}
exports.CaaRecord = CaaRecord;
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaAmazonRecord extends CaaRecord {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            values: [
                {
                    flag: 0,
                    tag: CaaTag.ISSUE,
                    value: 'amazon.com'
                }
            ],
        });
    }
}
exports.CaaAmazonRecord = CaaAmazonRecord;
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
class MxRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.MX,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.hostName}`))
        });
    }
}
exports.MxRecord = MxRecord;
/**
 * A record to delegate further lookups to a different set of name servers.
 */
class ZoneDelegationRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.NS,
            target: RecordTarget.fromValues(...core_1.Token.isUnresolved(props.nameServers)
                ? props.nameServers // Can't map a string-array token!
                : props.nameServers.map(ns => (core_1.Token.isUnresolved(ns) || ns.endsWith('.')) ? ns : `${ns}.`)),
            ttl: props.ttl || core_1.Duration.days(2)
        });
    }
}
exports.ZoneDelegationRecord = ZoneDelegationRecord;
//# sourceMappingURL=data:application/json;base64,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