"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
/**
 * Validates a zone name is valid by Route53 specifc naming rules,
 * and that there is no trailing dot in the name.
 *
 * @param zoneName the zone name to be validated.
 * @returns +zoneName+
 * @throws ValidationError if the name is not valid.
 */
function validateZoneName(zoneName) {
    if (zoneName.endsWith('.')) {
        throw new ValidationError('zone name must not end with a trailing dot');
    }
    if (zoneName.length > 255) {
        throw new ValidationError('zone name cannot be more than 255 bytes long');
    }
    if (zoneName.split('.').find(label => label.length > 63)) {
        throw new ValidationError('zone name labels cannot be more than 63 bytes long');
    }
    if (!zoneName.match(/^[a-z0-9!"#$%&'()*+,/:;<=>?@[\\\]^_`{|}~.-]+$/i)) {
        throw new ValidationError('zone names can only contain a-z, 0-9, -, ! " # $ % & \' ( ) * + , - / : ; < = > ? @ [ \ ] ^ _ ` { | } ~ .');
    }
}
exports.validateZoneName = validateZoneName;
class ValidationError extends Error {
    constructor(message) {
        super(message);
    }
}
/**
 * Route53 requires the record names are specified as fully qualified names, but this
 * forces lots of redundant work on the user (repeating the zone name over and over).
 * This function allows the user to be lazier and offers a nicer experience, by
 * qualifying relative names appropriately:
 *
 * @param providedName the user-specified name of the record.
 * @param zoneName   the fully-qualified name of the zone the record will be created in.
 *
 * @returns <ul>
 *        <li>If +providedName+ ends with a +.+, use it as-is</li>
 *        <li>If +providedName+ ends with or equals +zoneName+, append a trailing +.+</li>
 *        <li>Otherwise, append +.+, +zoneName+ and a trailing +.+</li>
 *      </ul>
 */
function determineFullyQualifiedDomainName(providedName, hostedZone) {
    if (providedName.endsWith('.')) {
        return providedName;
    }
    const hostedZoneName = hostedZone.zoneName.endsWith('.')
        ? hostedZone.zoneName.substring(0, hostedZone.zoneName.length - 1)
        : hostedZone.zoneName;
    const suffix = `.${hostedZoneName}`;
    if (providedName.endsWith(suffix) || providedName === hostedZoneName) {
        return `${providedName}.`;
    }
    return `${providedName}${suffix}.`;
}
exports.determineFullyQualifiedDomainName = determineFullyQualifiedDomainName;
function makeHostedZoneArn(construct, hostedZoneId) {
    return core_1.Stack.of(construct).formatArn({
        account: '',
        region: '',
        service: 'route53',
        resource: 'hostedzone',
        resourceName: hostedZoneId
    });
}
exports.makeHostedZoneArn = makeHostedZoneArn;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInV0aWwudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBaUQ7QUFHakQ7Ozs7Ozs7R0FPRztBQUNILFNBQWdCLGdCQUFnQixDQUFDLFFBQWdCO0lBQy9DLElBQUksUUFBUSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUMxQixNQUFNLElBQUksZUFBZSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7S0FDekU7SUFDRCxJQUFJLFFBQVEsQ0FBQyxNQUFNLEdBQUcsR0FBRyxFQUFFO1FBQ3pCLE1BQU0sSUFBSSxlQUFlLENBQUMsOENBQThDLENBQUMsQ0FBQztLQUMzRTtJQUNELElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLEVBQUUsQ0FBQyxFQUFFO1FBQ3hELE1BQU0sSUFBSSxlQUFlLENBQUMsb0RBQW9ELENBQUMsQ0FBQztLQUNqRjtJQUNELElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGdEQUFnRCxDQUFDLEVBQUU7UUFDckUsTUFBTSxJQUFJLGVBQWUsQ0FBQywyR0FBMkcsQ0FBQyxDQUFDO0tBQ3hJO0FBQ0gsQ0FBQztBQWJELDRDQWFDO0FBRUQsTUFBTSxlQUFnQixTQUFRLEtBQUs7SUFDakMsWUFBWSxPQUFlO1FBQ3pCLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNqQixDQUFDO0NBQ0Y7QUFFRDs7Ozs7Ozs7Ozs7Ozs7R0FjRztBQUNILFNBQWdCLGlDQUFpQyxDQUFDLFlBQW9CLEVBQUUsVUFBdUI7SUFDN0YsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQzlCLE9BQU8sWUFBWSxDQUFDO0tBQ3JCO0lBRUQsTUFBTSxjQUFjLEdBQUksVUFBVSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQ3ZELENBQUMsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsVUFBVSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1FBQ2xFLENBQUMsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDO0lBRXhCLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxFQUFFLENBQUM7SUFDcEMsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLFlBQVksS0FBSyxjQUFjLEVBQUU7UUFDcEUsT0FBTyxHQUFHLFlBQVksR0FBRyxDQUFDO0tBQzNCO0lBRUQsT0FBTyxHQUFHLFlBQVksR0FBRyxNQUFNLEdBQUcsQ0FBQztBQUNyQyxDQUFDO0FBZkQsOEVBZUM7QUFFRCxTQUFnQixpQkFBaUIsQ0FBQyxTQUFvQixFQUFFLFlBQW9CO0lBQzFFLE9BQU8sWUFBSyxDQUFDLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDbkMsT0FBTyxFQUFFLEVBQUU7UUFDWCxNQUFNLEVBQUUsRUFBRTtRQUNWLE9BQU8sRUFBRSxTQUFTO1FBQ2xCLFFBQVEsRUFBRSxZQUFZO1FBQ3RCLFlBQVksRUFBRSxZQUFZO0tBQzNCLENBQUMsQ0FBQztBQUNMLENBQUM7QUFSRCw4Q0FRQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbnN0cnVjdCwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IElIb3N0ZWRab25lIH0gZnJvbSAnLi9ob3N0ZWQtem9uZS1yZWYnO1xuXG4vKipcbiAqIFZhbGlkYXRlcyBhIHpvbmUgbmFtZSBpcyB2YWxpZCBieSBSb3V0ZTUzIHNwZWNpZmMgbmFtaW5nIHJ1bGVzLFxuICogYW5kIHRoYXQgdGhlcmUgaXMgbm8gdHJhaWxpbmcgZG90IGluIHRoZSBuYW1lLlxuICpcbiAqIEBwYXJhbSB6b25lTmFtZSB0aGUgem9uZSBuYW1lIHRvIGJlIHZhbGlkYXRlZC5cbiAqIEByZXR1cm5zICt6b25lTmFtZStcbiAqIEB0aHJvd3MgVmFsaWRhdGlvbkVycm9yIGlmIHRoZSBuYW1lIGlzIG5vdCB2YWxpZC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHZhbGlkYXRlWm9uZU5hbWUoem9uZU5hbWU6IHN0cmluZykge1xuICBpZiAoem9uZU5hbWUuZW5kc1dpdGgoJy4nKSkge1xuICAgIHRocm93IG5ldyBWYWxpZGF0aW9uRXJyb3IoJ3pvbmUgbmFtZSBtdXN0IG5vdCBlbmQgd2l0aCBhIHRyYWlsaW5nIGRvdCcpO1xuICB9XG4gIGlmICh6b25lTmFtZS5sZW5ndGggPiAyNTUpIHtcbiAgICB0aHJvdyBuZXcgVmFsaWRhdGlvbkVycm9yKCd6b25lIG5hbWUgY2Fubm90IGJlIG1vcmUgdGhhbiAyNTUgYnl0ZXMgbG9uZycpO1xuICB9XG4gIGlmICh6b25lTmFtZS5zcGxpdCgnLicpLmZpbmQobGFiZWwgPT4gbGFiZWwubGVuZ3RoID4gNjMpKSB7XG4gICAgdGhyb3cgbmV3IFZhbGlkYXRpb25FcnJvcignem9uZSBuYW1lIGxhYmVscyBjYW5ub3QgYmUgbW9yZSB0aGFuIDYzIGJ5dGVzIGxvbmcnKTtcbiAgfVxuICBpZiAoIXpvbmVOYW1lLm1hdGNoKC9eW2EtejAtOSFcIiMkJSYnKCkqKywvOjs8PT4/QFtcXFxcXFxdXl9ge3x9fi4tXSskL2kpKSB7XG4gICAgdGhyb3cgbmV3IFZhbGlkYXRpb25FcnJvcignem9uZSBuYW1lcyBjYW4gb25seSBjb250YWluIGEteiwgMC05LCAtLCAhIFwiICMgJCAlICYgXFwnICggKSAqICsgLCAtIC8gOiA7IDwgPSA+ID8gQCBbIFxcIF0gXiBfIGAgeyB8IH0gfiAuJyk7XG4gIH1cbn1cblxuY2xhc3MgVmFsaWRhdGlvbkVycm9yIGV4dGVuZHMgRXJyb3Ige1xuICBjb25zdHJ1Y3RvcihtZXNzYWdlOiBzdHJpbmcpIHtcbiAgICBzdXBlcihtZXNzYWdlKTtcbiAgfVxufVxuXG4vKipcbiAqIFJvdXRlNTMgcmVxdWlyZXMgdGhlIHJlY29yZCBuYW1lcyBhcmUgc3BlY2lmaWVkIGFzIGZ1bGx5IHF1YWxpZmllZCBuYW1lcywgYnV0IHRoaXNcbiAqIGZvcmNlcyBsb3RzIG9mIHJlZHVuZGFudCB3b3JrIG9uIHRoZSB1c2VyIChyZXBlYXRpbmcgdGhlIHpvbmUgbmFtZSBvdmVyIGFuZCBvdmVyKS5cbiAqIFRoaXMgZnVuY3Rpb24gYWxsb3dzIHRoZSB1c2VyIHRvIGJlIGxhemllciBhbmQgb2ZmZXJzIGEgbmljZXIgZXhwZXJpZW5jZSwgYnlcbiAqIHF1YWxpZnlpbmcgcmVsYXRpdmUgbmFtZXMgYXBwcm9wcmlhdGVseTpcbiAqXG4gKiBAcGFyYW0gcHJvdmlkZWROYW1lIHRoZSB1c2VyLXNwZWNpZmllZCBuYW1lIG9mIHRoZSByZWNvcmQuXG4gKiBAcGFyYW0gem9uZU5hbWUgICB0aGUgZnVsbHktcXVhbGlmaWVkIG5hbWUgb2YgdGhlIHpvbmUgdGhlIHJlY29yZCB3aWxsIGJlIGNyZWF0ZWQgaW4uXG4gKlxuICogQHJldHVybnMgPHVsPlxuICogICAgICAgIDxsaT5JZiArcHJvdmlkZWROYW1lKyBlbmRzIHdpdGggYSArLissIHVzZSBpdCBhcy1pczwvbGk+XG4gKiAgICAgICAgPGxpPklmICtwcm92aWRlZE5hbWUrIGVuZHMgd2l0aCBvciBlcXVhbHMgK3pvbmVOYW1lKywgYXBwZW5kIGEgdHJhaWxpbmcgKy4rPC9saT5cbiAqICAgICAgICA8bGk+T3RoZXJ3aXNlLCBhcHBlbmQgKy4rLCArem9uZU5hbWUrIGFuZCBhIHRyYWlsaW5nICsuKzwvbGk+XG4gKiAgICAgIDwvdWw+XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBkZXRlcm1pbmVGdWxseVF1YWxpZmllZERvbWFpbk5hbWUocHJvdmlkZWROYW1lOiBzdHJpbmcsIGhvc3RlZFpvbmU6IElIb3N0ZWRab25lKTogc3RyaW5nIHtcbiAgaWYgKHByb3ZpZGVkTmFtZS5lbmRzV2l0aCgnLicpKSB7XG4gICAgcmV0dXJuIHByb3ZpZGVkTmFtZTtcbiAgfVxuXG4gIGNvbnN0IGhvc3RlZFpvbmVOYW1lID0gIGhvc3RlZFpvbmUuem9uZU5hbWUuZW5kc1dpdGgoJy4nKVxuICAgID8gaG9zdGVkWm9uZS56b25lTmFtZS5zdWJzdHJpbmcoMCwgaG9zdGVkWm9uZS56b25lTmFtZS5sZW5ndGggLSAxKVxuICAgIDogaG9zdGVkWm9uZS56b25lTmFtZTtcblxuICBjb25zdCBzdWZmaXggPSBgLiR7aG9zdGVkWm9uZU5hbWV9YDtcbiAgaWYgKHByb3ZpZGVkTmFtZS5lbmRzV2l0aChzdWZmaXgpIHx8IHByb3ZpZGVkTmFtZSA9PT0gaG9zdGVkWm9uZU5hbWUpIHtcbiAgICByZXR1cm4gYCR7cHJvdmlkZWROYW1lfS5gO1xuICB9XG5cbiAgcmV0dXJuIGAke3Byb3ZpZGVkTmFtZX0ke3N1ZmZpeH0uYDtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIG1ha2VIb3N0ZWRab25lQXJuKGNvbnN0cnVjdDogQ29uc3RydWN0LCBob3N0ZWRab25lSWQ6IHN0cmluZyk6IHN0cmluZyB7XG4gIHJldHVybiBTdGFjay5vZihjb25zdHJ1Y3QpLmZvcm1hdEFybih7XG4gICAgYWNjb3VudDogJycsXG4gICAgcmVnaW9uOiAnJyxcbiAgICBzZXJ2aWNlOiAncm91dGU1MycsXG4gICAgcmVzb3VyY2U6ICdob3N0ZWR6b25lJyxcbiAgICByZXNvdXJjZU5hbWU6IGhvc3RlZFpvbmVJZFxuICB9KTtcbn1cbiJdfQ==