#pragma once

#include "pdu/io.h"
#include "pdu/query.h"

#include <boost/variant.hpp>
#include <gsl/gsl-lite.hpp>

#include <stack>

enum class Operation : uint8_t { Unary_Minus, Add, Subtract, Divide, Multiply };

inline void execute(Operation op, std::stack<double>& stack);

class ExpressionIterator : public iterator_facade<ExpressionIterator, Sample> {
public:
    ExpressionIterator(
            std::vector<boost::variant<Operation, CrossIndexSeries, double>>
                    ops);

    void increment();
    const Sample& dereference() const {
        return currentResult;
    }

    bool is_end() const {
        return finished;
    }

private:
    using SeriesRef = uint32_t;

    void add(Operation op);
    void add(const CrossIndexSeries& cis);
    void add(double constant);

    void evaluate();

    void evaluate_single(Operation op);
    void evaluate_single(SeriesRef op);
    void evaluate_single(double op);
    std::vector<boost::variant<Operation, SeriesRef, double>> operations;
    std::vector<CrossIndexSampleIterator> iterators;
    std::vector<double> previousValues;
    std::stack<double> stack;
    Sample currentResult;
    int64_t lastTimestamp = 0;
    bool finished = false;
};

class ResamplingIterator : public iterator_facade<ResamplingIterator, Sample> {
public:
    ResamplingIterator(ExpressionIterator itr,
                       std::chrono::milliseconds interval);

    void increment();
    const Sample& dereference() const {
        return newSample;
    }

    bool is_end() const {
        return itr == end(itr);
    }

private:
    ExpressionIterator itr;
    Sample newSample;
    uint64_t interval;
    int64_t nextTimestamp = std::numeric_limits<int64_t>::max();
};

/**
 * A series which has been generated by applying operations to one
 * or more actual Prometheus time series.
 *
 * Used to support basic maths (+ - / *) operations on time series.
 */
class Expression {
public:
    Expression(CrossIndexSeries cis);

    Expression(double constantValue);

    ResamplingIterator resample(std::chrono::milliseconds interval) const;

    auto begin() const {
        return ExpressionIterator(operations);
    }

    EndSentinel end() const {
        return {};
    }

    Expression unary_minus() const;

    Expression operator-=(const Expression& other);
    Expression operator+=(const Expression& other);
    Expression operator/=(const Expression& other);
    Expression operator*=(const Expression& other);

    void copy_operations_from(const Expression& other);

    std::vector<boost::variant<Operation, CrossIndexSeries, double>> operations;
};

Expression operator-(const Expression&);
Expression operator+(const Expression&);

Expression operator-(Expression, const Expression&);
Expression operator+(Expression, const Expression&);
Expression operator/(Expression, const Expression&);
Expression operator*(Expression, const Expression&);