![test](https://github.com/rabizao/oka/workflows/test/badge.svg)
[![codecov](https://codecov.io/gh/rabizao/oka/branch/main/graph/badge.svg)](https://codecov.io/gh/rabizao/oka)
<a href="https://pypi.org/project/oka">
<img src="https://img.shields.io/pypi/v/oka.svg?label=release&color=blue&style=flat-square" alt="pypi">
</a>
![Python version](https://img.shields.io/badge/python-3.8%20%7C%203.9-blue.svg)
[![license: GPL v3](https://img.shields.io/badge/License-GPLv3-blue.svg)](https://www.gnu.org/licenses/gpl-3.0)

[![DOI](https://zenodo.org/badge/DOI/10.5281/zenodo.5501845.svg)](https://doi.org/10.5281/zenodo.5501845)
[![arXiv](https://img.shields.io/badge/arXiv-2109.06028-b31b1b.svg?style=flat-square)](https://arxiv.org/abs/2109.06028)
[![User Manual](https://img.shields.io/badge/doc-user%20manual-a0a0a0.svg)](https://rabizao.github.io/oka)
[![API Documentation](https://img.shields.io/badge/doc-API%20%28auto%29-a0a0a0.svg)](https://rabizao.github.io/oka/api)

# oka - Client for OKA repository
[Latest version as a package](https://pypi.org/project/oka)

[Current code](https://github.com/rabizao/oka)

[User manual](https://rabizao.github.io/oka)

[API documentation](https://rabizao.github.io/oka/api)

## Overview
`oka` is a client for Oka repository.
It also provides utilities to process data.

## Installation
### ...as a standalone lib
```bash
# Set up a virtualenv. 
python3 -m venv venv
source venv/bin/activate

# Install from PyPI...
pip install --upgrade pip
pip install -U oka
pip install -U oka[full]  # use the flag 'full' for extra functionality (recommended)

# ...or, install from updated source code.
pip install git+https://github.com/rabizao/oka
```
    
### ...from source
```bash
sudo apt install python3.8-venv python3.8-dev python3.8-distutils # For Debian-like systems.
git clone https://github.com/rabizao/oka
cd oka
python3.8 -m venv venv
source venv/bin/activate
pip install -e .
```

## Usage






**Hello world**
<details>
<p>

```python3
from oka import Oka, generate_token, toy_df

# Create a pandas dataframe.
df = toy_df()
print(df.head())
"""
   attr1  attr2  class
0    5.1    6.4      0
1    1.1    2.5      1
2    6.1    3.6      0
3    1.1    3.5      1
4    3.1    2.5      0
"""
```

```python3

# Login.
token = generate_token("http://localhost:5000")
client = Oka(token, "http://localhost:5000")

# Store.
id = client.send(df)

# Store again.
id = client.send(df)
"""
Content already stored for id iJ_e4463c51904e9efb800533d25082af2a7bf77
"""

# Fetch.
df = client.get(id)

print(df.head())
"""
   attr1  attr2  class
0    5.1    6.4      0
1    1.1    2.5      1
2    6.1    3.6      0
3    1.1    3.5      1
4    3.1    2.5      0
"""
```

</p>
</details>









**DataFrame by hand**
<details>
<p>

```python3
import pandas as pd
from oka import Oka, generate_token

# Create a pandas dataframe.
df = pd.DataFrame(
    [[1, 2, "+"],
     [3, 4, "-"]],
    index=["row 1", "row 2"],
    columns=["col 1", "col 2", "class"],
)
print(df.head())
"""
       col 1  col 2 class
row 1      1      2     +
row 2      3      4     -
"""
```

```python3

# Login.
token = generate_token("http://localhost:5000")
client = Oka(token, "http://localhost:5000")

# Store.
id = client.send(df)

# Store again.
id = client.send(df)
"""
Content already stored for id f7_6b9deafec2562edde56bfdc573b336b55cb16
"""

# Fetch.
df = client.get(id)

print(df.head())
"""
       col 1  col 2 class
row 1      1      2     +
row 2      3      4     -
"""
```






**Machine Learning workflow**
<details>
<p>

```python3

from sklearn.ensemble import RandomForestClassifier as RF

from idict import let, idict
from idict.function.classification import fit, predict
from idict.function.evaluation import split

cache = {}
d = (
        idict.fromtoy()
        >> split
        >> let(fit, algorithm=RF, config={"n_estimators": 55}, Xin="Xtr", yin="ytr")
        >> let(predict, Xin="Xts")
        >> (lambda X: {"X2": X * X})
        >> [cache]
)
print(d.y)
"""
[0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1]
"""
```

```python3

d.show()
"""
{
    "X2": "→(↑ X)",
    "z": "→(↑ input Xin yout version Xts→(input config X y) model→(algorithm config Xin yin output version Xtr→(input config X y) ytr→(input config X y)))",
    "_history": "split----------------------sklearn-1.0.1 fit--------------------------------idict predict----------------------------idict",
    "model": "→(↑ algorithm config Xin yin output version Xtr→(input config X y) ytr→(input config X y))",
    "Xtr": "→(↑ input config X y)",
    "ytr": "→(↑ input config X y)",
    "Xts": "→(↑ input config X y)",
    "yts": "→(↑ input config X y)",
    "X": "«{'attr1': {0: 5.1, 1: 1.1, 2: 6.1, 3: 1.1, 4: 3.1, 5: 4.7, 6: 9.1, 7: 8.3, 8: 9.1, 9: 2.5, 10: 7.1, 11: 0.1, 12: 2.1, 13: 0.1, 14: 5.1, 15: 31.1, 16: 1.1, 17: 2.2, 18: 3.1, 19: 1.1}, 'attr2': {0: 6.4, 1: 2.5, 2: 3.6, 3: 3.5, 4: 2.5, 5: 4.9, 6: 3.5, 7: 2.9, 8: 7.2, 9: 4.5, 10: 6.6, 11: 4.3, 12: 0.1, 13: 4.0, 14: 4.5, 15: 4.7, 16: 3.2, 17: 8.5, 18: 2.5, 19: 8.5}}»",
    "y": "«[0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1]»",
    "_id": "[38;5;121m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mX[0m[38;5;187m[1m[48;5;0mB[0m[38;5;119m[1m[48;5;0mj[0m[38;5;151m[1m[48;5;0mw[0m[38;5;120m[1m[48;5;0m4[0m[38;5;192m[1m[48;5;0m5[0m[38;5;155m[1m[48;5;0my[0m[38;5;155m[1m[48;5;0mu[0m[38;5;114m[1m[48;5;0mj[0m[38;5;120m[1m[48;5;0mV[0m[38;5;155m[1m[48;5;0mO[0m[38;5;150m[1m[48;5;0m0[0m[38;5;84m[1m[48;5;0mb[0m[38;5;155m[1m[48;5;0mU[0m[38;5;150m[1m[48;5;0m0[0m[38;5;121m[1m[48;5;0mc[0m[38;5;155m[1m[48;5;0mf[0m[38;5;187m[1m[48;5;0mk[0m[38;5;119m[1m[48;5;0mL[0m[38;5;151m[1m[48;5;0mB[0m[38;5;120m[1m[48;5;0me[0m[38;5;192m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mx[0m[38;5;155m[1m[48;5;0mY[0m[38;5;114m[1m[48;5;0mW[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mR[0m[38;5;150m[1m[48;5;0mk[0m[38;5;84m[1m[48;5;0ml[0m[38;5;155m[1m[48;5;0m.[0m[38;5;150m[1m[48;5;0mi[0m[38;5;121m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mz[0m[38;5;187m[1m[48;5;0m4[0m[38;5;119m[1m[48;5;0mS[0m[38;5;151m[1m[48;5;0mO[0m[38;5;120m[1m[48;5;0mz[0m[38;5;192m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mE[0m",
    "_ids": {
        "X2": "[38;5;209m[1m[48;5;0mM[0m[38;5;203m[1m[48;5;0mS[0m[38;5;167m[1m[48;5;0mw[0m[38;5;167m[1m[48;5;0mB[0m[38;5;167m[1m[48;5;0mJ[0m[38;5;167m[1m[48;5;0mq[0m[38;5;168m[1m[48;5;0my[0m[38;5;210m[1m[48;5;0m7[0m[38;5;203m[1m[48;5;0m.[0m[38;5;203m[1m[48;5;0m9[0m[38;5;167m[1m[48;5;0mI[0m[38;5;167m[1m[48;5;0mH[0m[38;5;167m[1m[48;5;0mZ[0m[38;5;167m[1m[48;5;0mK[0m[38;5;168m[1m[48;5;0mq[0m[38;5;169m[1m[48;5;0mB[0m[38;5;209m[1m[48;5;0mN[0m[38;5;203m[1m[48;5;0mi[0m[38;5;167m[1m[48;5;0my[0m[38;5;167m[1m[48;5;0mp[0m[38;5;167m[1m[48;5;0mn[0m[38;5;167m[1m[48;5;0mk[0m[38;5;168m[1m[48;5;0mR[0m[38;5;210m[1m[48;5;0m-[0m[38;5;203m[1m[48;5;0mj[0m[38;5;203m[1m[48;5;0mk[0m[38;5;167m[1m[48;5;0mB[0m[38;5;167m[1m[48;5;0m1[0m[38;5;167m[1m[48;5;0mV[0m[38;5;167m[1m[48;5;0m3[0m[38;5;168m[1m[48;5;0mR[0m[38;5;169m[1m[48;5;0mt[0m[38;5;209m[1m[48;5;0mH[0m[38;5;203m[1m[48;5;0mu[0m[38;5;167m[1m[48;5;0mR[0m[38;5;167m[1m[48;5;0mQ[0m[38;5;167m[1m[48;5;0my[0m[38;5;167m[1m[48;5;0m0[0m[38;5;168m[1m[48;5;0mI[0m[38;5;210m[1m[48;5;0md[0m",
        "z": "[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mJ[0m[38;5;63m[1m[48;5;0mg[0m[38;5;135m[1m[48;5;0mj[0m[38;5;135m[1m[48;5;0mu[0m[38;5;134m[1m[48;5;0mS[0m[38;5;135m[1m[48;5;0mW[0m[38;5;97m[1m[48;5;0m2[0m[38;5;134m[1m[48;5;0mN[0m[38;5;62m[1m[48;5;0mD[0m[38;5;99m[1m[48;5;0mg[0m[38;5;99m[1m[48;5;0mV[0m[38;5;141m[1m[48;5;0mJ[0m[38;5;133m[1m[48;5;0mh[0m[38;5;133m[1m[48;5;0mA[0m[38;5;62m[1m[48;5;0mX[0m[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mu[0m[38;5;63m[1m[48;5;0mn[0m[38;5;135m[1m[48;5;0mM[0m[38;5;135m[1m[48;5;0mj[0m[38;5;134m[1m[48;5;0mC[0m[38;5;135m[1m[48;5;0mF[0m[38;5;97m[1m[48;5;0mq[0m[38;5;134m[1m[48;5;0m-[0m[38;5;62m[1m[48;5;0m-[0m[38;5;99m[1m[48;5;0m5[0m[38;5;99m[1m[48;5;0mT[0m[38;5;141m[1m[48;5;0mo[0m[38;5;133m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0md[0m[38;5;62m[1m[48;5;0mt[0m[38;5;61m[1m[48;5;0mR[0m[38;5;99m[1m[48;5;0m9[0m[38;5;63m[1m[48;5;0mz[0m[38;5;135m[1m[48;5;0mF[0m[38;5;135m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mw[0m[38;5;135m[1m[48;5;0mA[0m[38;5;97m[1m[48;5;0mU[0m",
        "_history": "[38;5;87m[1m[48;5;0mo[0m[38;5;123m[1m[48;5;0mf[0m[38;5;159m[1m[48;5;0mE[0m[38;5;159m[1m[48;5;0mb[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0mn[0m[38;5;122m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mS[0m[38;5;122m[1m[48;5;0mY[0m[38;5;153m[1m[48;5;0ms[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0ms[0m[38;5;158m[1m[48;5;0mg[0m[38;5;117m[1m[48;5;0mA[0m[38;5;121m[1m[48;5;0mn[0m[38;5;116m[1m[48;5;0mn[0m[38;5;87m[1m[48;5;0my[0m[38;5;123m[1m[48;5;0mp[0m[38;5;159m[1m[48;5;0m4[0m[38;5;159m[1m[48;5;0mK[0m[38;5;159m[1m[48;5;0mY[0m[38;5;159m[1m[48;5;0mF[0m[38;5;122m[1m[48;5;0mo[0m[38;5;117m[1m[48;5;0mv[0m[38;5;122m[1m[48;5;0mZ[0m[38;5;153m[1m[48;5;0ma[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0mO[0m[38;5;158m[1m[48;5;0mV[0m[38;5;117m[1m[48;5;0m6[0m[38;5;121m[1m[48;5;0m0[0m[38;5;116m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m0[0m[38;5;123m[1m[48;5;0ms[0m[38;5;159m[1m[48;5;0mv[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;122m[1m[48;5;0m.[0m[38;5;117m[1m[48;5;0m-[0m",
        "model": "[38;5;97m[1m[48;5;0mB[0m[38;5;134m[1m[48;5;0mb[0m[38;5;105m[1m[48;5;0ms[0m[38;5;138m[1m[48;5;0mA[0m[38;5;169m[1m[48;5;0m4[0m[38;5;97m[1m[48;5;0mN[0m[38;5;97m[1m[48;5;0mV[0m[38;5;97m[1m[48;5;0m-[0m[38;5;96m[1m[48;5;0mO[0m[38;5;98m[1m[48;5;0m2[0m[38;5;103m[1m[48;5;0mx[0m[38;5;132m[1m[48;5;0mh[0m[38;5;98m[1m[48;5;0ml[0m[38;5;98m[1m[48;5;0mY[0m[38;5;139m[1m[48;5;0mO[0m[38;5;134m[1m[48;5;0m0[0m[38;5;97m[1m[48;5;0mo[0m[38;5;134m[1m[48;5;0mc[0m[38;5;105m[1m[48;5;0mB[0m[38;5;138m[1m[48;5;0mO[0m[38;5;169m[1m[48;5;0mM[0m[38;5;97m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0mQ[0m[38;5;96m[1m[48;5;0mU[0m[38;5;98m[1m[48;5;0ms[0m[38;5;103m[1m[48;5;0m7[0m[38;5;132m[1m[48;5;0m8[0m[38;5;98m[1m[48;5;0mA[0m[38;5;98m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mL[0m[38;5;97m[1m[48;5;0m9[0m[38;5;134m[1m[48;5;0m9[0m[38;5;105m[1m[48;5;0mi[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m9[0m[38;5;97m[1m[48;5;0mh[0m",
        "Xtr": "[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mE[0m[38;5;81m[1m[48;5;0m3[0m[38;5;117m[1m[48;5;0mW[0m[38;5;110m[1m[48;5;0mM[0m[38;5;74m[1m[48;5;0m8[0m[38;5;75m[1m[48;5;0mc[0m[38;5;81m[1m[48;5;0mu[0m[38;5;111m[1m[48;5;0m5[0m[38;5;87m[1m[48;5;0me[0m[38;5;123m[1m[48;5;0m1[0m[38;5;117m[1m[48;5;0mX[0m[38;5;75m[1m[48;5;0mr[0m[38;5;80m[1m[48;5;0mv[0m[38;5;74m[1m[48;5;0mm[0m[38;5;74m[1m[48;5;0mP[0m[38;5;69m[1m[48;5;0md[0m[38;5;81m[1m[48;5;0mK[0m[38;5;81m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mV[0m[38;5;110m[1m[48;5;0mr[0m[38;5;74m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mD[0m[38;5;81m[1m[48;5;0mJ[0m[38;5;111m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m-[0m[38;5;123m[1m[48;5;0mL[0m[38;5;117m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mm[0m[38;5;80m[1m[48;5;0ml[0m[38;5;74m[1m[48;5;0me[0m[38;5;74m[1m[48;5;0ma[0m[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mj[0m[38;5;81m[1m[48;5;0m-[0m[38;5;117m[1m[48;5;0m1[0m[38;5;110m[1m[48;5;0m.[0m[38;5;74m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0m.[0m[38;5;81m[1m[48;5;0m2[0m",
        "ytr": "[38;5;240m[1m[48;5;0mb[0m[38;5;96m[1m[48;5;0mu[0m[38;5;61m[1m[48;5;0mT[0m[38;5;103m[1m[48;5;0mG[0m[38;5;133m[1m[48;5;0mO[0m[38;5;132m[1m[48;5;0mR[0m[38;5;132m[1m[48;5;0mt[0m[38;5;97m[1m[48;5;0m9[0m[38;5;133m[1m[48;5;0mM[0m[38;5;132m[1m[48;5;0mC[0m[38;5;132m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mq[0m[38;5;131m[1m[48;5;0mU[0m[38;5;95m[1m[48;5;0mm[0m[38;5;60m[1m[48;5;0mt[0m[38;5;240m[1m[48;5;0mR[0m[38;5;96m[1m[48;5;0mA[0m[38;5;61m[1m[48;5;0ma[0m[38;5;103m[1m[48;5;0mC[0m[38;5;133m[1m[48;5;0mU[0m[38;5;132m[1m[48;5;0mY[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0md[0m[38;5;133m[1m[48;5;0mo[0m[38;5;132m[1m[48;5;0mt[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mk[0m[38;5;131m[1m[48;5;0ml[0m[38;5;95m[1m[48;5;0me[0m[38;5;60m[1m[48;5;0ma[0m[38;5;240m[1m[48;5;0mr[0m[38;5;96m[1m[48;5;0mn[0m[38;5;61m[1m[48;5;0m-[0m[38;5;103m[1m[48;5;0m1[0m[38;5;133m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m3[0m",
        "Xts": "[38;5;138m[1m[48;5;0m6[0m[38;5;169m[1m[48;5;0mZ[0m[38;5;140m[1m[48;5;0m4[0m[38;5;180m[1m[48;5;0m2[0m[38;5;169m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0mr[0m[38;5;133m[1m[48;5;0my[0m[38;5;103m[1m[48;5;0ml[0m[38;5;175m[1m[48;5;0mB[0m[38;5;139m[1m[48;5;0my[0m[38;5;140m[1m[48;5;0m0[0m[38;5;180m[1m[48;5;0mc[0m[38;5;205m[1m[48;5;0mJ[0m[38;5;97m[1m[48;5;0mV[0m[38;5;138m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0mt[0m[38;5;140m[1m[48;5;0mv[0m[38;5;180m[1m[48;5;0mi[0m[38;5;169m[1m[48;5;0m7[0m[38;5;139m[1m[48;5;0mn[0m[38;5;169m[1m[48;5;0ma[0m[38;5;132m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0mL[0m[38;5;103m[1m[48;5;0mY[0m[38;5;175m[1m[48;5;0mO[0m[38;5;139m[1m[48;5;0ms[0m[38;5;140m[1m[48;5;0mk[0m[38;5;180m[1m[48;5;0ml[0m[38;5;205m[1m[48;5;0me[0m[38;5;97m[1m[48;5;0ma[0m[38;5;138m[1m[48;5;0mr[0m[38;5;169m[1m[48;5;0mn[0m[38;5;140m[1m[48;5;0m-[0m[38;5;180m[1m[48;5;0m1[0m[38;5;169m[1m[48;5;0m.[0m[38;5;139m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m4[0m",
        "yts": "[38;5;77m[1m[48;5;0me[0m[38;5;119m[1m[48;5;0mw[0m[38;5;143m[1m[48;5;0mH[0m[38;5;83m[1m[48;5;0mi[0m[38;5;155m[1m[48;5;0m4[0m[38;5;143m[1m[48;5;0m1[0m[38;5;83m[1m[48;5;0m2[0m[38;5;149m[1m[48;5;0mm[0m[38;5;107m[1m[48;5;0m.[0m[38;5;71m[1m[48;5;0m1[0m[38;5;78m[1m[48;5;0mH[0m[38;5;120m[1m[48;5;0m0[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0mk[0m[38;5;107m[1m[48;5;0mM[0m[38;5;71m[1m[48;5;0mK[0m[38;5;77m[1m[48;5;0md[0m[38;5;119m[1m[48;5;0ml[0m[38;5;143m[1m[48;5;0mQ[0m[38;5;83m[1m[48;5;0m-[0m[38;5;155m[1m[48;5;0mt[0m[38;5;143m[1m[48;5;0mC[0m[38;5;83m[1m[48;5;0mi[0m[38;5;149m[1m[48;5;0mb[0m[38;5;107m[1m[48;5;0m7[0m[38;5;71m[1m[48;5;0ms[0m[38;5;78m[1m[48;5;0m4[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mk[0m[38;5;143m[1m[48;5;0ml[0m[38;5;107m[1m[48;5;0me[0m[38;5;71m[1m[48;5;0ma[0m[38;5;77m[1m[48;5;0mr[0m[38;5;119m[1m[48;5;0mn[0m[38;5;143m[1m[48;5;0m-[0m[38;5;83m[1m[48;5;0m1[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0m0[0m[38;5;83m[1m[48;5;0m.[0m[38;5;149m[1m[48;5;0m5[0m",
        "X": "[38;5;97m[1m[48;5;0m-[0m[38;5;98m[1m[48;5;0m2[0m[38;5;104m[1m[48;5;0m_[0m[38;5;135m[1m[48;5;0m3[0m[38;5;248m[1m[48;5;0m6[0m[38;5;168m[1m[48;5;0m0[0m[38;5;98m[1m[48;5;0m6[0m[38;5;68m[1m[48;5;0m0[0m[38;5;139m[1m[48;5;0m6[0m[38;5;134m[1m[48;5;0m0[0m[38;5;105m[1m[48;5;0mc[0m[38;5;138m[1m[48;5;0m7[0m[38;5;169m[1m[48;5;0m7[0m[38;5;60m[1m[48;5;0md[0m[38;5;97m[1m[48;5;0mc[0m[38;5;61m[1m[48;5;0m8[0m[38;5;97m[1m[48;5;0mf[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m8[0m[38;5;135m[1m[48;5;0m0[0m[38;5;248m[1m[48;5;0m7[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0me[0m[38;5;68m[1m[48;5;0me[0m[38;5;139m[1m[48;5;0mc[0m[38;5;134m[1m[48;5;0m6[0m[38;5;105m[1m[48;5;0m6[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0ma[0m[38;5;60m[1m[48;5;0mc[0m[38;5;97m[1m[48;5;0m5[0m[38;5;61m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0m2[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m5[0m[38;5;135m[1m[48;5;0m7[0m[38;5;248m[1m[48;5;0m8[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0me[0m (content: 34_1738c83af436029507def2710bc5125f58d0e)",
        "y": "[38;5;99m[1m[48;5;0mo[0m[38;5;69m[1m[48;5;0m.[0m[38;5;81m[1m[48;5;0m_[0m[38;5;111m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0m8[0m[38;5;69m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0mb[0m[38;5;63m[1m[48;5;0mf[0m[38;5;75m[1m[48;5;0m2[0m[38;5;68m[1m[48;5;0ma[0m[38;5;63m[1m[48;5;0m3[0m[38;5;69m[1m[48;5;0m5[0m[38;5;68m[1m[48;5;0md[0m[38;5;63m[1m[48;5;0m8[0m[38;5;75m[1m[48;5;0ma[0m[38;5;74m[1m[48;5;0mb[0m[38;5;99m[1m[48;5;0m2[0m[38;5;69m[1m[48;5;0m7[0m[38;5;81m[1m[48;5;0m5[0m[38;5;111m[1m[48;5;0md[0m[38;5;75m[1m[48;5;0m4[0m[38;5;69m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0m5[0m[38;5;63m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0md[0m[38;5;68m[1m[48;5;0mf[0m[38;5;63m[1m[48;5;0md[0m[38;5;69m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0m2[0m[38;5;63m[1m[48;5;0mb[0m[38;5;75m[1m[48;5;0m9[0m[38;5;74m[1m[48;5;0m3[0m[38;5;99m[1m[48;5;0m9[0m[38;5;69m[1m[48;5;0m1[0m[38;5;81m[1m[48;5;0m0[0m[38;5;111m[1m[48;5;0m4[0m[38;5;75m[1m[48;5;0mf[0m[38;5;69m[1m[48;5;0me[0m[38;5;68m[1m[48;5;0ma[0m[38;5;63m[1m[48;5;0mc[0m (content: S0_b6360d62ccafa275d4051dfd02b939104feac)"
    }
}
"""
```

```python3

print(d.z)
"""
[1 0 1 0 1 1 1]
"""
```

```python3

d.show()
"""
{
    "X2": "→(↑ X)",
    "z": "«[1 0 1 0 1 1 1]»",
    "_history": "split----------------------sklearn-1.0.1 fit--------------------------------idict predict----------------------------idict",
    "model": "RandomForestClassifier(n_estimators=55)",
    "Xtr": "→(↑ input config X y)",
    "ytr": "→(↑ input config X y)",
    "Xts": "«{'attr1': {13: 0.1, 6: 9.1, 9: 2.5, 10: 7.1, 0: 5.1, 14: 5.1, 15: 31.1}, 'attr2': {13: 4.0, 6: 3.5, 9: 4.5, 10: 6.6, 0: 6.4, 14: 4.5, 15: 4.7}}»",
    "yts": "→(↑ input config X y)",
    "X": "«{'attr1': {0: 5.1, 1: 1.1, 2: 6.1, 3: 1.1, 4: 3.1, 5: 4.7, 6: 9.1, 7: 8.3, 8: 9.1, 9: 2.5, 10: 7.1, 11: 0.1, 12: 2.1, 13: 0.1, 14: 5.1, 15: 31.1, 16: 1.1, 17: 2.2, 18: 3.1, 19: 1.1}, 'attr2': {0: 6.4, 1: 2.5, 2: 3.6, 3: 3.5, 4: 2.5, 5: 4.9, 6: 3.5, 7: 2.9, 8: 7.2, 9: 4.5, 10: 6.6, 11: 4.3, 12: 0.1, 13: 4.0, 14: 4.5, 15: 4.7, 16: 3.2, 17: 8.5, 18: 2.5, 19: 8.5}}»",
    "y": "«[0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1]»",
    "_id": "[38;5;121m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mX[0m[38;5;187m[1m[48;5;0mB[0m[38;5;119m[1m[48;5;0mj[0m[38;5;151m[1m[48;5;0mw[0m[38;5;120m[1m[48;5;0m4[0m[38;5;192m[1m[48;5;0m5[0m[38;5;155m[1m[48;5;0my[0m[38;5;155m[1m[48;5;0mu[0m[38;5;114m[1m[48;5;0mj[0m[38;5;120m[1m[48;5;0mV[0m[38;5;155m[1m[48;5;0mO[0m[38;5;150m[1m[48;5;0m0[0m[38;5;84m[1m[48;5;0mb[0m[38;5;155m[1m[48;5;0mU[0m[38;5;150m[1m[48;5;0m0[0m[38;5;121m[1m[48;5;0mc[0m[38;5;155m[1m[48;5;0mf[0m[38;5;187m[1m[48;5;0mk[0m[38;5;119m[1m[48;5;0mL[0m[38;5;151m[1m[48;5;0mB[0m[38;5;120m[1m[48;5;0me[0m[38;5;192m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mx[0m[38;5;155m[1m[48;5;0mY[0m[38;5;114m[1m[48;5;0mW[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mR[0m[38;5;150m[1m[48;5;0mk[0m[38;5;84m[1m[48;5;0ml[0m[38;5;155m[1m[48;5;0m.[0m[38;5;150m[1m[48;5;0mi[0m[38;5;121m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mz[0m[38;5;187m[1m[48;5;0m4[0m[38;5;119m[1m[48;5;0mS[0m[38;5;151m[1m[48;5;0mO[0m[38;5;120m[1m[48;5;0mz[0m[38;5;192m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mE[0m",
    "_ids": {
        "X2": "[38;5;209m[1m[48;5;0mM[0m[38;5;203m[1m[48;5;0mS[0m[38;5;167m[1m[48;5;0mw[0m[38;5;167m[1m[48;5;0mB[0m[38;5;167m[1m[48;5;0mJ[0m[38;5;167m[1m[48;5;0mq[0m[38;5;168m[1m[48;5;0my[0m[38;5;210m[1m[48;5;0m7[0m[38;5;203m[1m[48;5;0m.[0m[38;5;203m[1m[48;5;0m9[0m[38;5;167m[1m[48;5;0mI[0m[38;5;167m[1m[48;5;0mH[0m[38;5;167m[1m[48;5;0mZ[0m[38;5;167m[1m[48;5;0mK[0m[38;5;168m[1m[48;5;0mq[0m[38;5;169m[1m[48;5;0mB[0m[38;5;209m[1m[48;5;0mN[0m[38;5;203m[1m[48;5;0mi[0m[38;5;167m[1m[48;5;0my[0m[38;5;167m[1m[48;5;0mp[0m[38;5;167m[1m[48;5;0mn[0m[38;5;167m[1m[48;5;0mk[0m[38;5;168m[1m[48;5;0mR[0m[38;5;210m[1m[48;5;0m-[0m[38;5;203m[1m[48;5;0mj[0m[38;5;203m[1m[48;5;0mk[0m[38;5;167m[1m[48;5;0mB[0m[38;5;167m[1m[48;5;0m1[0m[38;5;167m[1m[48;5;0mV[0m[38;5;167m[1m[48;5;0m3[0m[38;5;168m[1m[48;5;0mR[0m[38;5;169m[1m[48;5;0mt[0m[38;5;209m[1m[48;5;0mH[0m[38;5;203m[1m[48;5;0mu[0m[38;5;167m[1m[48;5;0mR[0m[38;5;167m[1m[48;5;0mQ[0m[38;5;167m[1m[48;5;0my[0m[38;5;167m[1m[48;5;0m0[0m[38;5;168m[1m[48;5;0mI[0m[38;5;210m[1m[48;5;0md[0m",
        "z": "[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mJ[0m[38;5;63m[1m[48;5;0mg[0m[38;5;135m[1m[48;5;0mj[0m[38;5;135m[1m[48;5;0mu[0m[38;5;134m[1m[48;5;0mS[0m[38;5;135m[1m[48;5;0mW[0m[38;5;97m[1m[48;5;0m2[0m[38;5;134m[1m[48;5;0mN[0m[38;5;62m[1m[48;5;0mD[0m[38;5;99m[1m[48;5;0mg[0m[38;5;99m[1m[48;5;0mV[0m[38;5;141m[1m[48;5;0mJ[0m[38;5;133m[1m[48;5;0mh[0m[38;5;133m[1m[48;5;0mA[0m[38;5;62m[1m[48;5;0mX[0m[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mu[0m[38;5;63m[1m[48;5;0mn[0m[38;5;135m[1m[48;5;0mM[0m[38;5;135m[1m[48;5;0mj[0m[38;5;134m[1m[48;5;0mC[0m[38;5;135m[1m[48;5;0mF[0m[38;5;97m[1m[48;5;0mq[0m[38;5;134m[1m[48;5;0m-[0m[38;5;62m[1m[48;5;0m-[0m[38;5;99m[1m[48;5;0m5[0m[38;5;99m[1m[48;5;0mT[0m[38;5;141m[1m[48;5;0mo[0m[38;5;133m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0md[0m[38;5;62m[1m[48;5;0mt[0m[38;5;61m[1m[48;5;0mR[0m[38;5;99m[1m[48;5;0m9[0m[38;5;63m[1m[48;5;0mz[0m[38;5;135m[1m[48;5;0mF[0m[38;5;135m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mw[0m[38;5;135m[1m[48;5;0mA[0m[38;5;97m[1m[48;5;0mU[0m",
        "_history": "[38;5;87m[1m[48;5;0mo[0m[38;5;123m[1m[48;5;0mf[0m[38;5;159m[1m[48;5;0mE[0m[38;5;159m[1m[48;5;0mb[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0mn[0m[38;5;122m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mS[0m[38;5;122m[1m[48;5;0mY[0m[38;5;153m[1m[48;5;0ms[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0ms[0m[38;5;158m[1m[48;5;0mg[0m[38;5;117m[1m[48;5;0mA[0m[38;5;121m[1m[48;5;0mn[0m[38;5;116m[1m[48;5;0mn[0m[38;5;87m[1m[48;5;0my[0m[38;5;123m[1m[48;5;0mp[0m[38;5;159m[1m[48;5;0m4[0m[38;5;159m[1m[48;5;0mK[0m[38;5;159m[1m[48;5;0mY[0m[38;5;159m[1m[48;5;0mF[0m[38;5;122m[1m[48;5;0mo[0m[38;5;117m[1m[48;5;0mv[0m[38;5;122m[1m[48;5;0mZ[0m[38;5;153m[1m[48;5;0ma[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0mO[0m[38;5;158m[1m[48;5;0mV[0m[38;5;117m[1m[48;5;0m6[0m[38;5;121m[1m[48;5;0m0[0m[38;5;116m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m0[0m[38;5;123m[1m[48;5;0ms[0m[38;5;159m[1m[48;5;0mv[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;122m[1m[48;5;0m.[0m[38;5;117m[1m[48;5;0m-[0m",
        "model": "[38;5;97m[1m[48;5;0mB[0m[38;5;134m[1m[48;5;0mb[0m[38;5;105m[1m[48;5;0ms[0m[38;5;138m[1m[48;5;0mA[0m[38;5;169m[1m[48;5;0m4[0m[38;5;97m[1m[48;5;0mN[0m[38;5;97m[1m[48;5;0mV[0m[38;5;97m[1m[48;5;0m-[0m[38;5;96m[1m[48;5;0mO[0m[38;5;98m[1m[48;5;0m2[0m[38;5;103m[1m[48;5;0mx[0m[38;5;132m[1m[48;5;0mh[0m[38;5;98m[1m[48;5;0ml[0m[38;5;98m[1m[48;5;0mY[0m[38;5;139m[1m[48;5;0mO[0m[38;5;134m[1m[48;5;0m0[0m[38;5;97m[1m[48;5;0mo[0m[38;5;134m[1m[48;5;0mc[0m[38;5;105m[1m[48;5;0mB[0m[38;5;138m[1m[48;5;0mO[0m[38;5;169m[1m[48;5;0mM[0m[38;5;97m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0mQ[0m[38;5;96m[1m[48;5;0mU[0m[38;5;98m[1m[48;5;0ms[0m[38;5;103m[1m[48;5;0m7[0m[38;5;132m[1m[48;5;0m8[0m[38;5;98m[1m[48;5;0mA[0m[38;5;98m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mL[0m[38;5;97m[1m[48;5;0m9[0m[38;5;134m[1m[48;5;0m9[0m[38;5;105m[1m[48;5;0mi[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m9[0m[38;5;97m[1m[48;5;0mh[0m",
        "Xtr": "[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mE[0m[38;5;81m[1m[48;5;0m3[0m[38;5;117m[1m[48;5;0mW[0m[38;5;110m[1m[48;5;0mM[0m[38;5;74m[1m[48;5;0m8[0m[38;5;75m[1m[48;5;0mc[0m[38;5;81m[1m[48;5;0mu[0m[38;5;111m[1m[48;5;0m5[0m[38;5;87m[1m[48;5;0me[0m[38;5;123m[1m[48;5;0m1[0m[38;5;117m[1m[48;5;0mX[0m[38;5;75m[1m[48;5;0mr[0m[38;5;80m[1m[48;5;0mv[0m[38;5;74m[1m[48;5;0mm[0m[38;5;74m[1m[48;5;0mP[0m[38;5;69m[1m[48;5;0md[0m[38;5;81m[1m[48;5;0mK[0m[38;5;81m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mV[0m[38;5;110m[1m[48;5;0mr[0m[38;5;74m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mD[0m[38;5;81m[1m[48;5;0mJ[0m[38;5;111m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m-[0m[38;5;123m[1m[48;5;0mL[0m[38;5;117m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mm[0m[38;5;80m[1m[48;5;0ml[0m[38;5;74m[1m[48;5;0me[0m[38;5;74m[1m[48;5;0ma[0m[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mj[0m[38;5;81m[1m[48;5;0m-[0m[38;5;117m[1m[48;5;0m1[0m[38;5;110m[1m[48;5;0m.[0m[38;5;74m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0m.[0m[38;5;81m[1m[48;5;0m2[0m",
        "ytr": "[38;5;240m[1m[48;5;0mb[0m[38;5;96m[1m[48;5;0mu[0m[38;5;61m[1m[48;5;0mT[0m[38;5;103m[1m[48;5;0mG[0m[38;5;133m[1m[48;5;0mO[0m[38;5;132m[1m[48;5;0mR[0m[38;5;132m[1m[48;5;0mt[0m[38;5;97m[1m[48;5;0m9[0m[38;5;133m[1m[48;5;0mM[0m[38;5;132m[1m[48;5;0mC[0m[38;5;132m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mq[0m[38;5;131m[1m[48;5;0mU[0m[38;5;95m[1m[48;5;0mm[0m[38;5;60m[1m[48;5;0mt[0m[38;5;240m[1m[48;5;0mR[0m[38;5;96m[1m[48;5;0mA[0m[38;5;61m[1m[48;5;0ma[0m[38;5;103m[1m[48;5;0mC[0m[38;5;133m[1m[48;5;0mU[0m[38;5;132m[1m[48;5;0mY[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0md[0m[38;5;133m[1m[48;5;0mo[0m[38;5;132m[1m[48;5;0mt[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mk[0m[38;5;131m[1m[48;5;0ml[0m[38;5;95m[1m[48;5;0me[0m[38;5;60m[1m[48;5;0ma[0m[38;5;240m[1m[48;5;0mr[0m[38;5;96m[1m[48;5;0mn[0m[38;5;61m[1m[48;5;0m-[0m[38;5;103m[1m[48;5;0m1[0m[38;5;133m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m3[0m",
        "Xts": "[38;5;138m[1m[48;5;0m6[0m[38;5;169m[1m[48;5;0mZ[0m[38;5;140m[1m[48;5;0m4[0m[38;5;180m[1m[48;5;0m2[0m[38;5;169m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0mr[0m[38;5;133m[1m[48;5;0my[0m[38;5;103m[1m[48;5;0ml[0m[38;5;175m[1m[48;5;0mB[0m[38;5;139m[1m[48;5;0my[0m[38;5;140m[1m[48;5;0m0[0m[38;5;180m[1m[48;5;0mc[0m[38;5;205m[1m[48;5;0mJ[0m[38;5;97m[1m[48;5;0mV[0m[38;5;138m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0mt[0m[38;5;140m[1m[48;5;0mv[0m[38;5;180m[1m[48;5;0mi[0m[38;5;169m[1m[48;5;0m7[0m[38;5;139m[1m[48;5;0mn[0m[38;5;169m[1m[48;5;0ma[0m[38;5;132m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0mL[0m[38;5;103m[1m[48;5;0mY[0m[38;5;175m[1m[48;5;0mO[0m[38;5;139m[1m[48;5;0ms[0m[38;5;140m[1m[48;5;0mk[0m[38;5;180m[1m[48;5;0ml[0m[38;5;205m[1m[48;5;0me[0m[38;5;97m[1m[48;5;0ma[0m[38;5;138m[1m[48;5;0mr[0m[38;5;169m[1m[48;5;0mn[0m[38;5;140m[1m[48;5;0m-[0m[38;5;180m[1m[48;5;0m1[0m[38;5;169m[1m[48;5;0m.[0m[38;5;139m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m4[0m",
        "yts": "[38;5;77m[1m[48;5;0me[0m[38;5;119m[1m[48;5;0mw[0m[38;5;143m[1m[48;5;0mH[0m[38;5;83m[1m[48;5;0mi[0m[38;5;155m[1m[48;5;0m4[0m[38;5;143m[1m[48;5;0m1[0m[38;5;83m[1m[48;5;0m2[0m[38;5;149m[1m[48;5;0mm[0m[38;5;107m[1m[48;5;0m.[0m[38;5;71m[1m[48;5;0m1[0m[38;5;78m[1m[48;5;0mH[0m[38;5;120m[1m[48;5;0m0[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0mk[0m[38;5;107m[1m[48;5;0mM[0m[38;5;71m[1m[48;5;0mK[0m[38;5;77m[1m[48;5;0md[0m[38;5;119m[1m[48;5;0ml[0m[38;5;143m[1m[48;5;0mQ[0m[38;5;83m[1m[48;5;0m-[0m[38;5;155m[1m[48;5;0mt[0m[38;5;143m[1m[48;5;0mC[0m[38;5;83m[1m[48;5;0mi[0m[38;5;149m[1m[48;5;0mb[0m[38;5;107m[1m[48;5;0m7[0m[38;5;71m[1m[48;5;0ms[0m[38;5;78m[1m[48;5;0m4[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mk[0m[38;5;143m[1m[48;5;0ml[0m[38;5;107m[1m[48;5;0me[0m[38;5;71m[1m[48;5;0ma[0m[38;5;77m[1m[48;5;0mr[0m[38;5;119m[1m[48;5;0mn[0m[38;5;143m[1m[48;5;0m-[0m[38;5;83m[1m[48;5;0m1[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0m0[0m[38;5;83m[1m[48;5;0m.[0m[38;5;149m[1m[48;5;0m5[0m",
        "X": "[38;5;97m[1m[48;5;0m-[0m[38;5;98m[1m[48;5;0m2[0m[38;5;104m[1m[48;5;0m_[0m[38;5;135m[1m[48;5;0m3[0m[38;5;248m[1m[48;5;0m6[0m[38;5;168m[1m[48;5;0m0[0m[38;5;98m[1m[48;5;0m6[0m[38;5;68m[1m[48;5;0m0[0m[38;5;139m[1m[48;5;0m6[0m[38;5;134m[1m[48;5;0m0[0m[38;5;105m[1m[48;5;0mc[0m[38;5;138m[1m[48;5;0m7[0m[38;5;169m[1m[48;5;0m7[0m[38;5;60m[1m[48;5;0md[0m[38;5;97m[1m[48;5;0mc[0m[38;5;61m[1m[48;5;0m8[0m[38;5;97m[1m[48;5;0mf[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m8[0m[38;5;135m[1m[48;5;0m0[0m[38;5;248m[1m[48;5;0m7[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0me[0m[38;5;68m[1m[48;5;0me[0m[38;5;139m[1m[48;5;0mc[0m[38;5;134m[1m[48;5;0m6[0m[38;5;105m[1m[48;5;0m6[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0ma[0m[38;5;60m[1m[48;5;0mc[0m[38;5;97m[1m[48;5;0m5[0m[38;5;61m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0m2[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m5[0m[38;5;135m[1m[48;5;0m7[0m[38;5;248m[1m[48;5;0m8[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0me[0m (content: 34_1738c83af436029507def2710bc5125f58d0e)",
        "y": "[38;5;99m[1m[48;5;0mo[0m[38;5;69m[1m[48;5;0m.[0m[38;5;81m[1m[48;5;0m_[0m[38;5;111m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0m8[0m[38;5;69m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0mb[0m[38;5;63m[1m[48;5;0mf[0m[38;5;75m[1m[48;5;0m2[0m[38;5;68m[1m[48;5;0ma[0m[38;5;63m[1m[48;5;0m3[0m[38;5;69m[1m[48;5;0m5[0m[38;5;68m[1m[48;5;0md[0m[38;5;63m[1m[48;5;0m8[0m[38;5;75m[1m[48;5;0ma[0m[38;5;74m[1m[48;5;0mb[0m[38;5;99m[1m[48;5;0m2[0m[38;5;69m[1m[48;5;0m7[0m[38;5;81m[1m[48;5;0m5[0m[38;5;111m[1m[48;5;0md[0m[38;5;75m[1m[48;5;0m4[0m[38;5;69m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0m5[0m[38;5;63m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0md[0m[38;5;68m[1m[48;5;0mf[0m[38;5;63m[1m[48;5;0md[0m[38;5;69m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0m2[0m[38;5;63m[1m[48;5;0mb[0m[38;5;75m[1m[48;5;0m9[0m[38;5;74m[1m[48;5;0m3[0m[38;5;99m[1m[48;5;0m9[0m[38;5;69m[1m[48;5;0m1[0m[38;5;81m[1m[48;5;0m0[0m[38;5;111m[1m[48;5;0m4[0m[38;5;75m[1m[48;5;0mf[0m[38;5;69m[1m[48;5;0me[0m[38;5;68m[1m[48;5;0ma[0m[38;5;63m[1m[48;5;0mc[0m (content: S0_b6360d62ccafa275d4051dfd02b939104feac)"
    }
}
"""
```

```python3

# A field '_' means this function is a noop process triggered only once by accessing one of the other provided fields."
d >>= (lambda _, X2, y: print("Some logging/printing that doesn't affect data... X²=\n", X2[:3]))
d.show()
"""
{
    "X2": "→(X2→(↑ X) y _)",
    "y": "→(X2→(↑ X) y _)",
    "z": "«[1 0 1 0 1 1 1]»",
    "_history": "split----------------------sklearn-1.0.1 fit--------------------------------idict predict----------------------------idict",
    "model": "RandomForestClassifier(n_estimators=55)",
    "Xtr": "→(↑ input config X y)",
    "ytr": "→(↑ input config X y)",
    "Xts": "«{'attr1': {13: 0.1, 6: 9.1, 9: 2.5, 10: 7.1, 0: 5.1, 14: 5.1, 15: 31.1}, 'attr2': {13: 4.0, 6: 3.5, 9: 4.5, 10: 6.6, 0: 6.4, 14: 4.5, 15: 4.7}}»",
    "yts": "→(↑ input config X y)",
    "X": "«{'attr1': {0: 5.1, 1: 1.1, 2: 6.1, 3: 1.1, 4: 3.1, 5: 4.7, 6: 9.1, 7: 8.3, 8: 9.1, 9: 2.5, 10: 7.1, 11: 0.1, 12: 2.1, 13: 0.1, 14: 5.1, 15: 31.1, 16: 1.1, 17: 2.2, 18: 3.1, 19: 1.1}, 'attr2': {0: 6.4, 1: 2.5, 2: 3.6, 3: 3.5, 4: 2.5, 5: 4.9, 6: 3.5, 7: 2.9, 8: 7.2, 9: 4.5, 10: 6.6, 11: 4.3, 12: 0.1, 13: 4.0, 14: 4.5, 15: 4.7, 16: 3.2, 17: 8.5, 18: 2.5, 19: 8.5}}»",
    "_id": "[38;5;121m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mX[0m[38;5;187m[1m[48;5;0mB[0m[38;5;119m[1m[48;5;0mj[0m[38;5;151m[1m[48;5;0mw[0m[38;5;120m[1m[48;5;0m4[0m[38;5;192m[1m[48;5;0m5[0m[38;5;155m[1m[48;5;0my[0m[38;5;155m[1m[48;5;0mu[0m[38;5;114m[1m[48;5;0mj[0m[38;5;120m[1m[48;5;0mV[0m[38;5;155m[1m[48;5;0mO[0m[38;5;150m[1m[48;5;0m0[0m[38;5;84m[1m[48;5;0mb[0m[38;5;155m[1m[48;5;0mU[0m[38;5;150m[1m[48;5;0m0[0m[38;5;121m[1m[48;5;0mc[0m[38;5;155m[1m[48;5;0mf[0m[38;5;187m[1m[48;5;0mk[0m[38;5;119m[1m[48;5;0mL[0m[38;5;151m[1m[48;5;0mB[0m[38;5;120m[1m[48;5;0me[0m[38;5;192m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mx[0m[38;5;155m[1m[48;5;0mY[0m[38;5;114m[1m[48;5;0mW[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mR[0m[38;5;150m[1m[48;5;0mk[0m[38;5;84m[1m[48;5;0ml[0m[38;5;155m[1m[48;5;0m.[0m[38;5;150m[1m[48;5;0mi[0m[38;5;121m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mz[0m[38;5;187m[1m[48;5;0m4[0m[38;5;119m[1m[48;5;0mS[0m[38;5;151m[1m[48;5;0mO[0m[38;5;120m[1m[48;5;0mz[0m[38;5;192m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mE[0m",
    "_ids": {
        "X2": "[38;5;104m[1m[48;5;0mz[0m[38;5;105m[1m[48;5;0mJ[0m[38;5;110m[1m[48;5;0mU[0m[38;5;141m[1m[48;5;0mw[0m[38;5;110m[1m[48;5;0mM[0m[38;5;140m[1m[48;5;0mP[0m[38;5;103m[1m[48;5;0mt[0m[38;5;97m[1m[48;5;0m5[0m[38;5;62m[1m[48;5;0mj[0m[38;5;73m[1m[48;5;0m6[0m[38;5;104m[1m[48;5;0mq[0m[38;5;110m[1m[48;5;0mH[0m[38;5;139m[1m[48;5;0mN[0m[38;5;97m[1m[48;5;0mp[0m[38;5;62m[1m[48;5;0mp[0m[38;5;67m[1m[48;5;0mv[0m[38;5;104m[1m[48;5;0mN[0m[38;5;105m[1m[48;5;0mW[0m[38;5;110m[1m[48;5;0md[0m[38;5;141m[1m[48;5;0mH[0m[38;5;110m[1m[48;5;0mr[0m[38;5;140m[1m[48;5;0mG[0m[38;5;103m[1m[48;5;0mS[0m[38;5;97m[1m[48;5;0mS[0m[38;5;62m[1m[48;5;0mC[0m[38;5;73m[1m[48;5;0mc[0m[38;5;104m[1m[48;5;0mx[0m[38;5;110m[1m[48;5;0mK[0m[38;5;139m[1m[48;5;0mW[0m[38;5;97m[1m[48;5;0m3[0m[38;5;62m[1m[48;5;0mR[0m[38;5;67m[1m[48;5;0mt[0m[38;5;104m[1m[48;5;0mH[0m[38;5;105m[1m[48;5;0mq[0m[38;5;110m[1m[48;5;0mR[0m[38;5;141m[1m[48;5;0mQ[0m[38;5;110m[1m[48;5;0my[0m[38;5;140m[1m[48;5;0m0[0m[38;5;103m[1m[48;5;0mI[0m[38;5;97m[1m[48;5;0me[0m",
        "y": "[38;5;87m[1m[48;5;0mo[0m[38;5;123m[1m[48;5;0mf[0m[38;5;159m[1m[48;5;0mE[0m[38;5;159m[1m[48;5;0mb[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0mn[0m[38;5;122m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mS[0m[38;5;122m[1m[48;5;0mY[0m[38;5;153m[1m[48;5;0ms[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0ms[0m[38;5;158m[1m[48;5;0mg[0m[38;5;117m[1m[48;5;0mA[0m[38;5;121m[1m[48;5;0mn[0m[38;5;116m[1m[48;5;0mn[0m[38;5;87m[1m[48;5;0my[0m[38;5;123m[1m[48;5;0mp[0m[38;5;159m[1m[48;5;0m4[0m[38;5;159m[1m[48;5;0mK[0m[38;5;159m[1m[48;5;0mY[0m[38;5;159m[1m[48;5;0mF[0m[38;5;122m[1m[48;5;0mo[0m[38;5;117m[1m[48;5;0mv[0m[38;5;122m[1m[48;5;0mZ[0m[38;5;153m[1m[48;5;0ma[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0mO[0m[38;5;158m[1m[48;5;0mV[0m[38;5;117m[1m[48;5;0m6[0m[38;5;121m[1m[48;5;0m0[0m[38;5;116m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m0[0m[38;5;123m[1m[48;5;0ms[0m[38;5;159m[1m[48;5;0mv[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;122m[1m[48;5;0m.[0m[38;5;117m[1m[48;5;0m-[0m",
        "z": "[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mJ[0m[38;5;63m[1m[48;5;0mg[0m[38;5;135m[1m[48;5;0mj[0m[38;5;135m[1m[48;5;0mu[0m[38;5;134m[1m[48;5;0mS[0m[38;5;135m[1m[48;5;0mW[0m[38;5;97m[1m[48;5;0m2[0m[38;5;134m[1m[48;5;0mN[0m[38;5;62m[1m[48;5;0mD[0m[38;5;99m[1m[48;5;0mg[0m[38;5;99m[1m[48;5;0mV[0m[38;5;141m[1m[48;5;0mJ[0m[38;5;133m[1m[48;5;0mh[0m[38;5;133m[1m[48;5;0mA[0m[38;5;62m[1m[48;5;0mX[0m[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mu[0m[38;5;63m[1m[48;5;0mn[0m[38;5;135m[1m[48;5;0mM[0m[38;5;135m[1m[48;5;0mj[0m[38;5;134m[1m[48;5;0mC[0m[38;5;135m[1m[48;5;0mF[0m[38;5;97m[1m[48;5;0mq[0m[38;5;134m[1m[48;5;0m-[0m[38;5;62m[1m[48;5;0m-[0m[38;5;99m[1m[48;5;0m5[0m[38;5;99m[1m[48;5;0mT[0m[38;5;141m[1m[48;5;0mo[0m[38;5;133m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0md[0m[38;5;62m[1m[48;5;0mt[0m[38;5;61m[1m[48;5;0mR[0m[38;5;99m[1m[48;5;0m9[0m[38;5;63m[1m[48;5;0mz[0m[38;5;135m[1m[48;5;0mF[0m[38;5;135m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mw[0m[38;5;135m[1m[48;5;0mA[0m[38;5;97m[1m[48;5;0mU[0m",
        "_history": "[38;5;87m[1m[48;5;0mo[0m[38;5;123m[1m[48;5;0mf[0m[38;5;159m[1m[48;5;0mE[0m[38;5;159m[1m[48;5;0mb[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0mn[0m[38;5;122m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mS[0m[38;5;122m[1m[48;5;0mY[0m[38;5;153m[1m[48;5;0ms[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0ms[0m[38;5;158m[1m[48;5;0mg[0m[38;5;117m[1m[48;5;0mA[0m[38;5;121m[1m[48;5;0mn[0m[38;5;116m[1m[48;5;0mn[0m[38;5;87m[1m[48;5;0my[0m[38;5;123m[1m[48;5;0mp[0m[38;5;159m[1m[48;5;0m4[0m[38;5;159m[1m[48;5;0mK[0m[38;5;159m[1m[48;5;0mY[0m[38;5;159m[1m[48;5;0mF[0m[38;5;122m[1m[48;5;0mo[0m[38;5;117m[1m[48;5;0mv[0m[38;5;122m[1m[48;5;0mZ[0m[38;5;153m[1m[48;5;0ma[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0mO[0m[38;5;158m[1m[48;5;0mV[0m[38;5;117m[1m[48;5;0m6[0m[38;5;121m[1m[48;5;0m0[0m[38;5;116m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m0[0m[38;5;123m[1m[48;5;0ms[0m[38;5;159m[1m[48;5;0mv[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;122m[1m[48;5;0m.[0m[38;5;117m[1m[48;5;0m-[0m",
        "model": "[38;5;97m[1m[48;5;0mB[0m[38;5;134m[1m[48;5;0mb[0m[38;5;105m[1m[48;5;0ms[0m[38;5;138m[1m[48;5;0mA[0m[38;5;169m[1m[48;5;0m4[0m[38;5;97m[1m[48;5;0mN[0m[38;5;97m[1m[48;5;0mV[0m[38;5;97m[1m[48;5;0m-[0m[38;5;96m[1m[48;5;0mO[0m[38;5;98m[1m[48;5;0m2[0m[38;5;103m[1m[48;5;0mx[0m[38;5;132m[1m[48;5;0mh[0m[38;5;98m[1m[48;5;0ml[0m[38;5;98m[1m[48;5;0mY[0m[38;5;139m[1m[48;5;0mO[0m[38;5;134m[1m[48;5;0m0[0m[38;5;97m[1m[48;5;0mo[0m[38;5;134m[1m[48;5;0mc[0m[38;5;105m[1m[48;5;0mB[0m[38;5;138m[1m[48;5;0mO[0m[38;5;169m[1m[48;5;0mM[0m[38;5;97m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0mQ[0m[38;5;96m[1m[48;5;0mU[0m[38;5;98m[1m[48;5;0ms[0m[38;5;103m[1m[48;5;0m7[0m[38;5;132m[1m[48;5;0m8[0m[38;5;98m[1m[48;5;0mA[0m[38;5;98m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mL[0m[38;5;97m[1m[48;5;0m9[0m[38;5;134m[1m[48;5;0m9[0m[38;5;105m[1m[48;5;0mi[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m9[0m[38;5;97m[1m[48;5;0mh[0m",
        "Xtr": "[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mE[0m[38;5;81m[1m[48;5;0m3[0m[38;5;117m[1m[48;5;0mW[0m[38;5;110m[1m[48;5;0mM[0m[38;5;74m[1m[48;5;0m8[0m[38;5;75m[1m[48;5;0mc[0m[38;5;81m[1m[48;5;0mu[0m[38;5;111m[1m[48;5;0m5[0m[38;5;87m[1m[48;5;0me[0m[38;5;123m[1m[48;5;0m1[0m[38;5;117m[1m[48;5;0mX[0m[38;5;75m[1m[48;5;0mr[0m[38;5;80m[1m[48;5;0mv[0m[38;5;74m[1m[48;5;0mm[0m[38;5;74m[1m[48;5;0mP[0m[38;5;69m[1m[48;5;0md[0m[38;5;81m[1m[48;5;0mK[0m[38;5;81m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mV[0m[38;5;110m[1m[48;5;0mr[0m[38;5;74m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mD[0m[38;5;81m[1m[48;5;0mJ[0m[38;5;111m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m-[0m[38;5;123m[1m[48;5;0mL[0m[38;5;117m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mm[0m[38;5;80m[1m[48;5;0ml[0m[38;5;74m[1m[48;5;0me[0m[38;5;74m[1m[48;5;0ma[0m[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mj[0m[38;5;81m[1m[48;5;0m-[0m[38;5;117m[1m[48;5;0m1[0m[38;5;110m[1m[48;5;0m.[0m[38;5;74m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0m.[0m[38;5;81m[1m[48;5;0m2[0m",
        "ytr": "[38;5;240m[1m[48;5;0mb[0m[38;5;96m[1m[48;5;0mu[0m[38;5;61m[1m[48;5;0mT[0m[38;5;103m[1m[48;5;0mG[0m[38;5;133m[1m[48;5;0mO[0m[38;5;132m[1m[48;5;0mR[0m[38;5;132m[1m[48;5;0mt[0m[38;5;97m[1m[48;5;0m9[0m[38;5;133m[1m[48;5;0mM[0m[38;5;132m[1m[48;5;0mC[0m[38;5;132m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mq[0m[38;5;131m[1m[48;5;0mU[0m[38;5;95m[1m[48;5;0mm[0m[38;5;60m[1m[48;5;0mt[0m[38;5;240m[1m[48;5;0mR[0m[38;5;96m[1m[48;5;0mA[0m[38;5;61m[1m[48;5;0ma[0m[38;5;103m[1m[48;5;0mC[0m[38;5;133m[1m[48;5;0mU[0m[38;5;132m[1m[48;5;0mY[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0md[0m[38;5;133m[1m[48;5;0mo[0m[38;5;132m[1m[48;5;0mt[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mk[0m[38;5;131m[1m[48;5;0ml[0m[38;5;95m[1m[48;5;0me[0m[38;5;60m[1m[48;5;0ma[0m[38;5;240m[1m[48;5;0mr[0m[38;5;96m[1m[48;5;0mn[0m[38;5;61m[1m[48;5;0m-[0m[38;5;103m[1m[48;5;0m1[0m[38;5;133m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m3[0m",
        "Xts": "[38;5;138m[1m[48;5;0m6[0m[38;5;169m[1m[48;5;0mZ[0m[38;5;140m[1m[48;5;0m4[0m[38;5;180m[1m[48;5;0m2[0m[38;5;169m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0mr[0m[38;5;133m[1m[48;5;0my[0m[38;5;103m[1m[48;5;0ml[0m[38;5;175m[1m[48;5;0mB[0m[38;5;139m[1m[48;5;0my[0m[38;5;140m[1m[48;5;0m0[0m[38;5;180m[1m[48;5;0mc[0m[38;5;205m[1m[48;5;0mJ[0m[38;5;97m[1m[48;5;0mV[0m[38;5;138m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0mt[0m[38;5;140m[1m[48;5;0mv[0m[38;5;180m[1m[48;5;0mi[0m[38;5;169m[1m[48;5;0m7[0m[38;5;139m[1m[48;5;0mn[0m[38;5;169m[1m[48;5;0ma[0m[38;5;132m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0mL[0m[38;5;103m[1m[48;5;0mY[0m[38;5;175m[1m[48;5;0mO[0m[38;5;139m[1m[48;5;0ms[0m[38;5;140m[1m[48;5;0mk[0m[38;5;180m[1m[48;5;0ml[0m[38;5;205m[1m[48;5;0me[0m[38;5;97m[1m[48;5;0ma[0m[38;5;138m[1m[48;5;0mr[0m[38;5;169m[1m[48;5;0mn[0m[38;5;140m[1m[48;5;0m-[0m[38;5;180m[1m[48;5;0m1[0m[38;5;169m[1m[48;5;0m.[0m[38;5;139m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m4[0m",
        "yts": "[38;5;77m[1m[48;5;0me[0m[38;5;119m[1m[48;5;0mw[0m[38;5;143m[1m[48;5;0mH[0m[38;5;83m[1m[48;5;0mi[0m[38;5;155m[1m[48;5;0m4[0m[38;5;143m[1m[48;5;0m1[0m[38;5;83m[1m[48;5;0m2[0m[38;5;149m[1m[48;5;0mm[0m[38;5;107m[1m[48;5;0m.[0m[38;5;71m[1m[48;5;0m1[0m[38;5;78m[1m[48;5;0mH[0m[38;5;120m[1m[48;5;0m0[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0mk[0m[38;5;107m[1m[48;5;0mM[0m[38;5;71m[1m[48;5;0mK[0m[38;5;77m[1m[48;5;0md[0m[38;5;119m[1m[48;5;0ml[0m[38;5;143m[1m[48;5;0mQ[0m[38;5;83m[1m[48;5;0m-[0m[38;5;155m[1m[48;5;0mt[0m[38;5;143m[1m[48;5;0mC[0m[38;5;83m[1m[48;5;0mi[0m[38;5;149m[1m[48;5;0mb[0m[38;5;107m[1m[48;5;0m7[0m[38;5;71m[1m[48;5;0ms[0m[38;5;78m[1m[48;5;0m4[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mk[0m[38;5;143m[1m[48;5;0ml[0m[38;5;107m[1m[48;5;0me[0m[38;5;71m[1m[48;5;0ma[0m[38;5;77m[1m[48;5;0mr[0m[38;5;119m[1m[48;5;0mn[0m[38;5;143m[1m[48;5;0m-[0m[38;5;83m[1m[48;5;0m1[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0m0[0m[38;5;83m[1m[48;5;0m.[0m[38;5;149m[1m[48;5;0m5[0m",
        "X": "[38;5;97m[1m[48;5;0m-[0m[38;5;98m[1m[48;5;0m2[0m[38;5;104m[1m[48;5;0m_[0m[38;5;135m[1m[48;5;0m3[0m[38;5;248m[1m[48;5;0m6[0m[38;5;168m[1m[48;5;0m0[0m[38;5;98m[1m[48;5;0m6[0m[38;5;68m[1m[48;5;0m0[0m[38;5;139m[1m[48;5;0m6[0m[38;5;134m[1m[48;5;0m0[0m[38;5;105m[1m[48;5;0mc[0m[38;5;138m[1m[48;5;0m7[0m[38;5;169m[1m[48;5;0m7[0m[38;5;60m[1m[48;5;0md[0m[38;5;97m[1m[48;5;0mc[0m[38;5;61m[1m[48;5;0m8[0m[38;5;97m[1m[48;5;0mf[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m8[0m[38;5;135m[1m[48;5;0m0[0m[38;5;248m[1m[48;5;0m7[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0me[0m[38;5;68m[1m[48;5;0me[0m[38;5;139m[1m[48;5;0mc[0m[38;5;134m[1m[48;5;0m6[0m[38;5;105m[1m[48;5;0m6[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0ma[0m[38;5;60m[1m[48;5;0mc[0m[38;5;97m[1m[48;5;0m5[0m[38;5;61m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0m2[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m5[0m[38;5;135m[1m[48;5;0m7[0m[38;5;248m[1m[48;5;0m8[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0me[0m (content: 34_1738c83af436029507def2710bc5125f58d0e)"
    }
}
"""
```

```python3

print("Triggering noop function by accessing 'y'...")
print("y", d.y[:3])
"""
Triggering noop function by accessing 'y'...
Some logging/printing that doesn't affect data... X²=
    attr1  attr2
0  26.01  40.96
1   1.21   6.25
2  37.21  12.96
y [0 1 0]
"""
```

```python3

d.show()
"""
{
    "X2": "     attr1  attr2\n0    26.01  40.96\n1     1.21   6.25\n2    37.21  12.96\n3     1.21  12.25\n4     9.61   6.25\n5    22.09  24.01\n6    82.81  12.25\n7    68.89   8.41\n8    82.81  51.84\n9     6.25  20.25\n10   50.41  43.56\n11    0.01  18.49\n12    4.41   0.01\n13    0.01  16.00\n14   26.01  20.25\n15  967.21  22.09\n16    1.21  10.24\n17    4.84  72.25\n18    9.61   6.25\n19    1.21  72.25",
    "y": "«[0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1 0 1]»",
    "z": "«[1 0 1 0 1 1 1]»",
    "_history": "split----------------------sklearn-1.0.1 fit--------------------------------idict predict----------------------------idict",
    "model": "RandomForestClassifier(n_estimators=55)",
    "Xtr": "→(↑ input config X y)",
    "ytr": "→(↑ input config X y)",
    "Xts": "«{'attr1': {13: 0.1, 6: 9.1, 9: 2.5, 10: 7.1, 0: 5.1, 14: 5.1, 15: 31.1}, 'attr2': {13: 4.0, 6: 3.5, 9: 4.5, 10: 6.6, 0: 6.4, 14: 4.5, 15: 4.7}}»",
    "yts": "→(↑ input config X y)",
    "X": "«{'attr1': {0: 5.1, 1: 1.1, 2: 6.1, 3: 1.1, 4: 3.1, 5: 4.7, 6: 9.1, 7: 8.3, 8: 9.1, 9: 2.5, 10: 7.1, 11: 0.1, 12: 2.1, 13: 0.1, 14: 5.1, 15: 31.1, 16: 1.1, 17: 2.2, 18: 3.1, 19: 1.1}, 'attr2': {0: 6.4, 1: 2.5, 2: 3.6, 3: 3.5, 4: 2.5, 5: 4.9, 6: 3.5, 7: 2.9, 8: 7.2, 9: 4.5, 10: 6.6, 11: 4.3, 12: 0.1, 13: 4.0, 14: 4.5, 15: 4.7, 16: 3.2, 17: 8.5, 18: 2.5, 19: 8.5}}»",
    "_id": "[38;5;121m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mX[0m[38;5;187m[1m[48;5;0mB[0m[38;5;119m[1m[48;5;0mj[0m[38;5;151m[1m[48;5;0mw[0m[38;5;120m[1m[48;5;0m4[0m[38;5;192m[1m[48;5;0m5[0m[38;5;155m[1m[48;5;0my[0m[38;5;155m[1m[48;5;0mu[0m[38;5;114m[1m[48;5;0mj[0m[38;5;120m[1m[48;5;0mV[0m[38;5;155m[1m[48;5;0mO[0m[38;5;150m[1m[48;5;0m0[0m[38;5;84m[1m[48;5;0mb[0m[38;5;155m[1m[48;5;0mU[0m[38;5;150m[1m[48;5;0m0[0m[38;5;121m[1m[48;5;0mc[0m[38;5;155m[1m[48;5;0mf[0m[38;5;187m[1m[48;5;0mk[0m[38;5;119m[1m[48;5;0mL[0m[38;5;151m[1m[48;5;0mB[0m[38;5;120m[1m[48;5;0me[0m[38;5;192m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mx[0m[38;5;155m[1m[48;5;0mY[0m[38;5;114m[1m[48;5;0mW[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mR[0m[38;5;150m[1m[48;5;0mk[0m[38;5;84m[1m[48;5;0ml[0m[38;5;155m[1m[48;5;0m.[0m[38;5;150m[1m[48;5;0mi[0m[38;5;121m[1m[48;5;0mn[0m[38;5;155m[1m[48;5;0mz[0m[38;5;187m[1m[48;5;0m4[0m[38;5;119m[1m[48;5;0mS[0m[38;5;151m[1m[48;5;0mO[0m[38;5;120m[1m[48;5;0mz[0m[38;5;192m[1m[48;5;0mm[0m[38;5;155m[1m[48;5;0mE[0m",
    "_ids": {
        "X2": "[38;5;104m[1m[48;5;0mz[0m[38;5;105m[1m[48;5;0mJ[0m[38;5;110m[1m[48;5;0mU[0m[38;5;141m[1m[48;5;0mw[0m[38;5;110m[1m[48;5;0mM[0m[38;5;140m[1m[48;5;0mP[0m[38;5;103m[1m[48;5;0mt[0m[38;5;97m[1m[48;5;0m5[0m[38;5;62m[1m[48;5;0mj[0m[38;5;73m[1m[48;5;0m6[0m[38;5;104m[1m[48;5;0mq[0m[38;5;110m[1m[48;5;0mH[0m[38;5;139m[1m[48;5;0mN[0m[38;5;97m[1m[48;5;0mp[0m[38;5;62m[1m[48;5;0mp[0m[38;5;67m[1m[48;5;0mv[0m[38;5;104m[1m[48;5;0mN[0m[38;5;105m[1m[48;5;0mW[0m[38;5;110m[1m[48;5;0md[0m[38;5;141m[1m[48;5;0mH[0m[38;5;110m[1m[48;5;0mr[0m[38;5;140m[1m[48;5;0mG[0m[38;5;103m[1m[48;5;0mS[0m[38;5;97m[1m[48;5;0mS[0m[38;5;62m[1m[48;5;0mC[0m[38;5;73m[1m[48;5;0mc[0m[38;5;104m[1m[48;5;0mx[0m[38;5;110m[1m[48;5;0mK[0m[38;5;139m[1m[48;5;0mW[0m[38;5;97m[1m[48;5;0m3[0m[38;5;62m[1m[48;5;0mR[0m[38;5;67m[1m[48;5;0mt[0m[38;5;104m[1m[48;5;0mH[0m[38;5;105m[1m[48;5;0mq[0m[38;5;110m[1m[48;5;0mR[0m[38;5;141m[1m[48;5;0mQ[0m[38;5;110m[1m[48;5;0my[0m[38;5;140m[1m[48;5;0m0[0m[38;5;103m[1m[48;5;0mI[0m[38;5;97m[1m[48;5;0me[0m",
        "y": "[38;5;87m[1m[48;5;0mo[0m[38;5;123m[1m[48;5;0mf[0m[38;5;159m[1m[48;5;0mE[0m[38;5;159m[1m[48;5;0mb[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0mn[0m[38;5;122m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mS[0m[38;5;122m[1m[48;5;0mY[0m[38;5;153m[1m[48;5;0ms[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0ms[0m[38;5;158m[1m[48;5;0mg[0m[38;5;117m[1m[48;5;0mA[0m[38;5;121m[1m[48;5;0mn[0m[38;5;116m[1m[48;5;0mn[0m[38;5;87m[1m[48;5;0my[0m[38;5;123m[1m[48;5;0mp[0m[38;5;159m[1m[48;5;0m4[0m[38;5;159m[1m[48;5;0mK[0m[38;5;159m[1m[48;5;0mY[0m[38;5;159m[1m[48;5;0mF[0m[38;5;122m[1m[48;5;0mo[0m[38;5;117m[1m[48;5;0mv[0m[38;5;122m[1m[48;5;0mZ[0m[38;5;153m[1m[48;5;0ma[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0mO[0m[38;5;158m[1m[48;5;0mV[0m[38;5;117m[1m[48;5;0m6[0m[38;5;121m[1m[48;5;0m0[0m[38;5;116m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m0[0m[38;5;123m[1m[48;5;0ms[0m[38;5;159m[1m[48;5;0mv[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;122m[1m[48;5;0m.[0m[38;5;117m[1m[48;5;0m-[0m",
        "z": "[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mJ[0m[38;5;63m[1m[48;5;0mg[0m[38;5;135m[1m[48;5;0mj[0m[38;5;135m[1m[48;5;0mu[0m[38;5;134m[1m[48;5;0mS[0m[38;5;135m[1m[48;5;0mW[0m[38;5;97m[1m[48;5;0m2[0m[38;5;134m[1m[48;5;0mN[0m[38;5;62m[1m[48;5;0mD[0m[38;5;99m[1m[48;5;0mg[0m[38;5;99m[1m[48;5;0mV[0m[38;5;141m[1m[48;5;0mJ[0m[38;5;133m[1m[48;5;0mh[0m[38;5;133m[1m[48;5;0mA[0m[38;5;62m[1m[48;5;0mX[0m[38;5;61m[1m[48;5;0mI[0m[38;5;99m[1m[48;5;0mu[0m[38;5;63m[1m[48;5;0mn[0m[38;5;135m[1m[48;5;0mM[0m[38;5;135m[1m[48;5;0mj[0m[38;5;134m[1m[48;5;0mC[0m[38;5;135m[1m[48;5;0mF[0m[38;5;97m[1m[48;5;0mq[0m[38;5;134m[1m[48;5;0m-[0m[38;5;62m[1m[48;5;0m-[0m[38;5;99m[1m[48;5;0m5[0m[38;5;99m[1m[48;5;0mT[0m[38;5;141m[1m[48;5;0mo[0m[38;5;133m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0md[0m[38;5;62m[1m[48;5;0mt[0m[38;5;61m[1m[48;5;0mR[0m[38;5;99m[1m[48;5;0m9[0m[38;5;63m[1m[48;5;0mz[0m[38;5;135m[1m[48;5;0mF[0m[38;5;135m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mw[0m[38;5;135m[1m[48;5;0mA[0m[38;5;97m[1m[48;5;0mU[0m",
        "_history": "[38;5;87m[1m[48;5;0mo[0m[38;5;123m[1m[48;5;0mf[0m[38;5;159m[1m[48;5;0mE[0m[38;5;159m[1m[48;5;0mb[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0mn[0m[38;5;122m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mS[0m[38;5;122m[1m[48;5;0mY[0m[38;5;153m[1m[48;5;0ms[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0ms[0m[38;5;158m[1m[48;5;0mg[0m[38;5;117m[1m[48;5;0mA[0m[38;5;121m[1m[48;5;0mn[0m[38;5;116m[1m[48;5;0mn[0m[38;5;87m[1m[48;5;0my[0m[38;5;123m[1m[48;5;0mp[0m[38;5;159m[1m[48;5;0m4[0m[38;5;159m[1m[48;5;0mK[0m[38;5;159m[1m[48;5;0mY[0m[38;5;159m[1m[48;5;0mF[0m[38;5;122m[1m[48;5;0mo[0m[38;5;117m[1m[48;5;0mv[0m[38;5;122m[1m[48;5;0mZ[0m[38;5;153m[1m[48;5;0ma[0m[38;5;87m[1m[48;5;0mU[0m[38;5;194m[1m[48;5;0mO[0m[38;5;158m[1m[48;5;0mV[0m[38;5;117m[1m[48;5;0m6[0m[38;5;121m[1m[48;5;0m0[0m[38;5;116m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m0[0m[38;5;123m[1m[48;5;0ms[0m[38;5;159m[1m[48;5;0mv[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;159m[1m[48;5;0m.[0m[38;5;122m[1m[48;5;0m.[0m[38;5;117m[1m[48;5;0m-[0m",
        "model": "[38;5;97m[1m[48;5;0mB[0m[38;5;134m[1m[48;5;0mb[0m[38;5;105m[1m[48;5;0ms[0m[38;5;138m[1m[48;5;0mA[0m[38;5;169m[1m[48;5;0m4[0m[38;5;97m[1m[48;5;0mN[0m[38;5;97m[1m[48;5;0mV[0m[38;5;97m[1m[48;5;0m-[0m[38;5;96m[1m[48;5;0mO[0m[38;5;98m[1m[48;5;0m2[0m[38;5;103m[1m[48;5;0mx[0m[38;5;132m[1m[48;5;0mh[0m[38;5;98m[1m[48;5;0ml[0m[38;5;98m[1m[48;5;0mY[0m[38;5;139m[1m[48;5;0mO[0m[38;5;134m[1m[48;5;0m0[0m[38;5;97m[1m[48;5;0mo[0m[38;5;134m[1m[48;5;0mc[0m[38;5;105m[1m[48;5;0mB[0m[38;5;138m[1m[48;5;0mO[0m[38;5;169m[1m[48;5;0mM[0m[38;5;97m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0mQ[0m[38;5;96m[1m[48;5;0mU[0m[38;5;98m[1m[48;5;0ms[0m[38;5;103m[1m[48;5;0m7[0m[38;5;132m[1m[48;5;0m8[0m[38;5;98m[1m[48;5;0mA[0m[38;5;98m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0m3[0m[38;5;134m[1m[48;5;0mL[0m[38;5;97m[1m[48;5;0m9[0m[38;5;134m[1m[48;5;0m9[0m[38;5;105m[1m[48;5;0mi[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0mg[0m[38;5;97m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m9[0m[38;5;97m[1m[48;5;0mh[0m",
        "Xtr": "[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mE[0m[38;5;81m[1m[48;5;0m3[0m[38;5;117m[1m[48;5;0mW[0m[38;5;110m[1m[48;5;0mM[0m[38;5;74m[1m[48;5;0m8[0m[38;5;75m[1m[48;5;0mc[0m[38;5;81m[1m[48;5;0mu[0m[38;5;111m[1m[48;5;0m5[0m[38;5;87m[1m[48;5;0me[0m[38;5;123m[1m[48;5;0m1[0m[38;5;117m[1m[48;5;0mX[0m[38;5;75m[1m[48;5;0mr[0m[38;5;80m[1m[48;5;0mv[0m[38;5;74m[1m[48;5;0mm[0m[38;5;74m[1m[48;5;0mP[0m[38;5;69m[1m[48;5;0md[0m[38;5;81m[1m[48;5;0mK[0m[38;5;81m[1m[48;5;0mR[0m[38;5;117m[1m[48;5;0mV[0m[38;5;110m[1m[48;5;0mr[0m[38;5;74m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mD[0m[38;5;81m[1m[48;5;0mJ[0m[38;5;111m[1m[48;5;0m0[0m[38;5;87m[1m[48;5;0m-[0m[38;5;123m[1m[48;5;0mL[0m[38;5;117m[1m[48;5;0m9[0m[38;5;75m[1m[48;5;0mm[0m[38;5;80m[1m[48;5;0ml[0m[38;5;74m[1m[48;5;0me[0m[38;5;74m[1m[48;5;0ma[0m[38;5;69m[1m[48;5;0mr[0m[38;5;81m[1m[48;5;0mj[0m[38;5;81m[1m[48;5;0m-[0m[38;5;117m[1m[48;5;0m1[0m[38;5;110m[1m[48;5;0m.[0m[38;5;74m[1m[48;5;0m0[0m[38;5;75m[1m[48;5;0m.[0m[38;5;81m[1m[48;5;0m2[0m",
        "ytr": "[38;5;240m[1m[48;5;0mb[0m[38;5;96m[1m[48;5;0mu[0m[38;5;61m[1m[48;5;0mT[0m[38;5;103m[1m[48;5;0mG[0m[38;5;133m[1m[48;5;0mO[0m[38;5;132m[1m[48;5;0mR[0m[38;5;132m[1m[48;5;0mt[0m[38;5;97m[1m[48;5;0m9[0m[38;5;133m[1m[48;5;0mM[0m[38;5;132m[1m[48;5;0mC[0m[38;5;132m[1m[48;5;0mq[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mq[0m[38;5;131m[1m[48;5;0mU[0m[38;5;95m[1m[48;5;0mm[0m[38;5;60m[1m[48;5;0mt[0m[38;5;240m[1m[48;5;0mR[0m[38;5;96m[1m[48;5;0mA[0m[38;5;61m[1m[48;5;0ma[0m[38;5;103m[1m[48;5;0mC[0m[38;5;133m[1m[48;5;0mU[0m[38;5;132m[1m[48;5;0mY[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0md[0m[38;5;133m[1m[48;5;0mo[0m[38;5;132m[1m[48;5;0mt[0m[38;5;132m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0mt[0m[38;5;96m[1m[48;5;0mk[0m[38;5;131m[1m[48;5;0ml[0m[38;5;95m[1m[48;5;0me[0m[38;5;60m[1m[48;5;0ma[0m[38;5;240m[1m[48;5;0mr[0m[38;5;96m[1m[48;5;0mn[0m[38;5;61m[1m[48;5;0m-[0m[38;5;103m[1m[48;5;0m1[0m[38;5;133m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0m.[0m[38;5;97m[1m[48;5;0m3[0m",
        "Xts": "[38;5;138m[1m[48;5;0m6[0m[38;5;169m[1m[48;5;0mZ[0m[38;5;140m[1m[48;5;0m4[0m[38;5;180m[1m[48;5;0m2[0m[38;5;169m[1m[48;5;0mm[0m[38;5;139m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0m0[0m[38;5;132m[1m[48;5;0mr[0m[38;5;133m[1m[48;5;0my[0m[38;5;103m[1m[48;5;0ml[0m[38;5;175m[1m[48;5;0mB[0m[38;5;139m[1m[48;5;0my[0m[38;5;140m[1m[48;5;0m0[0m[38;5;180m[1m[48;5;0mc[0m[38;5;205m[1m[48;5;0mJ[0m[38;5;97m[1m[48;5;0mV[0m[38;5;138m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0mt[0m[38;5;140m[1m[48;5;0mv[0m[38;5;180m[1m[48;5;0mi[0m[38;5;169m[1m[48;5;0m7[0m[38;5;139m[1m[48;5;0mn[0m[38;5;169m[1m[48;5;0ma[0m[38;5;132m[1m[48;5;0mI[0m[38;5;133m[1m[48;5;0mL[0m[38;5;103m[1m[48;5;0mY[0m[38;5;175m[1m[48;5;0mO[0m[38;5;139m[1m[48;5;0ms[0m[38;5;140m[1m[48;5;0mk[0m[38;5;180m[1m[48;5;0ml[0m[38;5;205m[1m[48;5;0me[0m[38;5;97m[1m[48;5;0ma[0m[38;5;138m[1m[48;5;0mr[0m[38;5;169m[1m[48;5;0mn[0m[38;5;140m[1m[48;5;0m-[0m[38;5;180m[1m[48;5;0m1[0m[38;5;169m[1m[48;5;0m.[0m[38;5;139m[1m[48;5;0m0[0m[38;5;169m[1m[48;5;0m.[0m[38;5;132m[1m[48;5;0m4[0m",
        "yts": "[38;5;77m[1m[48;5;0me[0m[38;5;119m[1m[48;5;0mw[0m[38;5;143m[1m[48;5;0mH[0m[38;5;83m[1m[48;5;0mi[0m[38;5;155m[1m[48;5;0m4[0m[38;5;143m[1m[48;5;0m1[0m[38;5;83m[1m[48;5;0m2[0m[38;5;149m[1m[48;5;0mm[0m[38;5;107m[1m[48;5;0m.[0m[38;5;71m[1m[48;5;0m1[0m[38;5;78m[1m[48;5;0mH[0m[38;5;120m[1m[48;5;0m0[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0mk[0m[38;5;107m[1m[48;5;0mM[0m[38;5;71m[1m[48;5;0mK[0m[38;5;77m[1m[48;5;0md[0m[38;5;119m[1m[48;5;0ml[0m[38;5;143m[1m[48;5;0mQ[0m[38;5;83m[1m[48;5;0m-[0m[38;5;155m[1m[48;5;0mt[0m[38;5;143m[1m[48;5;0mC[0m[38;5;83m[1m[48;5;0mi[0m[38;5;149m[1m[48;5;0mb[0m[38;5;107m[1m[48;5;0m7[0m[38;5;71m[1m[48;5;0ms[0m[38;5;78m[1m[48;5;0m4[0m[38;5;120m[1m[48;5;0mt[0m[38;5;155m[1m[48;5;0mk[0m[38;5;143m[1m[48;5;0ml[0m[38;5;107m[1m[48;5;0me[0m[38;5;71m[1m[48;5;0ma[0m[38;5;77m[1m[48;5;0mr[0m[38;5;119m[1m[48;5;0mn[0m[38;5;143m[1m[48;5;0m-[0m[38;5;83m[1m[48;5;0m1[0m[38;5;155m[1m[48;5;0m.[0m[38;5;143m[1m[48;5;0m0[0m[38;5;83m[1m[48;5;0m.[0m[38;5;149m[1m[48;5;0m5[0m",
        "X": "[38;5;97m[1m[48;5;0m-[0m[38;5;98m[1m[48;5;0m2[0m[38;5;104m[1m[48;5;0m_[0m[38;5;135m[1m[48;5;0m3[0m[38;5;248m[1m[48;5;0m6[0m[38;5;168m[1m[48;5;0m0[0m[38;5;98m[1m[48;5;0m6[0m[38;5;68m[1m[48;5;0m0[0m[38;5;139m[1m[48;5;0m6[0m[38;5;134m[1m[48;5;0m0[0m[38;5;105m[1m[48;5;0mc[0m[38;5;138m[1m[48;5;0m7[0m[38;5;169m[1m[48;5;0m7[0m[38;5;60m[1m[48;5;0md[0m[38;5;97m[1m[48;5;0mc[0m[38;5;61m[1m[48;5;0m8[0m[38;5;97m[1m[48;5;0mf[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m8[0m[38;5;135m[1m[48;5;0m0[0m[38;5;248m[1m[48;5;0m7[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0me[0m[38;5;68m[1m[48;5;0me[0m[38;5;139m[1m[48;5;0mc[0m[38;5;134m[1m[48;5;0m6[0m[38;5;105m[1m[48;5;0m6[0m[38;5;138m[1m[48;5;0mf[0m[38;5;169m[1m[48;5;0ma[0m[38;5;60m[1m[48;5;0mc[0m[38;5;97m[1m[48;5;0m5[0m[38;5;61m[1m[48;5;0m1[0m[38;5;97m[1m[48;5;0m2[0m[38;5;98m[1m[48;5;0m0[0m[38;5;104m[1m[48;5;0m5[0m[38;5;135m[1m[48;5;0m7[0m[38;5;248m[1m[48;5;0m8[0m[38;5;168m[1m[48;5;0md[0m[38;5;98m[1m[48;5;0m0[0m[38;5;68m[1m[48;5;0me[0m (content: 34_1738c83af436029507def2710bc5125f58d0e)"
    }
}
"""
```

```python3

# The same workflow will not be processed again if the same cache is used.
d = (
        idict.fromtoy()
        >> split
        >> let(fit, algorithm=RF, config={"n_estimators": 55}, Xin="Xtr", yin="ytr")
        >> let(predict, Xin="Xts")
        >> (lambda X: {"X2": X * X})
        >> (lambda _, X2, y: print("Some logging/printing that doesn't affect data...", X2.head()))
        >> [cache]
)
print(d.z)
"""
[1 0 1 0 1 1 1]
"""
```


</p>
</details>







## More info
Aside from the papers on [identification](https://arxiv.org/abs/2109.06028)
and on [similarity (not ready yet)](https://), the [PyPI package](https://pypi.org/project/oka) 
and [GitHub repository](https://github.com/davips/rabizao/oka), 
<!-- one can find more information, at a higher level application perspective,  -->
A lower level perspective is provided in the [API documentation](https://rabizao.github.io/oka).



## Grants
This work was supported by Fapesp under supervision of
Prof. André C. P. L. F. de Carvalho at CEPID-CeMEAI (Grants 2013/07375-0 – 2019/01735-0).

.>>>>>>>>> outros <<<<<<<<<<<.
