// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use inherent::inherent;
use std::sync::Arc;

use glean_core::metrics::MetricType;
use glean_core::ErrorType;

use crate::dispatcher;

// We need to wrap the glean-core type: otherwise if we try to implement
// the trait for the metric in `glean_core::metrics` we hit error[E0117]:
// only traits defined in the current crate can be implemented for arbitrary
// types.

/// This implements the developer-facing API for recording string list metrics.
///
/// Instances of this class type are automatically generated by the parsers
/// at build time, allowing developers to record values that were previously
/// registered in the metrics.yaml file.
#[derive(Clone)]
pub struct StringListMetric(pub(crate) Arc<glean_core::metrics::StringListMetric>);

impl StringListMetric {
    /// The public constructor used by automatically generated metrics.
    pub fn new(meta: glean_core::CommonMetricData) -> Self {
        Self(Arc::new(glean_core::metrics::StringListMetric::new(meta)))
    }
}

#[inherent(pub)]
impl glean_core::traits::StringList for StringListMetric {
    fn add<S: Into<String>>(&self, value: S) {
        let metric = Arc::clone(&self.0);
        let new_value = value.into();
        dispatcher::launch(move || crate::with_glean(|glean| metric.add(glean, new_value)));
    }

    fn set(&self, value: Vec<String>) {
        let metric = Arc::clone(&self.0);
        dispatcher::launch(move || crate::with_glean(|glean| metric.set(glean, value)));
    }

    fn test_get_value<'a, S: Into<Option<&'a str>>>(&self, ping_name: S) -> Option<Vec<String>> {
        crate::block_on_dispatcher();

        let queried_ping_name = ping_name
            .into()
            .unwrap_or_else(|| &self.0.meta().send_in_pings[0]);

        crate::with_glean(|glean| self.0.test_get_value(glean, queried_ping_name))
    }

    fn test_get_num_recorded_errors<'a, S: Into<Option<&'a str>>>(
        &self,
        error: ErrorType,
        ping_name: S,
    ) -> i32 {
        crate::block_on_dispatcher();

        crate::with_glean_mut(|glean| {
            glean_core::test_get_num_recorded_errors(&glean, self.0.meta(), error, ping_name.into())
                .unwrap_or(0)
        })
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use crate::common_test::{lock_test, new_glean};
    use crate::{CommonMetricData, ErrorType};

    #[test]
    fn string_list_metric_docs() {
        let _lock = lock_test();
        let _t = new_glean(None, true);

        let engine_metric: StringListMetric = StringListMetric::new(CommonMetricData {
            name: "event".into(),
            category: "test".into(),
            send_in_pings: vec!["test1".into()],
            ..Default::default()
        });

        let engines: Vec<String> = vec!["Google".to_string(), "DuckDuckGo".to_string()];

        // Add them one at a time
        engines.iter().for_each(|x| engine_metric.add(x));

        // Set them in one go
        engine_metric.set(engines);

        assert!(engine_metric.test_get_value(None).is_some());

        assert_eq!(
            vec!["Google".to_string(), "DuckDuckGo".to_string()],
            engine_metric.test_get_value(None).unwrap()
        );

        assert_eq!(
            0,
            engine_metric.test_get_num_recorded_errors(ErrorType::InvalidValue, None)
        );
    }
}
