"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @experimental
 */
class S3Origin {
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, props) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(scope, 'S3Origin');
            this.bucket.grantRead(this.originAccessIdentity);
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtb3JpZ2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiczMtb3JpZ2luLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNEQUFzRDtBQUd0RCwrQ0FBMkM7QUFpQjNDOzs7Ozs7OztHQVFHO0FBQ0gsTUFBYSxRQUFRO0lBR25CLFlBQVksTUFBa0IsRUFBRSxRQUF1QixFQUFFO1FBQ3ZELElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQzlCLElBQUksd0JBQVUsQ0FBQyxNQUFNLENBQUMsdUJBQXVCLEVBQUU7Z0JBQzdDLGNBQWMsRUFBRSxVQUFVLENBQUMsb0JBQW9CLENBQUMsU0FBUztnQkFDekQsR0FBRyxLQUFLO2FBQ1QsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLGNBQWMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVNLElBQUksQ0FBQyxLQUFvQixFQUFFLE9BQXFDO1FBQ3JFLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzFDLENBQUM7Q0FFRjtBQWhCRCw0QkFnQkM7QUFFRDs7OztHQUlHO0FBQ0gsTUFBTSxjQUFlLFNBQVEsVUFBVSxDQUFDLFVBQVU7SUFHaEQsWUFBNkIsTUFBa0IsRUFBRSxLQUFvQjtRQUNuRSxLQUFLLENBQUMsTUFBTSxDQUFDLHdCQUF3QixFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRG5CLFdBQU0sR0FBTixNQUFNLENBQVk7SUFFL0MsQ0FBQztJQUVNLElBQUksQ0FBQyxLQUFvQixFQUFFLE9BQXFDO1FBQ3JFLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLEVBQUU7WUFDOUIsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksVUFBVSxDQUFDLG9CQUFvQixDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztZQUNuRixJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQztTQUNsRDtRQUNELE9BQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVTLG9CQUFvQjtRQUM1QixPQUFPLEVBQUUsb0JBQW9CLEVBQUUscUNBQXFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyx3QkFBd0IsRUFBRSxFQUFFLENBQUM7SUFDN0gsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWRmcm9udCBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWRmcm9udCc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgSHR0cE9yaWdpbiB9IGZyb20gJy4vaHR0cC1vcmlnaW4nO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgdG8gdXNlIHRvIGN1c3RvbWl6ZSBhbiBTMyBPcmlnaW4uXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzT3JpZ2luUHJvcHMge1xuICAvKipcbiAgICogQW4gb3B0aW9uYWwgcGF0aCB0aGF0IENsb3VkRnJvbnQgYXBwZW5kcyB0byB0aGUgb3JpZ2luIGRvbWFpbiBuYW1lIHdoZW4gQ2xvdWRGcm9udCByZXF1ZXN0cyBjb250ZW50IGZyb20gdGhlIG9yaWdpbi5cbiAgICogTXVzdCBiZWdpbiwgYnV0IG5vdCBlbmQsIHdpdGggJy8nIChlLmcuLCAnL3Byb2R1Y3Rpb24vaW1hZ2VzJykuXG4gICAqXG4gICAqIEBkZWZhdWx0ICcvJ1xuICAgKi9cbiAgcmVhZG9ubHkgb3JpZ2luUGF0aD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBBbiBPcmlnaW4gdGhhdCBpcyBiYWNrZWQgYnkgYW4gUzMgYnVja2V0LlxuICpcbiAqIElmIHRoZSBidWNrZXQgaXMgY29uZmlndXJlZCBmb3Igd2Vic2l0ZSBob3N0aW5nLCB0aGlzIG9yaWdpbiB3aWxsIGJlIGNvbmZpZ3VyZWQgdG8gdXNlIHRoZSBidWNrZXQgYXMgYW5cbiAqIEhUVFAgc2VydmVyIG9yaWdpbiBhbmQgd2lsbCB1c2UgdGhlIGJ1Y2tldCdzIGNvbmZpZ3VyZWQgd2Vic2l0ZSByZWRpcmVjdHMgYW5kIGVycm9yIGhhbmRsaW5nLiBPdGhlcndpc2UsXG4gKiB0aGUgb3JpZ2luIGlzIGNyZWF0ZWQgYXMgYSBidWNrZXQgb3JpZ2luIGFuZCB3aWxsIHVzZSBDbG91ZEZyb250J3MgcmVkaXJlY3QgYW5kIGVycm9yIGhhbmRsaW5nLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIFMzT3JpZ2luIGltcGxlbWVudHMgY2xvdWRmcm9udC5JT3JpZ2luIHtcbiAgcHJpdmF0ZSByZWFkb25seSBvcmlnaW46IGNsb3VkZnJvbnQuSU9yaWdpbjtcblxuICBjb25zdHJ1Y3RvcihidWNrZXQ6IHMzLklCdWNrZXQsIHByb3BzOiBTM09yaWdpblByb3BzID0ge30pIHtcbiAgICB0aGlzLm9yaWdpbiA9IGJ1Y2tldC5pc1dlYnNpdGUgP1xuICAgICAgbmV3IEh0dHBPcmlnaW4oYnVja2V0LmJ1Y2tldFdlYnNpdGVEb21haW5OYW1lLCB7XG4gICAgICAgIHByb3RvY29sUG9saWN5OiBjbG91ZGZyb250Lk9yaWdpblByb3RvY29sUG9saWN5LkhUVFBfT05MWSwgLy8gUzMgb25seSBzdXBwb3J0cyBIVFRQIGZvciB3ZWJzaXRlIGJ1Y2tldHNcbiAgICAgICAgLi4ucHJvcHMsXG4gICAgICB9KSA6XG4gICAgICBuZXcgUzNCdWNrZXRPcmlnaW4oYnVja2V0LCBwcm9wcyk7XG4gIH1cblxuICBwdWJsaWMgYmluZChzY29wZTogY2RrLkNvbnN0cnVjdCwgb3B0aW9uczogY2xvdWRmcm9udC5PcmlnaW5CaW5kT3B0aW9ucyk6IGNsb3VkZnJvbnQuT3JpZ2luQmluZENvbmZpZyB7XG4gICAgcmV0dXJuIHRoaXMub3JpZ2luLmJpbmQoc2NvcGUsIG9wdGlvbnMpO1xuICB9XG5cbn1cblxuLyoqXG4gKiBBbiBPcmlnaW4gc3BlY2lmaWMgdG8gYSBTMyBidWNrZXQgKG5vdCBjb25maWd1cmVkIGZvciB3ZWJzaXRlIGhvc3RpbmcpLlxuICpcbiAqIENvbnRhaW5zIGFkZGl0aW9uYWwgbG9naWMgYXJvdW5kIGJ1Y2tldCBwZXJtaXNzaW9ucyBhbmQgb3JpZ2luIGFjY2VzcyBpZGVudGl0aWVzLlxuICovXG5jbGFzcyBTM0J1Y2tldE9yaWdpbiBleHRlbmRzIGNsb3VkZnJvbnQuT3JpZ2luQmFzZSB7XG4gIHByaXZhdGUgb3JpZ2luQWNjZXNzSWRlbnRpdHkhOiBjbG91ZGZyb250Lk9yaWdpbkFjY2Vzc0lkZW50aXR5O1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0LCBwcm9wczogUzNPcmlnaW5Qcm9wcykge1xuICAgIHN1cGVyKGJ1Y2tldC5idWNrZXRSZWdpb25hbERvbWFpbk5hbWUsIHByb3BzKTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBvcHRpb25zOiBjbG91ZGZyb250Lk9yaWdpbkJpbmRPcHRpb25zKTogY2xvdWRmcm9udC5PcmlnaW5CaW5kQ29uZmlnIHtcbiAgICBpZiAoIXRoaXMub3JpZ2luQWNjZXNzSWRlbnRpdHkpIHtcbiAgICAgIHRoaXMub3JpZ2luQWNjZXNzSWRlbnRpdHkgPSBuZXcgY2xvdWRmcm9udC5PcmlnaW5BY2Nlc3NJZGVudGl0eShzY29wZSwgJ1MzT3JpZ2luJyk7XG4gICAgICB0aGlzLmJ1Y2tldC5ncmFudFJlYWQodGhpcy5vcmlnaW5BY2Nlc3NJZGVudGl0eSk7XG4gICAgfVxuICAgIHJldHVybiBzdXBlci5iaW5kKHNjb3BlLCBvcHRpb25zKTtcbiAgfVxuXG4gIHByb3RlY3RlZCByZW5kZXJTM09yaWdpbkNvbmZpZygpOiBjbG91ZGZyb250LkNmbkRpc3RyaWJ1dGlvbi5TM09yaWdpbkNvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4geyBvcmlnaW5BY2Nlc3NJZGVudGl0eTogYG9yaWdpbi1hY2Nlc3MtaWRlbnRpdHkvY2xvdWRmcm9udC8ke3RoaXMub3JpZ2luQWNjZXNzSWRlbnRpdHkub3JpZ2luQWNjZXNzSWRlbnRpdHlOYW1lfWAgfTtcbiAgfVxufVxuIl19